//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        fonts.C
//
// Purpose:     implementation of VRMLScene concerning fonts
//
// Created:     19 Jun 96   Michael Pichler
//
// Changed:     15 Jul 96   Michael Pichler
//
// $Id: fonts.C,v 1.6 1997/02/25 17:03:58 mpichler Exp $
//
//</file>


#include "fonts.h"
#include "vrmlscene.h"
#include "scene3d.h"

#include <vrml/QvDB.h>
#include <vrml/QvInput.h>
#include <vrml/QvNode.h>
#include <vrml/QvGroup.h>

#include <hyperg/utils/verbose.h>
#include <hyperg/utils/str.h>



/***** FontChar *****/


void FontChar::setGlyph (QvNode* node, float width)
{
  glyph_ = node;
  if (node && !width)
    width_ = node->omax_.x;  // negative x coords allowed to overlap previous char
  else
    width_ = width;
}



/***** VRMLScene *****/


// font definitions static to avoid reload on each new scene
FontChar* VRMLScene::fontchars_ [fnt_numdefined] = { 0 };


// getFontChars
// get glyphs and other info for each character of a font

FontChar* VRMLScene::getFontChars (int family, int& bold, int& italic)
{
  int font = family;
  if (bold)
    font += fnt_bold;
  if (italic)
    font += fnt_italic;

  static const char* familyfilename [3] = { "serf", "sans", "type" };

  if (font < 0 || font >= fnt_numdefined || family < 0 || family > 2)
  { cerr << "VRMLScene::getTextNodes. internal error: font index "
         << font << " or family " << family << " out of range.";
    return 0;
  }

  // here some font definitions could be mapped onto others
  // e.g. wireframe font can be made bold by changing linewidth, font -= fnt_bold

  RString fname = scene_->fontDir ();
  fname += "/";
  fname += scene_->fontFileBase ();
  fname += familyfilename [family];
  if (bold)
    fname += "b";
  if (italic)
    fname += "i";

  FontChar* fontchar = loadFont (fname, fontchars_ [font]);

  // solid fonts: load boald font, italicize it
  // (would need reverse order when having italic versions of wireframe fonts)
  if (italic)
  {
    if (fontchar)
    { italic = 0;
      return fontchar;
    }
    else
    { italic = 0;  // search non-italic font instead
      fontchars_ [font] = fontchar = getFontChars (family, bold, italic);
      italic = 1;
    }
  }

  if (bold)
  {
    if (fontchar)
    { bold = 0;
      return fontchar;
    }
    else
    { bold = 0;  // search non-bold font instead
      fontchars_ [font] = fontchar = getFontChars (family, bold, italic);
      bold = 1;
    }
  }

  // sanserif may substitute typewriter (fixed)
  if (!fontchar && family == fnt_fixed)
    fontchars_ [font] = fontchar = getFontChars (fnt_sans, bold, italic);

  if (!fontchar)
    cerr << "VRweb. error: failed loading font file " << fname << ".wrl" << endl;

  // if using substitute font, set pointer accordingly to prevent subsequent tries to open that file
  // if finding no fonts at all, set them to new FontChar [256]

  return fontchar;

} // getFontChars


// loadFont
// helper for getFontChars to load a font file

FontChar* VRMLScene::loadFont (const char* fname, FontChar*& thisfont)
{
  if (thisfont)  // font already loaded
    return thisfont;

  RString filename (fname);
  filename += ".wrl";
  // TODO: care for decompression (via Scene3D/SceneWindow; accept .wrl, .wrl.gz, .wrz)

  DEBUGNL ("trying to read font file " << filename);
  FILE* fontfile = fopen (filename, "rb");
  if (!fontfile)
  { DEBUGNL ("failed opening font file " << filename);
    return 0;
  }

  QvInput in;
  in.setFilePointer (fontfile);

  DEBUGNL ("reading font file " << filename);
  QvNode* fontroot = 0;

  int ok = QvDB::read (&in, fontroot) && fontroot;
  fclose (fontfile);

  if (!ok)  // calling routine may return another font in this case
  { cerr << "error on reading font file " << filename << endl;
    return 0;
  }
  fontroot->ref ();  // will not be deleted (reused for further scenes)

  // build (preprocessing)
  buildFont (fontroot);

  // assuming one Separator at root level with named character children
  // (could search node in graph to allow for unrestricted structure)
  if (!fontroot->isGroupNode () || !((QvGroup*) fontroot)->getNumChildren ())
  { cerr << "font file " << filename << ": unexpected font file structure (no nodes?)" << endl;
    fontroot->unref ();
    return 0;
  }
  QvGroup* glyphroot = (QvGroup*) ((QvGroup*) fontroot)->getChild (0);
  if (!glyphroot->isGroupNode () || !glyphroot->getNumChildren ())
  { cerr << "font file " << filename << ": unexpected font file structure "
    "(missing or empty root level Separator?)" << endl;
    fontroot->unref ();
    return 0;
  }

  // allocate the font
  thisfont = new FontChar [256];

  static const char* hexdigit = "0123456789abcdef";

  // find character glyps
  int i = glyphroot->getNumChildren ();
  while (i--)
  {
    QvNode* charglyph = glyphroot->getChild (i);
    const char* name = charglyph->getName ().getString ();
    // cerr << "found node " << name << endl;

    if (!name || !*name)
      continue;

    if (strlen (name) != 7 || strncmp (name, "chr0x", 5))
    { DEBUGNL ("node " << name << " does not define a character (expected: chr0xNN)");
      continue;
    }
    const char* digit = strchr (hexdigit, 5 [name]);  // :-)
    int charnum = (digit ? digit - hexdigit : 0) << 4;
    digit = strchr (hexdigit, name [6]);
    charnum += (digit ? digit - hexdigit : 0);
    // charnum < 256

    if (thisfont [charnum].glyph ())
    { // traversed back to front, so last one "wins"
      cerr << "font file " << filename << ": multiple definition of character " << charnum
           << " (" << name << ")" << endl;
      continue;
    }

    thisfont [charnum].setGlyph (charglyph);  // also width for fixed width fonts!

  } // while

  // space: width of 'i'
  thisfont [' '].setWidth (thisfont ['i'].width ());

  return thisfont;
} // loadFont
