/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>

#include <qlabel.h>
#include <qscrollview.h>

#include "StilDialog.h"
#include "GlobalIcon.h"

StilDialog::StilDialog(QWidget* parent, const char* name)
: QDialog(parent,name)
{
    haveContent = false;
        
    setCaption( "STIL view" );
    setBackgroundMode(PaletteBackground);
    setIcon(*mainIcon);
    
    scrollView = new QScrollView( this, "STIL view scroll area" );
    scrollView->setGeometry( 0, 0, 200, 200 );
    scrollView->setVScrollBarMode(QScrollView::Auto);
    scrollView->setHScrollBarMode(QScrollView::AlwaysOff);
    scrollView->setResizePolicy(QScrollView::Manual);
    
    // QVBox not used because it doesn't allow to extend the width
    // of child QLabels in order to get a right border.
    
    globalLabel = new QLabel(scrollView->viewport());
    globalLabel->setAlignment(AlignLeft|AlignTop);
    globalLabel->setIndent(4);
    globalLabel->hide();
    globalLabel->clear();
    {
        QColorGroup globalColours(black,black,black,black,black,black,black,
                                  QColor(192,192,192),QColor(192,192,192));
        QPalette palette(globalColours,globalColours,globalColours);
        globalLabel->setPalette(palette);
    }
    
    stilLabel = new QLabel(scrollView->viewport());
    stilLabel->setAlignment(AlignLeft|AlignTop);
    stilLabel->setIndent(4);
    stilLabel->hide();
    stilLabel->clear();
    {
        QColorGroup stilColours(black,black,black,black,black,black,black,
                                QColor(255,242,194), QColor(255,242,194));
        QPalette palette(stilColours,stilColours,stilColours);
        stilLabel->setPalette(palette);
    }

    bugLabel = new QLabel(scrollView->viewport());
    bugLabel->setAlignment(AlignLeft|AlignTop);
    bugLabel->setIndent(4);
    bugLabel->hide();
    bugLabel->clear();
    {
        QColorGroup bugColours(white,black,black,black,black,white,black,
                               red,red);
        QPalette palette(bugColours,bugColours,bugColours);
        bugLabel->setPalette(palette);
    }

    resize(300,300);
}

void StilDialog::setConfig(const HVSC_Config& inConfig)
{
    // Copy without verification.
    config = inConfig;
}

void StilDialog::setLabels(const char* global,
                           const char* stil,
                           const char* bug)
{
    int maxHeight = (atoi)(config.maxHeight);

    // Where to position the labels.
    int x = 0, y = 0;
    
    // Scroll view contents geometry.
    // Total width depends on the most wide label.
    int contentsWidth = 0, contentsHeight = 0;

    scrollView->removeChild(globalLabel);
    if ( global!=0 && config.showGlobalComments )
    {
        globalLabel->setText(globalString=global);
        contentsWidth = globalLabel->sizeHint().width();
        contentsHeight += globalLabel->sizeHint().height();
        scrollView->addChild(globalLabel,x,y);
        globalLabel->show();
        y += globalLabel->sizeHint().height();
    }
    else
    {
        globalLabel->hide();
        globalLabel->clear();
    }
    
    scrollView->removeChild(stilLabel);
    if ((stil!=0) && config.showStilEntries)
    {
        stilLabel->setText(stilString=stil);
        if (stilLabel->sizeHint().width() > contentsWidth)
            contentsWidth = stilLabel->sizeHint().width();
        contentsHeight += stilLabel->sizeHint().height();
        scrollView->addChild(stilLabel,x,y);
        stilLabel->show();
        y += stilLabel->sizeHint().height();
    }
    else
    {
        stilLabel->hide();
        stilLabel->clear();
    }

    scrollView->removeChild(bugLabel);
    if ((bug!=0) && config.showBugListEntries)
    {
        bugLabel->setText(bugString=bug);
        if (bugLabel->sizeHint().width() > contentsWidth)
            contentsWidth = bugLabel->sizeHint().width();
        contentsHeight += bugLabel->sizeHint().height();
        scrollView->addChild(bugLabel,x,y);
        bugLabel->show();
        y += bugLabel->sizeHint().height();
    }
    else
    {
        bugLabel->hide();
        bugLabel->clear();
    }

    haveContent = ( contentsWidth!=0 && contentsHeight!=0 );
    if ( haveContent )
    {
        // Right margin (scrollbar safety).
        contentsWidth += 20;
    
        // Bottom margin (border safety).
        int dlgHeight = contentsHeight + 4;
    
        // Resize each label to contents width.
        globalLabel->resize(contentsWidth,globalLabel->sizeHint().height());
        stilLabel->resize(contentsWidth,stilLabel->sizeHint().height());
        bugLabel->resize(contentsWidth,bugLabel->sizeHint().height());
        
        scrollView->resizeContents(contentsWidth,contentsHeight);

        setMaximumSize(contentsWidth,dlgHeight);
        if (config.autoResize)
        {
            if (dlgHeight > maxHeight)
                dlgHeight = maxHeight;
            resize(contentsWidth,dlgHeight);
            scrollView->resize(width(),height());
            show();
        }
    }
    else  // no contents
    {
        hide();
    }
}

void StilDialog::resizeEvent(QResizeEvent*)
{
    scrollView->resize(width(),height());
}
