//! Main rendering and event processing for the application.

use std::sync::mpsc;
use std::time::{Duration, Instant};

use crate::config::{Config, Peaks};
use crate::event::MonitorEvent;

use anyhow::{anyhow, Result};

use ratatui::{
    prelude::{Buffer, Constraint, Direction, Layout, Position, Rect},
    text::{Line, Span},
    widgets::{StatefulWidget, Widget},
    DefaultTerminal, Frame,
};

use crossterm::event::{
    Event as CrosstermEvent, KeyEvent, KeyEventKind, MouseButton, MouseEvent,
    MouseEventKind,
};

use serde::Deserialize;
use smallvec::{smallvec, SmallVec};

use crate::capture_manager::CaptureManager;
use crate::command::Command;
use crate::device_kind::DeviceKind;
use crate::event::Event;
use crate::object::ObjectId;
use crate::object_list::{ObjectList, ObjectListWidget};
use crate::state::{State, StateDirty};
use crate::view::{self, ListKind, View};

#[cfg(feature = "trace")]
use crate::{trace, trace_dbg};

/// A UI action.
///
/// Used internally as the result of input events.
///
/// Also generated by interaction with [`MouseArea`]s.
#[derive(Debug, Clone, Copy, Deserialize)]
#[cfg_attr(test, derive(PartialEq))]
pub enum Action {
    SelectTab(usize),
    MoveUp,
    MoveDown,
    TabLeft,
    TabRight,
    CloseDropdown,
    ActivateDropdown,
    #[serde(skip_deserializing)]
    SelectObject(ObjectId),
    #[serde(skip_deserializing)]
    SetTarget(view::Target),
    ToggleMute,
    SetAbsoluteVolume(f32),
    SetRelativeVolume(f32),
    SetDefault,
    Exit,
    // This can be used to delete a default keybinding - make it do nothing.
    Nothing,
}

struct Tab {
    title: String,
    list: ObjectList,
}

impl Tab {
    fn new(title: String, list: ObjectList) -> Self {
        Self { title, list }
    }
}

#[derive(
    Deserialize, Default, Debug, Clone, Copy, PartialEq, clap::ValueEnum,
)]
#[serde(rename_all = "lowercase")]
#[cfg_attr(test, derive(strum::EnumIter))]
pub enum TabKind {
    #[default]
    Playback,
    Recording,
    Output,
    Input,
    Configuration,
}

impl TabKind {
    pub fn index(&self) -> usize {
        *self as usize
    }
}

// Mouse events matching one of the MouseEventKinds within the Rect will
// perform the Actions.
pub type MouseArea =
    (Rect, SmallVec<[MouseEventKind; 4]>, SmallVec<[Action; 4]>);

/// Handles the main UI for the application.
///
/// This runs the main loop to process PipeWire events and terminal input and
/// to render the main tabs of the application.
pub struct App {
    /// If set, tells the main loop it's time to exit
    exit: bool,
    /// [`Command`](`crate::command::Command`) channel
    tx: pipewire::channel::Sender<Command>,
    /// [`Event`](`crate::event::Event`) channel
    rx: mpsc::Receiver<Event>,
    /// An error message to return to [`main`](`crate::main`) on exit
    error_message: Option<String>,
    /// The main tabs
    tabs: Vec<Tab>,
    /// The index of the currently-visible tab
    current_tab_index: usize,
    /// Areas populated during rendering which define actions corresponding to
    /// mouse activity
    mouse_areas: Vec<MouseArea>,
    /// The monitor has received all initial information
    is_ready: bool,
    /// The current PipeWire state
    state: State,
    /// Tracks the nodes being captured
    capture_manager: CaptureManager,
    /// A rendering view based on the current PipeWire state
    view: View,
    /// The application configuration
    config: Config,
    /// The row on which the mouse is being dragged. While the left mouse
    /// button is held down, this is used in place of the real row to allow the
    /// mouse to move on the vertical axis during horizontal dragging.
    drag_row: Option<u16>,
}

macro_rules! current_list {
    ($self:expr) => {
        $self.tabs[$self.current_tab_index].list
    };
}

impl App {
    pub fn new(
        tx: pipewire::channel::Sender<Command>,
        rx: mpsc::Receiver<Event>,
        config: Config,
    ) -> Self {
        let tabs = vec![
            Tab::new(
                String::from("Playback"),
                ObjectList::new(ListKind::Node(view::NodeKind::Playback), None),
            ),
            Tab::new(
                String::from("Recording"),
                ObjectList::new(
                    ListKind::Node(view::NodeKind::Recording),
                    None,
                ),
            ),
            Tab::new(
                String::from("Output Devices"),
                ObjectList::new(
                    ListKind::Node(view::NodeKind::Output),
                    Some(DeviceKind::Sink),
                ),
            ),
            Tab::new(
                String::from("Input Devices"),
                ObjectList::new(
                    ListKind::Node(view::NodeKind::Input),
                    Some(DeviceKind::Source),
                ),
            ),
            Tab::new(
                String::from("Configuration"),
                ObjectList::new(ListKind::Device, None),
            ),
        ];
        App {
            exit: false,
            tx,
            rx,
            error_message: None,
            tabs,
            current_tab_index: config.tab.index(),
            mouse_areas: Vec::new(),
            is_ready: false,
            state: State::default(),
            capture_manager: CaptureManager::default(),
            view: View::default(),
            config,
            drag_row: None,
        }
    }

    pub fn run(mut self, terminal: &mut DefaultTerminal) -> Result<()> {
        #[cfg(feature = "trace")]
        trace::initialize_logging()?;

        // Wait until we've received all initial data from PipeWire
        let _ = terminal.draw(|frame| {
            frame.render_widget(Line::from("Initializing..."), frame.area());
        });
        while !self.exit && !self.is_ready {
            let _ = self.handle_events(None);
        }

        let mut pacer = RenderPacer::new(self.config.fps);

        // Did we handle any events and thus need to re-render?
        let mut needs_render = true;

        while !self.exit {
            // Update view if needed
            match self.state.dirty {
                StateDirty::Everything => {
                    self.view = View::from(&self.state, &self.config.names);
                }
                StateDirty::PeaksOnly => {
                    self.view.update_peaks(&self.state);
                }
                _ => {}
            }
            self.state.dirty = StateDirty::Clean;

            #[cfg(feature = "trace")]
            trace_dbg!(&self.view);

            if needs_render && pacer.is_time_to_render() {
                needs_render = false;

                self.mouse_areas.clear();

                terminal.draw(|frame| {
                    current_list!(self).update(frame.area(), &self.view);

                    self.draw(frame);
                })?;
            }

            needs_render |= self.handle_events(
                // If there's no fps limit, we definitely rendered in this
                // iteration, so needs_render is false, and there is no timeout.
                needs_render.then_some(pacer.duration_until_next_frame()),
            )?;
        }

        self.error_message.map_or(Ok(()), |s| Err(anyhow!(s)))
    }

    fn draw(&mut self, frame: &mut Frame) {
        let widget = AppWidget {
            current_tab_index: self.current_tab_index,
            view: &self.view,
            config: &self.config,
        };
        let mut widget_state = AppWidgetState {
            mouse_areas: &mut self.mouse_areas,
            tabs: &mut self.tabs,
        };

        frame.render_stateful_widget(widget, frame.area(), &mut widget_state);
    }

    fn exit(&mut self, error_message: Option<String>) {
        self.exit = true;
        self.error_message = error_message;
    }

    /// Handle events with optional timeout.
    /// Returns true if events were handled.
    fn handle_events(&mut self, timeout: Option<Duration>) -> Result<bool> {
        let mut were_events_handled = match timeout {
            Some(timeout) => match self.rx.recv_timeout(timeout) {
                Ok(event) => event.handle(self)?,
                Err(mpsc::RecvTimeoutError::Timeout) => return Ok(false),
                Err(e) => return Err(e.into()),
            },
            // Block on the next event.
            None => self.rx.recv()?.handle(self)?,
        };
        // Then handle the rest that are available.
        while let Ok(event) = self.rx.try_recv() {
            were_events_handled |= event.handle(self)?;
        }

        Ok(were_events_handled)
    }
}

struct RenderPacer {
    frame_duration: Duration,
    next_frame_time: Instant,
}

impl RenderPacer {
    fn new(fps: Option<f32>) -> Self {
        let frame_duration = fps.map_or(Default::default(), |fps| {
            Duration::from_secs_f32(1.0 / fps)
        });

        Self {
            frame_duration,
            next_frame_time: Instant::now(),
        }
    }

    fn is_time_to_render(&mut self) -> bool {
        let now = Instant::now();

        if now >= self.next_frame_time {
            if now > self.next_frame_time + self.frame_duration {
                // We're running behind, so reset the frame timing.
                self.next_frame_time = now + self.frame_duration;
            } else {
                self.next_frame_time += self.frame_duration;
            }

            return true;
        }

        false
    }

    fn duration_until_next_frame(&self) -> Duration {
        self.next_frame_time
            .saturating_duration_since(Instant::now())
    }
}

trait Handle {
    /// Handle some kind of event. Returns true if the event was handled which
    /// indicates that the UI needs to be redrawn.
    fn handle(self, app: &mut App) -> Result<bool>;
}

impl Handle for Event {
    fn handle(self, app: &mut App) -> Result<bool> {
        match self {
            Event::Input(event) => event.handle(app),
            Event::Monitor(event) => event.handle(app),
            Event::Error(event) => event.handle(app),
            Event::Ready => {
                app.is_ready = true;
                Ok(true)
            }
        }
    }
}

impl Handle for crossterm::event::Event {
    fn handle(self, app: &mut App) -> Result<bool> {
        match self {
            CrosstermEvent::Key(event) => event.handle(app),
            CrosstermEvent::Mouse(event) => event.handle(app),
            CrosstermEvent::Resize(..) => Ok(true),
            _ => Ok(false),
        }
    }
}

impl Handle for KeyEvent {
    fn handle(self, app: &mut App) -> Result<bool> {
        if self.kind != KeyEventKind::Press {
            return Ok(false);
        }

        if let Some(&action) = app.config.keybindings.get(&self) {
            return action.handle(app);
        }

        Ok(false)
    }
}

impl Handle for Action {
    fn handle(self, app: &mut App) -> Result<bool> {
        match self {
            Action::SelectTab(index) => {
                if index < app.tabs.len() {
                    app.current_tab_index = index;
                }
            }
            Action::MoveDown => {
                current_list!(app).down(&app.view);
            }
            Action::MoveUp => {
                current_list!(app).up(&app.view);
            }
            Action::TabLeft => {
                app.current_tab_index = app
                    .current_tab_index
                    .checked_sub(1)
                    .unwrap_or(app.tabs.len() - 1)
            }
            Action::TabRight => {
                app.current_tab_index =
                    (app.current_tab_index + 1) % app.tabs.len()
            }
            Action::CloseDropdown => {
                current_list!(app).dropdown_close();
            }
            Action::ActivateDropdown => {
                let commands = current_list!(app).dropdown_activate(&app.view);
                for command in commands {
                    let _ = app.tx.send(command);
                }
            }
            Action::SetTarget(target) => {
                let commands = current_list!(app).set_target(&app.view, target);
                for command in commands {
                    let _ = app.tx.send(command);
                }
            }
            Action::SelectObject(object_id) => {
                app.tabs[app.current_tab_index].list.selected = Some(object_id)
            }
            Action::ToggleMute => {
                let commands = current_list!(app).toggle_mute(&app.view);
                for command in commands {
                    let _ = app.tx.send(command);
                }
            }
            Action::SetAbsoluteVolume(volume) => {
                let commands =
                    current_list!(app).set_absolute_volume(&app.view, volume);
                for command in commands {
                    let _ = app.tx.send(command);
                }
            }
            Action::SetRelativeVolume(volume) => {
                let commands =
                    current_list!(app).set_relative_volume(&app.view, volume);
                for command in commands {
                    let _ = app.tx.send(command);
                }
            }
            Action::SetDefault => {
                let commands = current_list!(app).set_default(&app.view);
                for command in commands {
                    let _ = app.tx.send(command);
                }
            }
            Action::Exit => {
                app.exit(None);
            }
            Action::Nothing => {
                // Did nothing
                return Ok(false);
            }
        }

        Ok(true)
    }
}

impl Handle for MouseEvent {
    fn handle(self, app: &mut App) -> Result<bool> {
        match self.kind {
            MouseEventKind::Down(MouseButton::Left) => {
                app.drag_row = Some(self.row)
            }
            MouseEventKind::Up(MouseButton::Left) => app.drag_row = None,
            _ => {}
        }

        let actions = app
            .mouse_areas
            .iter()
            .rev()
            .find(|(rect, kinds, _)| {
                rect.contains(Position {
                    x: self.column,
                    y: app.drag_row.unwrap_or(self.row),
                }) && kinds.contains(&self.kind)
            })
            .map(|(_, _, action)| action.clone())
            .into_iter()
            .flatten();

        let mut handled_action = false;
        for action in actions {
            handled_action = true;
            let _ = action.handle(app);
        }

        Ok(handled_action)
    }
}

impl Handle for MonitorEvent {
    fn handle(self, app: &mut App) -> Result<bool> {
        app.state.update(&mut app.capture_manager, self);
        for command in app.capture_manager.flush() {
            // Filter out capture commands if capture is disabled
            match command {
                Command::NodeCaptureStart(..)
                | Command::NodeCaptureStop(..)
                    if app.config.peaks == Peaks::Off => {}
                command => {
                    let _ = app.tx.send(command);
                }
            }
        }
        Ok(true)
    }
}

impl Handle for String {
    fn handle(self, app: &mut App) -> Result<bool> {
        // Handle errors
        match self {
            // These happen when objects are removed while the monitor
            // is still in the process of setting up listeners
            error if error.starts_with("no global ") => {}
            error if error.starts_with("unknown resource ") => {}
            // I see this one when disconnecting a Bluetooth sink
            error if error == "Received error event" => {}
            // Not sure where this originates
            error if error == "Error: Buffer allocation failed" => {}
            _ => app.exit(Some(self)),
        }
        Ok(false) // This makes sense for now
    }
}

pub struct AppWidget<'a> {
    current_tab_index: usize,
    view: &'a View,
    config: &'a Config,
}

pub struct AppWidgetState<'a> {
    mouse_areas: &'a mut Vec<MouseArea>,
    tabs: &'a mut Vec<Tab>,
}

impl<'a> StatefulWidget for AppWidget<'a> {
    type State = AppWidgetState<'a>;

    fn render(self, area: Rect, buf: &mut Buffer, state: &mut Self::State) {
        let layout = Layout::default()
            .direction(Direction::Vertical)
            .constraints([
                Constraint::Min(0),    // list_area
                Constraint::Length(1), // menu_area
            ])
            .split(area);
        let list_area = layout[0];
        let menu_area = layout[1];

        let constraints: Vec<_> = state
            .tabs
            .iter()
            .map(|tab| Constraint::Length(tab.title.len() as u16 + 2))
            .collect();

        let menu_areas = Layout::default()
            .direction(Direction::Horizontal)
            .constraints(constraints)
            .split(menu_area);

        for (i, tab) in state.tabs.iter().enumerate() {
            let title_line = if i == self.current_tab_index {
                Line::from(vec![
                    Span::styled(
                        &self.config.char_set.tab_marker_left,
                        self.config.theme.tab_marker,
                    ),
                    Span::styled(&tab.title, self.config.theme.tab_selected),
                    Span::styled(
                        &self.config.char_set.tab_marker_right,
                        self.config.theme.tab_marker,
                    ),
                ])
            } else {
                Line::from(Span::styled(
                    format!(" {} ", tab.title),
                    self.config.theme.tab,
                ))
            };
            title_line.render(menu_areas[i], buf);

            state.mouse_areas.push((
                menu_areas[i],
                smallvec![MouseEventKind::Down(MouseButton::Left)],
                smallvec![Action::SelectTab(i)],
            ));
        }

        let mut widget = ObjectListWidget {
            object_list: &mut state.tabs[self.current_tab_index].list,
            view: self.view,
            config: self.config,
        };
        widget.render(list_area, buf, state.mouse_areas);
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use strum::IntoEnumIterator;

    #[test]
    fn select_tab_bounds() {
        let (command_tx, _) = pipewire::channel::channel::<Command>();
        let (_, event_rx) = mpsc::channel();

        let config = Config {
            remote: None,
            fps: None,
            mouse: false,
            peaks: Default::default(),
            char_set: Default::default(),
            theme: Default::default(),
            keybindings: Default::default(),
            names: Default::default(),
            tab: Default::default(),
        };
        let mut app = App::new(command_tx, event_rx, config);

        let _ = Action::SelectTab(app.tabs.len()).handle(&mut app);
        assert!(app.current_tab_index < app.tabs.len());
    }

    #[test]
    fn key_modifiers() {
        use crossterm::event::{KeyCode, KeyModifiers};
        use std::collections::HashMap;
        let (command_tx, _) = pipewire::channel::channel::<Command>();
        let (_, event_rx) = mpsc::channel();

        let x = KeyEvent::new(KeyCode::Char('x'), KeyModifiers::NONE);
        let ctrl_x = KeyEvent::new(KeyCode::Char('x'), KeyModifiers::CONTROL);

        let keybindings = HashMap::from([
            (x, Action::SelectTab(2)),
            (ctrl_x, Action::SelectTab(4)),
        ]);
        let config = Config {
            remote: None,
            fps: None,
            mouse: false,
            peaks: Default::default(),
            char_set: Default::default(),
            theme: Default::default(),
            keybindings,
            names: Default::default(),
            tab: Default::default(),
        };
        let mut app = App::new(command_tx, event_rx, config);

        let _ = x.handle(&mut app);
        assert_eq!(app.current_tab_index, 2);
        let _ = ctrl_x.handle(&mut app);
        assert_eq!(app.current_tab_index, 4);
        let _ = x.handle(&mut app);
        assert_eq!(app.current_tab_index, 2);
    }

    /// Ensure that the tabs enum variants are in the same order as the app's
    /// tab Vec. Making the initial tab configurable depends on this property
    /// because it uses the position of the enum variants to derivce an index
    /// into the tab Vec.
    #[test]
    fn tab_enum_order_matches_tab_vec() {
        let (command_tx, _) = pipewire::channel::channel::<Command>();
        let (_, event_rx) = mpsc::channel();

        let config = Config {
            remote: None,
            fps: None,
            mouse: false,
            peaks: Default::default(),
            char_set: Default::default(),
            theme: Default::default(),
            keybindings: Default::default(),
            names: Default::default(),
            tab: Default::default(),
        };
        let app = App::new(command_tx, event_rx, config);

        assert_eq!(TabKind::iter().count(), app.tabs.len());

        for (tab, Tab { title, .. }) in TabKind::iter().zip(app.tabs.iter()) {
            match tab {
                TabKind::Playback => assert_eq!(title, "Playback"),
                TabKind::Recording => assert_eq!(title, "Recording"),
                TabKind::Output => assert_eq!(title, "Output Devices"),
                TabKind::Input => assert_eq!(title, "Input Devices"),
                TabKind::Configuration => assert_eq!(title, "Configuration"),
            }
        }
    }
}
