/*
 * moves_order.c --- General package to store moves order.
 *
 * Copyright (c) 2001, 2002 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2002-12-31 18:46:55 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"moves_order.h"

#include	"main.h"	/* for InitialePieces */
#include	"pcpj.h"	/* for NbCoupsBlancs/NoirsRestants */

#include	<assert.h>

/*--------------------------------------------------------------------------*/

typedef struct moves_list_t
{
    struct one_move_t*		move;
    struct moves_list_t*	next;
} moves_list_t;

typedef struct one_move_t
{
    piece_index_t	piece_index;
    unsigned int	move_index;
    move_type_t		type;

    moves_list_t*	successors_list;

    bool_t		print_value;
} one_move_t;

typedef struct moves_order_t
{
    moves_list_t*	moves_list;
} moves_order_t;

/*--------------------------------------------------------------------------*/

static bool_t		Print_Value = FALSE;

static moves_order_t	Moves_Order;

/*--------------------------------------------------------------------------*/
/* Specification of the allocation package for moves_list_t */

#define	MAX_MOVES_LISTS		200
static moves_list_t	Moves_Lists[ MAX_MOVES_LISTS ];
static moves_list_t*	Free_Moves_Lists;

static void		initialise_moves_list();
static moves_list_t*	new_moves_list();
static void		delete_moves_list(
    moves_list_t* moves_list );

/*--------------------------------------------------------------------------*/
/* Specification of the allocation package for one_move_t */

#define	MAX_MOVES		100
static one_move_t	List_Of_Moves[ MAX_MOVES ];
static one_move_t*	Free_List_Of_Moves;

static void		initialise_list_of_moves();
static one_move_t*	new_one_move();
static void		delete_one_move( one_move_t* one_move );

/*--------------------------------------------------------------------------*/

static void		collect_all_moves(
    moves_list_t*		successors_list,
    moves_list_t*		all_moves );

static void		print_one_move(
    FILE*			fd,
    const one_move_t*		move,
    const exploration_t*	exploration );

static void		print_successors_list(
    FILE*			fd,
    unsigned int		level,
    moves_list_t*		successors_list,
    const exploration_t*	exploration );

static one_move_t*	find_in_successors_list(
    moves_list_t*		successors_list,
    piece_index_t		piece_index,
    unsigned int		move_index,
    move_type_t			type );

static bool_t		is_equal(
    const one_move_t*		move_1,
    const one_move_t*		move_2 );

/*--------------------------------------------------------------------------*/

struct moves_order_t*
create_moves_order( const exploration_t* exploration )
{
    unsigned int	i;
    moves_order_t*	moves_order = &Moves_Order;

    initialise_list_of_moves();
    initialise_moves_list();

    moves_order->moves_list = NULL;

    for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
    {
	unsigned int	j;
	moves_list_t*	moves_list;
	const piece_t*	piece 	      = &exploration->pieces[ i ];
	one_move_t*	previous_move = NULL;
	square_t	origin 	      = piece->caseInitiale;
	bool_t		hasPromoted   = FALSE;

	if ( piece->distance[ 0 ] == 0 )
	{
	    continue;		/* This piece never moves */
	}

	moves_list = new_moves_list();
	moves_list->move = NULL;
	moves_list->next = moves_order->moves_list;
	moves_order->moves_list = moves_list;

	for ( j = 0 ; j < piece->nbDestination ; j++ )
	{
	    int		deltaY 	  = row(origin) - row(piece->destination[ j ]);
	    one_move_t*	move      = new_one_move();
	    move->piece_index 	  = i;
	    move->move_index  	  = j;
	    move->type        	  = e_First_Move;
	    move->successors_list = NULL;
	    move->print_value     = Print_Value;
	    if ( previous_move == NULL )
	    {
		moves_list->move = move;
	    }
	    else
	    {
		previous_move->successors_list       = new_moves_list();
		previous_move->successors_list->move = move;
		previous_move->successors_list->next = NULL;
	    }
	    previous_move = move;

	    if ( piece->distance[ j ] == 1
		 && ( ( piece->camp == BLANC && NbCoupsBlancsRestants == 0 )
		      || ( piece->camp == NOIR && NbCoupsNoirsRestants == 0 )
		      || ( piece->typePiece == ROI
			   && piece->castling != NO_CASTLING
			   && j == 0 )
		      || ( piece->typePiece == PION
			   && !hasPromoted
			   && ( deltaY == 16 || deltaY == -16 ) ) ) )
	    {
		move->type = e_First_And_Last_Move;
	    }
	    else
	    {
		move                  = new_one_move();
		move->piece_index     = i;
		move->move_index      = j;
		move->type            = e_Last_Move;
		move->successors_list = NULL;
		move->print_value     = Print_Value;

		previous_move->successors_list       = new_moves_list();
		previous_move->successors_list->move = move;
		previous_move->successors_list->next = NULL;
		previous_move                        = move;
	    }
	    origin = piece->destination[ j ];
	    if ( origin == piece->casePromotion )
	    {
		hasPromoted = TRUE;
	    }
	}
    }
    return moves_order;
}

void
delete_moves_order( struct moves_order_t* moves_order )
{
#if 0
    moves_list_t* all_moves = new_moves_list();

    assert( moves_order != NULL );

    all_moves->move = NULL;
    all_moves->next = NULL;
    collect_all_moves( moves_order->moves_list, all_moves );

    if ( all_moves->next != NULL )
    {
	all_moves = all_moves->next;
	while ( all_moves != NULL )
	{
	    moves_list_t*	current_move = all_moves;
	    moves_list_t*	successors_list =
		all_moves->move->successors_list;

	    while ( successors_list != NULL )
	    {
		moves_list_t* current = successors_list;
		successors_list = successors_list->next;
		delete_moves_list( current );
	    }

	    delete_one_move( all_moves->move );

	    all_moves = all_moves->next;
	    delete_moves_list( current_move );
	}
    }
#endif
}

bool_t
add_move_order( struct moves_order_t*	moves_order,
		piece_index_t		before_piece_index,
		unsigned int		before_move_index,
		move_type_t		before_type,
		piece_index_t		after_piece_index,
		unsigned int		after_move_index,
		move_type_t		after_type )
{
    one_move_t*		before;
    one_move_t*		after;
    moves_list_t*	new_successor;
    moves_list_t*	moves_list;
    moves_list_t*	previous_in_list = NULL;

    assert( moves_order != NULL );
    assert( before_piece_index != after_piece_index
	    || before_move_index != after_move_index );

    before = find_in_successors_list(
	moves_order->moves_list,
	before_piece_index,
	before_move_index,
	before_type );

    after = find_in_successors_list(
	moves_order->moves_list,
	after_piece_index,
	after_move_index,
	after_type );

    assert( before != NULL );
    assert( after  != NULL );

    if ( find_in_successors_list( after->successors_list,
				  before_piece_index,
				  before_move_index,
				  before_type ) != NULL )
    {
	return FALSE;
    }

    new_successor = new_moves_list();
    new_successor->move = after;
    new_successor->next = before->successors_list;
    before->successors_list = new_successor;

    /* Remove after from the top level list if it is there */
    moves_list = moves_order->moves_list;
    while ( moves_list != NULL )
    {
	if ( is_equal( after, moves_list->move ) )
	{
	    if ( previous_in_list == NULL )
	    {
		moves_order->moves_list = moves_list->next;
	    }
	    else
	    {
		previous_in_list->next = moves_list->next;
	    }
	    delete_moves_list( moves_list );
	    break;
	}
	previous_in_list = moves_list;
	moves_list = moves_list->next;
    }

    /* For each successors of before, if it is a successor of after, remove it
     * from the list the successors of before.
     */
    moves_list = before->successors_list;
    previous_in_list = NULL;
    while ( moves_list != NULL )
    {
	if ( find_in_successors_list( after->successors_list,
				      moves_list->move->piece_index,
				      moves_list->move->move_index,
				      moves_list->move->type ) != NULL )
	{
	    if ( previous_in_list == NULL )
	    {
		before->successors_list = moves_list->next;
	    }
	    else
	    {
		previous_in_list->next = moves_list->next;
	    }
	    new_successor = moves_list->next;
	    delete_moves_list( moves_list );
	    moves_list = new_successor;
	}
	else
	{
	    previous_in_list = moves_list;
	    moves_list = moves_list->next;
	}
    }
    return TRUE;
}

void
print_moves_order( FILE*			fd,
		   const struct moves_order_t*	moves_order,
		   const exploration_t*		exploration )
{
    assert( fd != NULL );
    assert( moves_order != NULL );

    Print_Value = ! Print_Value;

    print_successors_list( fd,
			   0,
			   moves_order->moves_list,
			   exploration );
}

/*--------------------------------------------------------------------------*/

static void
collect_all_moves(
    moves_list_t*	successors_list,
    moves_list_t*	all_moves )
{
    while ( successors_list != NULL )
    {
	moves_list_t*	moves_list    = all_moves->next;
	moves_list_t*	previous      = all_moves;
	bool_t		must_be_added = TRUE;
	one_move_t*	move 	      = successors_list->move;

	while ( must_be_added && moves_list != NULL )
	{
	    if ( is_equal( move, moves_list->move ) )
	    {
		must_be_added = FALSE;
	    }
	    previous = moves_list;
	    moves_list = moves_list->next;
	}

	if ( must_be_added )
	{
	    previous->next = new_moves_list();
	    previous->next->move = move;
	    previous->next->next = NULL;
	}

	collect_all_moves( move->successors_list, all_moves );

	successors_list = successors_list->next;
    }
}

/*--------------------------------------------------------------------------*/

static void
print_one_move( FILE*			fd,
		const one_move_t*	move,
		const exploration_t*	exploration )
{
    square_t	from;
    piece_t	piece = exploration->pieces[ move->piece_index ];
    static char	move_type[] = { ' ', 'F', 'L', 'B' };

    if ( move->move_index == 0 )
    {
	from = piece.caseInitiale;
    }
    else
    {
	from = piece.destination[ move->move_index - 1 ];
    }

    fprintf( fd, "%c", InitialePieces[ piece.typePiece ] );
    if ( from != CASE_ARRIVEE_QUELCONQUE )
    {
	fprintf( fd, "%c%c", 'a' + column( from ), '1' + ( row( from ) >> 4 ) );
    }
    else
    {
	fprintf( fd, "??" );
    }
    if ( piece.destination[ move->move_index ] != CASE_ARRIVEE_QUELCONQUE )
    {
	if ( piece.typePiece == ROI
	     && move->move_index == 0
	     && piece.castling != NO_CASTLING )
	{
	    if ( piece.castling == KING_SIDE )
	    {
		fprintf( fd, "-OO" );
	    }
	    else
	    {
		fprintf( fd, "OOO" );
	    }
	}
	else
	{
	    fprintf( fd,
		     "-%c%c",
		     'a' + column( piece.destination[ move->move_index ] ),
		     '1'
		     + ( row( piece.destination[ move->move_index ] ) >> 4 ) );
	}
    }
    else
    {
	fprintf( fd, "-??" );
    }
    fprintf( fd, "(%d,%d", move->piece_index, move->move_index );
    if ( move->type != e_First_And_Last_Move )
    {
	fprintf( fd, ",%c)", move_type[ move->type ] );
    }
    else
    {
	fprintf( fd, ")  " );
    }
}

static void
print_successors_list(
    FILE*			fd,
    unsigned int		level,
    moves_list_t*		successors_list,
    const exploration_t*	exploration )
{
    bool_t first = TRUE;
    while ( successors_list != NULL )
    {
	one_move_t* move = successors_list->move;

	if ( !first )
	{
	    unsigned int	i;
	    for ( i = 0 ; i < level ; i++ )
	    {
		fprintf( fd, "                  " );
	    }
	}
	else
	{
	    if ( level > 0 )
	    {
		fprintf( fd, " -> " );
	    }
	    first = FALSE;
	}

	/* If we can print in one time the First and Last part of the same move,
	 * we do so.
	 */
	if ( move->type == e_First_Move
	     && move->successors_list != NULL
	     && move->successors_list->next == NULL
	     && move->piece_index == move->successors_list->move->piece_index )
	{
	    move->type = e_First_And_Last_Move;
	    print_one_move( fd, move, exploration );
	    move->type = e_First_Move;

	    move->print_value = Print_Value;
	    move = move->successors_list->move;
	}
	else
	{
	    print_one_move( fd, move, exploration );
	}

	if ( move->print_value != Print_Value )
	{
	    if ( move->successors_list == NULL )
	    {
		fputc( '\n', fd );
	    }
	    else
	    {
		print_successors_list( fd,
				       level + 1,
				       move->successors_list,
				       exploration );
	    }
	    move->print_value = Print_Value;
	}
	else
	{
	    if ( move->successors_list != NULL )
	    {
		fprintf( fd, " -> ..." );
	    }
	    fputc( '\n', fd );
	}

	successors_list = successors_list->next;
    }
}

/*--------------------------------------------------------------------------*/

static one_move_t*
find_in_successors_list(
    moves_list_t*	successors_list,
    piece_index_t	piece_index,
    unsigned int	move_index,
    move_type_t		type )
{
    while ( successors_list != NULL )
    {
	one_move_t* move = successors_list->move;

	if ( move->piece_index == piece_index
	     && move->move_index == move_index
	     && ( move->type & type ) != 0 )
	{
	    return move;
	}

	move = find_in_successors_list( move->successors_list,
					piece_index,
					move_index,
					type );
	if ( move != NULL )
	{
	    return move;
	}
	successors_list = successors_list->next;
    }
    return NULL;
}

/*--------------------------------------------------------------------------*/

static bool_t
is_equal( const one_move_t* move_1,
	  const one_move_t* move_2 )
{
    assert( move_1 != NULL );
    assert( move_2 != NULL );

    return
	move_1->piece_index == move_2->piece_index
	&& move_1->move_index == move_2->move_index
	&& ( move_1->type & move_2->type ) != 0;
}

/*--------------------------------------------------------------------------*/
/* Allocation package for moves_list_t */

static void
initialise_moves_list()
{
    unsigned int i;
    for ( i = 0 ; i < MAX_MOVES_LISTS ; i++ )
    {
	Moves_Lists[ i ].next = &Moves_Lists[ i + 1 ];
    }
    Moves_Lists[ i - 1 ].next = NULL;
    Free_Moves_Lists = Moves_Lists;
}

static moves_list_t*
new_moves_list()
{
    moves_list_t* moves_list = Free_Moves_Lists;
    assert( Free_Moves_Lists != NULL );
    Free_Moves_Lists = Free_Moves_Lists->next;
    return moves_list;
}

static void
delete_moves_list( moves_list_t* moves_list )
{
    moves_list->next = Free_Moves_Lists;
    Free_Moves_Lists = moves_list;
}

/*--------------------------------------------------------------------------*/
/* Allocation package for one_move_t */

static one_move_t	List_Of_Moves[ MAX_MOVES ];
static one_move_t*	Free_List_Of_Moves;

static void
initialise_list_of_moves()
{
    unsigned int i;
    for ( i = 0 ; i < MAX_MOVES ; i++ )
    {
	List_Of_Moves[ i ].successors_list =
	    (moves_list_t*)&List_Of_Moves[ i + 1 ];
    }
    List_Of_Moves[ i - 1 ].successors_list = NULL;
    Free_List_Of_Moves = List_Of_Moves;
}

static one_move_t*
new_one_move()
{
    one_move_t* one_move = Free_List_Of_Moves;
    assert( Free_List_Of_Moves != NULL );
    Free_List_Of_Moves = (one_move_t*)Free_List_Of_Moves->successors_list;
    return one_move;
}

static void
delete_one_move( one_move_t* one_move )
{
    one_move->successors_list = (moves_list_t*)Free_List_Of_Moves;
    Free_List_Of_Moves = one_move;
}
