/*
 * MyHashTable.c --- implementation of a hash table.
 *
 * Copyright (c) 2000, 2001, 2002 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2002-02-16 21:06:37 Pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#include	"myHashTable.h"
#include	"main.h"

#include	<assert.h>
#include	<stdlib.h>
#include	<string.h>
#include	<stdio.h>
#include	<time.h>

/*  #define HASH_STATS */

static	bool_t		Already_allocated = FALSE ;
static	unsigned long	Nb_Indexes ;
static	unsigned long	Index_Mask ;

/*
 * is_free_list_fragmented :
 *
 * This boolean has been introduced when we saw that initialising the free list
 * took up a huge amount of time when SPGs have a lot of positions and a lot of
 * memory is allocated to the hash-tables.  The free list becomes fragmented
 * only when the table is full and we compress it.  We never remove a position
 * from the hash-table except for making room for new positions.  Before that,
 * allocating an element is just taking the next portion of memory from the
 * table.  So we really use the free list only after all the table is full.
 * This means that it's not necessary to created a linked list at the creation
 * of the table, but just keep a pointer to the end of the memory currently
 * used.  This pointer is also free_list.
 */
typedef struct hash_table_t
{
    hash_element_t*	table ;
    char*		end_table ;
    hash_element_t**	indexes ;
    hash_element_t*	free_list ;
    bool_t		is_free_list_fragmented ;
    unsigned long	nb_elements ;
    size_t		element_size ;
    unsigned long	nb_max_elements ;
    unsigned long	nb_indexes ;

    /* For iterator */
    unsigned long	current_index ;
    hash_element_t*	current_element ;
    hash_element_t*	previous_element ;

#ifdef HASH_STATS
    unsigned long	nb_collisions ;
#endif
} hash_table_t ;
static hash_table_t*	Hash_table = NULL ;

struct hash_table_t *
dhtCreate(
    unsigned int	lg_key,
    unsigned long	nb_max_keys,
    unsigned long	max_memory )
{
    if ( ! Already_allocated )
    {
	unsigned long	max_table = max_memory - 4 * nb_max_keys ;
	Hash_table = (hash_table_t*)malloc( sizeof( hash_table_t ) );
	Hash_table->table = malloc( max_table );
	Hash_table->end_table = (char*)Hash_table->table + max_table ;
	Hash_table->indexes = malloc( nb_max_keys * sizeof( hash_element_t* ) );
	if ( Hash_table->table == NULL || Hash_table->indexes == NULL )
	{
	    fprintf( stderr,
		     "Error : Unable to allocate memory for hash table.\n" );
	    exit( EXIT_FAILURE );
	}

	Hash_table->nb_indexes = nb_max_keys ;
	Nb_Indexes = nb_max_keys ;
	Index_Mask = nb_max_keys - 1 ;

	Already_allocated = TRUE ;

#ifdef HASH_STATS
	fprintf( MainFD, "nb_indexes = %ld (0x%08lx)\n",
		 Hash_table->nb_indexes, Hash_table->nb_indexes );
	fprintf( MainFD, "max_memory = 0x%08lx\n", max_memory );
	fprintf( MainFD,
		 "malloc size = table : 0x%08lx + indexes : 0x%08lx Total = 0x%08lx\n",
		 max_table,
		 nb_max_keys * sizeof( hash_element_t* ),
		 max_table + nb_max_keys * sizeof( hash_element_t* ) );
#endif /* HASH_STATS */
    }

    Hash_table->element_size = sizeof( hash_element_t* ) +
	sizeof( unsigned int ) + lg_key ;
    Hash_table->element_size = 4 * ( ( Hash_table->element_size + 3 ) / 4 );

    Hash_table->free_list 		= Hash_table->table ;
    Hash_table->is_free_list_fragmented = FALSE ;

    Hash_table->nb_elements 	= 0 ;
    Hash_table->nb_max_elements = 
	( Hash_table->end_table - (char*)Hash_table->table )
	/ Hash_table->element_size ;

    memset( Hash_table->indexes,
	    0,
	    Hash_table->nb_indexes * sizeof( hash_element_t* ) );

#ifdef HASH_STATS
    fprintf( MainFD, "lg_key          = %d\n",  lg_key );
    fprintf( MainFD, "element_size    = %d\n",  Hash_table->element_size );
    fprintf( MainFD, "nb_max_elements = %ld\n", Hash_table->nb_max_elements );
    Hash_table->nb_collisions = 0 ;
#endif

    return Hash_table ;
}

hash_element_t *
dhtEnterElement(
    struct hash_table_t*	hash_table ,
    unsigned long		hash_value ,
    const unsigned char*	key        ,
    unsigned int		data       )
{
    hash_element_t*	current ;

    hash_table->nb_elements++;
#ifdef HASH_STATS
    if ( hash_table->indexes[ hash_value ] != NULL )
    {
	hash_table->nb_collisions++;
    }
#endif

    current = hash_table->free_list ;
    if ( hash_table->is_free_list_fragmented )
    {
	hash_table->free_list = hash_table->free_list->next ;
    }
    else
    {
	hash_table->free_list = (hash_element_t*)((char*)hash_table->free_list
	    + hash_table->element_size );
    }

    memcpy( current->key, key, Hash_Key_Length );
    current->data = data ;
    current->next = hash_table->indexes[ hash_value ];
    hash_table->indexes[ hash_value ] = current ;

    return current ;
}

void
dhtDestroy( struct hash_table_t* hash_table )
{
#ifdef HASH_STATS
    unsigned long	i ;
    unsigned long	max_chain  = 0 ;
    unsigned long	used_slots = 0 ;

    fprintf( MainFD,
	     "Number of positions  : %ld / %ld\n",
	     hash_table->nb_elements, myHash_nb_max_elements( hash_table ) );
    fprintf( MainFD,
	     "Number of collisions : %ld\n",
	     hash_table->nb_collisions );

    for( i = 0 ; i < hash_table->nb_indexes ; i++ )
    {
	hash_element_t*	current = hash_table->indexes[ i ];
	if ( current != NULL )
	{
	    unsigned long	lg_chain = 0 ;
	    used_slots++;
	    while( current != NULL )
	    {
		lg_chain++;
		current = current->next ;
	    }
	    if ( lg_chain > max_chain )
	    {
		max_chain = lg_chain ;
	    }
	}
    }

    fprintf( MainFD,
	     "Number of used slots : %ld (%2.2Lf%%)\n",
	     used_slots, (100.0L * used_slots) / hash_table->nb_indexes );
    fprintf( MainFD,
	     "Maximum chain length : %ld\n",
	     max_chain );
    fprintf( MainFD,
	     "Average chain length : %f\n",
	     used_slots > 0 ? (double)hash_table->nb_elements / used_slots : 0);
#endif /* HASH_STATS */
}

hash_element_t*
dhtLookupElement( struct hash_table_t*	hash_table ,
		  unsigned long		hash_value ,
		  const unsigned char*	key        )
{
    hash_element_t*	current = hash_table->indexes[ hash_value ];

    while( current != NULL )
    {
	if ( memcmp( key, current->key, Hash_Key_Length ) == 0 )
	{
	    return current ;
	}
	current = current->next ;
    }
    return dhtNilElement ;
}

hash_element_t*
dhtGetFirstElement( struct hash_table_t* hash_table )
{
    hash_table->free_list = NULL ;
    hash_table->is_free_list_fragmented = TRUE ;

    hash_table->current_index    = -1 ;
    hash_table->current_element  = NULL ;
    hash_table->previous_element = NULL ;
    return dhtGetNextElement( hash_table );
}

hash_element_t*
dhtGetNextElement( struct hash_table_t* hash_table )
{
    unsigned long	current_index = hash_table->current_index ;

    if ( current_index == hash_table->nb_indexes )
    {
	return dhtNilElement ;
    }

    if ( hash_table->current_element != NULL )
    {
	hash_table->previous_element = hash_table->current_element ;
	hash_table->current_element  = hash_table->current_element->next ;
    }
    else if ( hash_table->previous_element != NULL )
    {
	hash_table->current_element = hash_table->previous_element->next ;
    }
    if ( hash_table->current_element != NULL )
    {
	return hash_table->current_element ;
    }

    current_index++;
    while( hash_table->indexes[ current_index ] == NULL &&
	   current_index < hash_table->nb_indexes )
    {
	current_index++;
    }
    hash_table->current_index = current_index ;
    if ( current_index == hash_table->nb_indexes )
    {
	return dhtNilElement ;
    }

    hash_table->previous_element = NULL ;
    hash_table->current_element  = hash_table->indexes[ current_index ];

    return hash_table->current_element ;
}

void
myHash_remove_current_element( struct hash_table_t* hash_table )
{
    assert( hash_table->current_index != hash_table->nb_indexes );
    assert( hash_table->current_element != NULL );

    if ( hash_table->previous_element == NULL )
    {
	hash_table->indexes[ hash_table->current_index ] =
	    hash_table->current_element->next ;
	hash_table->current_index--;
    }
    else
    {
	hash_table->previous_element->next = hash_table->current_element->next ;
    }
    hash_table->current_element->next = hash_table->free_list ;
    hash_table->free_list = hash_table->current_element ;

    hash_table->current_element = NULL ;
    hash_table->nb_elements--;
}

unsigned long
dhtKeyCount( const struct hash_table_t* hash_table )
{
    return hash_table->nb_elements ;
}

unsigned long
myHash_nb_max_elements( struct hash_table_t* hash_table )
{
    return hash_table->nb_max_elements ;
}

#ifdef HASH_STATS
unsigned long
dhtCollisionCount( const struct hash_table_t* hash_table )
{
    return hash_table->nb_collisions ;
}
#endif

/* The following hash functions was found in Dr. Dobbs Journal, 1996 */

#define	mix(a, b, c)				\
{						\
    a -= b; a -= c; a ^= (c>>13);		\
    b -= c; b -= a; b ^= (a<<8);		\
    c -= a; c -= b; c ^= (b>>13);		\
    a -= b; a -= c; a ^= (c>>12);		\
    b -= c; b -= a; b ^= (a<<16);		\
    c -= a; c -= b; c ^= (b>>5);		\
    a -= b; a -= c; a ^= (c>>3);		\
    b -= c; b -= a; b ^= (a<<10);		\
    c -= a; c -= b; c ^= (b>>15);		\
}

typedef	unsigned long	ub4 ;

unsigned long
myHash_hash_value( const unsigned char* key )
{
    register unsigned long	a, b, c, len ;

    len = Hash_Key_Length ;
    a = b = 0x9e3779b9 ;
    c = 0 ;

    while ( len >= 12 )
    {
	a += (key[0] +((ub4)key[1]<<8) +((ub4)key[2]<<16) +((ub4)key[3]<<24));
	b += (key[4] +((ub4)key[5]<<8) +((ub4)key[6]<<16) +((ub4)key[7]<<24));
	c += (key[8] +((ub4)key[9]<<8) +((ub4)key[10]<<16) +((ub4)key[11]<<24));
	mix( a, b, c );
	key += 12 ;
	len -= 12 ;
    }

    /* c += Hash_Key_Length ; As all the keys have the same length, this is not
     * necessary for Natch.
     */
    switch ( len )
    {
    case 11:	c += ((ub4)key[10]<<24);
    case 10:	c += ((ub4)key[9]<<16);
    case 9:	c += ((ub4)key[8]<<8);
	/* First byte of is reserved for length, but is unused for Natch */
    case 8:	b += ((ub4)key[7]<<24);
    case 7:	b += ((ub4)key[6]<<16);
    case 6:	b += ((ub4)key[5]<<8);
    case 5:	b += key[4];
    case 4:	a += ((ub4)key[3]<<24);
    case 3:	a += ((ub4)key[2]<<16);
    case 2:	a += ((ub4)key[1]<<8);
    case 1:	a += key[0];
	/* case 0: nothing left to add */
    }
    mix( a, b, c );

    return c & Index_Mask ;
}
