/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * MinusSignOperator.java
 * ------------
 */
package org.pentaho.reporting.libraries.formula.operators;

import java.math.BigDecimal;

import org.pentaho.reporting.libraries.formula.EvaluationException;
import org.pentaho.reporting.libraries.formula.FormulaContext;
import org.pentaho.reporting.libraries.formula.LibFormulaErrorValue;
import org.pentaho.reporting.libraries.formula.util.NumberUtil;
import org.pentaho.reporting.libraries.formula.lvalues.TypeValuePair;
import org.pentaho.reporting.libraries.formula.typing.Type;
import org.pentaho.reporting.libraries.formula.typing.TypeRegistry;
import org.pentaho.reporting.libraries.formula.typing.coretypes.NumberType;

/**
 * Creation-Date: 02.11.2006, 10:27:03
 *
 * @author Thomas Morgner
 */
public class MinusSignOperator implements PrefixOperator
{
  private static final BigDecimal ZERO = new BigDecimal(0);
  private static final long serialVersionUID = 7453766552980074751L;


  public MinusSignOperator()
  {
  }

  public TypeValuePair evaluate(final FormulaContext context,
                                final TypeValuePair value1)
      throws EvaluationException
  {
    final Type type = value1.getType();
    final Object val = value1.getValue();
    if (val == null)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_NA_VALUE);
    }
    
    if (type.isFlagSet(Type.NUMERIC_TYPE))
    {
      final TypeRegistry typeRegistry = context.getTypeRegistry();
      // return the same as zero minus value.
      final Number number = typeRegistry.convertToNumber(type, val);
      if (number == null)
      {
        throw new EvaluationException
            (LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
      }
      final BigDecimal value = NumberUtil.getAsBigDecimal(number);
      return new TypeValuePair(NumberType.GENERIC_NUMBER, ZERO.subtract(value));
    }

    if(val instanceof Number)
    {
      final BigDecimal value = NumberUtil.getAsBigDecimal((Number)val);
      if (value == null)
      {
        throw new EvaluationException
            (LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
      }
      return new TypeValuePair(type, ZERO.subtract(value));
    }

    throw new EvaluationException
        (LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
  }

  public String toString()
  {
    return "-";
  }
}
