/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Gee;

namespace Unity {

private class Tools
{
  public static Variant hash_table_to_asv (HashTable<string, Variant> hash)
  {
    var b = new VariantBuilder (new VariantType ("a{sv}"));

    var iter = HashTableIter<string, Variant> (hash);
    
    string key;
    Variant val;
    while (iter.next (out key, out val))
    {
      b.add ("{sv}", key, val);
    }

    return b.end ();
  }
}

private class FiltersSynchronizer : GLib.Object
{
  public unowned Dee.Model provider { get; construct; }
  private Gee.ArrayList<unowned Dee.Model> _receivers;

  public FiltersSynchronizer (Dee.Model provider)
  {
    Object (provider:provider);
  }

  construct
  {
    _receivers = new Gee.ArrayList<unowned Dee.Model> ();

    provider.row_added.connect (on_row_added);
    provider.row_changed.connect (on_row_changed);
    provider.row_removed.connect (on_row_removed);
  }

  public void add_receiver (Dee.Model receiver)
  {
    receiver.clear();
    
    for (int i = 0; i < provider.get_n_rows(); ++i)
    {
      unowned Dee.ModelIter iter = provider.get_iter_at_row(i);
      add_row(iter, receiver);
    }
     _receivers.add(receiver);
  }

  private void on_row_added (Dee.Model model, Dee.ModelIter iter)
  {
    foreach (unowned Dee.Model receiver in _receivers)
    {
      add_row(iter, receiver);
    }
  }

  private void on_row_changed (Dee.Model model, Dee.ModelIter iter)
  {
    foreach (unowned Dee.Model receiver in _receivers)
    {
      unowned Dee.ModelIter riter = receiver.get_iter_at_row (model.get_position (iter));
      receiver.set(riter,
                   provider.get_string (iter, FilterColumn.ID),
                   provider.get_string (iter, FilterColumn.NAME),
                   provider.get_string (iter, FilterColumn.ICON_HINT),
                   provider.get_string (iter, FilterColumn.RENDERER_NAME),
                   provider.get_value (iter, FilterColumn.RENDERER_STATE),
                   provider.get_bool (iter, FilterColumn.VISIBLE),
                   provider.get_bool (iter, FilterColumn.COLLAPSED),
                   provider.get_bool (iter, FilterColumn.FILTERING));
    }
  }

  private void on_row_removed (Dee.Model model, Dee.ModelIter iter)
  {
    foreach (unowned Dee.Model receiver in _receivers)
    {
      receiver.remove (receiver.get_iter_at_row (model.get_position (iter)));
    }
  }

  private void add_row (Dee.ModelIter iter, Dee.Model receiver)
  {
    receiver.append(provider.get_string (iter, FilterColumn.ID),
                    provider.get_string (iter, FilterColumn.NAME),
                    provider.get_string (iter, FilterColumn.ICON_HINT),
                    provider.get_string (iter, FilterColumn.RENDERER_NAME),
                    provider.get_value (iter, FilterColumn.RENDERER_STATE),
                    provider.get_bool (iter, FilterColumn.VISIBLE),
                    provider.get_bool (iter, FilterColumn.COLLAPSED),
                    provider.get_bool (iter, FilterColumn.FILTERING));
  }
}


/*
 * Connects to the necessary signals to reflect changes in the providers on the
 * receiver Model
 */
private class ResultsSynchronizer : GLib.Object
{
  public unowned Dee.Model receiver { get; construct; }
  private Gee.ArrayList<unowned Dee.Model> _providers;
  
  private Gee.HashMap<unowned Dee.Model, unowned Dee.ModelTag> _provider_tags;

  public ResultsSynchronizer (Dee.Model receiver)
  {
    Object(receiver:receiver);
  }

  construct
  {
    _providers = new Gee.ArrayList<unowned Dee.Model> ();
    _provider_tags = new Gee.HashMap<unowned Dee.Model, unowned Dee.ModelTag> ();
  }

  public void add_provider (Dee.Model provider, string uid)
  { 
    provider.set_data<string>("uid", "%s:".printf(uid));

    _providers.add (provider);
    _provider_tags.set (provider, provider.register_tag ((d) => {}));

    provider.row_added.connect (on_row_added);
    provider.row_removed.connect (on_row_removed);
    provider.row_changed.connect (on_row_changed);
  }

  private void on_row_added (Dee.Model provider, Dee.ModelIter iter)
  {
    unowned Dee.ModelIter i;

    i = receiver.append (provider.get_data<string>("uid") + provider.get_string (iter, 0),
                         provider.get_string (iter, 1),
                         provider.get_uint32 (iter, 2),
                         provider.get_string (iter, 3),
                         provider.get_string (iter, 4),
                         provider.get_string (iter, 5),
                         provider.get_string (iter, 6));
    provider.set_tag (iter, _provider_tags.get (provider), i);
  }

  private void on_row_removed (Dee.Model provider, Dee.ModelIter iter)
  {
    unowned Dee.ModelTag tag = _provider_tags.get (provider);
    unowned Dee.ModelIter riter = (Dee.ModelIter)provider.get_tag (iter, tag);

    if (riter != null)
    {
      receiver.remove (riter);
    }
    else
    {
      warning (@"Could not find row to remove: $(provider.get_string (iter, 0))");
    }
  }

  private void on_row_changed (Dee.Model provider, Dee.ModelIter iter)
  {
    unowned Dee.ModelTag tag = _provider_tags.get (provider);
    unowned Dee.ModelIter riter = (Dee.ModelIter)provider.get_tag (iter, tag);

    if (riter != null)
    {
      receiver.set (riter,
                    provider.get_string (iter, 1),
                    provider.get_uint32 (iter, 2),
                    provider.get_string (iter, 3),
                    provider.get_string (iter, 4),
                    provider.get_string (iter, 5),
                    provider.get_string (iter, 6));
    }
    else
    {
      warning (@"Could not find row to change: $(provider.get_string (iter, 0))");
    }
  }
}

} /* namespace Unity */
