/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

private class ScopeProxyLocal : GLib.Object, ScopeProxy
{
  public Scope scope { get; construct; }
  public string[] sources {
    get { return scope.sources; }
    set {}
  }
  public bool search_in_global {
    get { return scope.search_in_global; }
    set {}
  }
  public Dee.SharedModel results_model {
    get { return scope.results_model; }
    set { }
  }
  public Dee.SharedModel global_results_model {
    get { return scope.global_results_model; }
    set { }
  }

  public Dee.SharedModel filters_model {
    get { return scope.get_filter_model (); }
    set { }
  }

  public ScopeProxyLocal (Scope scope)
  {
    Object (scope:scope);
  }

  construct
  {
    scope.bind_property ("search-in-global", this, "search-in-global", BindingFlags.DEFAULT);
    scope.bind_property ("sources", this, "sources", BindingFlags.DEFAULT);

    /* As we've set these in the constructor, let anyone that wants to watch
     * know that they've been set
     */
    Idle.add (()=>
    {
      notify_property ("search-in-global");
      notify_property ("sources");
      notify_property ("results-model");
      notify_property ("global-results-model");
      notify_property ("filters-model");
      notify_property ("sources");
      return false;
    });

    scope.search_finished.connect (on_search_finished);
    scope.global_search_finished.connect (on_global_search_finished);
  }

  /*
   * Implementation of the ScopeService interface
   */
  public async ActivationReplyRaw activate (string uri, uint action_type)
  {
    var reply = yield scope.activate (uri, action_type);
    return reply;
  }

  public async void search (string search_string,
                            HashTable<string, Variant> hints)
  {
    scope.search (search_string, hints);
  }

  public async void global_search (string search_string,
                                   HashTable<string, Variant> hints)
  {
    scope.global_search (search_string, hints);
  }

  public async PreviewReplyRaw preview (string uri)
  {
    var reply = yield scope.preview (uri);
    return reply;
  }

  public async void set_active (bool active)
  {
    scope.active = active;
  }

  public async void set_active_sources (string[] sources)
  {
    scope.set_active_sources_internal (sources);
  }
  
  public void on_search_finished (string search_string,
                                  HashTable<string, Variant> hints)
  {
    search_finished (search_string, hints);
  }
  
  public void on_global_search_finished (string search_string,
                                         HashTable<string, Variant> hints)
  {
    global_search_finished (search_string, hints);
  }
}

} /* Namespace */
