/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;

namespace Unity {

/*
 * Previews
 */

public abstract class Preview : InitiallyUnowned
{
  internal abstract string get_renderer_name();
  internal abstract HashTable<string, Variant> get_properties();
}

public class NoPreview : Preview
{
  public NoPreview ()
  {
  }

  internal override string get_renderer_name ()
  {
    return "preview-none";
  }

  internal override HashTable<string, Variant> get_properties ()
  {
    return new HashTable<string, Variant> (null, null);
  }
}

public class GenericPreview : Preview
{
  private HashTable<string, Variant> _hash;

  public GenericPreview (string name,
                         uint date_modified,
                         uint size,
                         string type,
                         string description,
                         string large_icon_hint,
                         string primary_action_name,
                         string primary_action_icon_hint,
                         string primary_action_uri,
                         string secondary_action_name="",
                         string secondary_action_icon_hint="",
                         string secondary_action_uri="",
                         string tertiary_action_name="",
                         string tertiary_action_icon_hint="",
                         string tertiary_action_uri="")
  {
    _hash = new HashTable<string, Variant> (null, null);

    _hash.insert ("name", name);
    _hash.insert ("date-modified", date_modified);
    _hash.insert ("size", size);
    _hash.insert ("type", type);
    _hash.insert ("description", description);
    _hash.insert ("icon-hint", large_icon_hint);
    _hash.insert ("primary-action-name", primary_action_name);
    _hash.insert ("primary-action-icon-hint", primary_action_icon_hint);
    _hash.insert ("primary-action-uri", primary_action_uri);
    _hash.insert ("secondary-action-name", secondary_action_name);
    _hash.insert ("secondary-action-icon-hint", secondary_action_icon_hint);
    _hash.insert ("secondary-action-uri", secondary_action_uri);
    _hash.insert ("tertiary-action-name", tertiary_action_name);
    _hash.insert ("tertiary-action-icon-hint", tertiary_action_icon_hint);
    _hash.insert ("tertiary-action-uri", tertiary_action_uri);
  }

  internal override string get_renderer_name ()
  {
    return "preview-generic";
  }

  internal override HashTable<string, Variant> get_properties ()
  {
    return _hash;
  }
}


public class ApplicationPreview : Preview
{
  private HashTable<string, Variant> _hash;

  public ApplicationPreview (string name,
                             string version,
                             string size,
                             string license,
                             string last_updated,
                             float  rating,
                             uint n_ratings,
                             string description,
                             string icon_hint,
                             string screenshot_icon_hint,
                             string primary_action_name,
                             string primary_action_icon_hint,
                             string primary_action_uri)
  {
    _hash = new HashTable<string, Variant> (null, null);

    _hash.insert ("name", name);
    _hash.insert ("version", version);
    _hash.insert ("size", size);
    _hash.insert ("license", license);
    _hash.insert ("last-updated", last_updated);
    _hash.insert ("rating", (double)rating);
    _hash.insert ("n-ratings", n_ratings);
    _hash.insert ("description", description);
    _hash.insert ("icon-hint", icon_hint);
    _hash.insert ("screenshot-icon-hint", screenshot_icon_hint);
    _hash.insert ("primary-action-name", primary_action_name);
    _hash.insert ("primary-action-icon-hint", primary_action_icon_hint);
    _hash.insert ("primary-action-uri", primary_action_uri);
  }

  internal override string get_renderer_name ()
  {
    return "preview-application";
  }

  internal override HashTable<string, Variant> get_properties ()
  {
    return _hash;
  }
}


/*
 * Creates a new TrackPreview instance
 *
 * primary_action_* corresponds to the main button in the preview
 * play/pause_action_uri is when the play/pause buttons are clicked
 */
public class TrackPreview : Preview
{
  private HashTable<string, Variant> _hash;

  public TrackPreview (uint number,
                       string title,
                       string artist,
                       string album,
                       uint length,
                       string[] genres,
                       string album_cover,
                       string primary_action_name,
                       string primary_action_icon_hint,
                       string primary_action_uri,
                       string play_action_uri,
                       string pause_action_uri)
  {
    _hash = new HashTable<string, Variant> (null, null);
    
    _hash.insert ("number", number);
    _hash.insert ("title", title);
    _hash.insert ("artist", artist);
    _hash.insert ("album", album);
    _hash.insert ("length", length);
    _hash.insert ("genres", genres);
    _hash.insert ("album-cover", album_cover);
    _hash.insert ("primary-action-name", primary_action_name);
    _hash.insert ("primary-action-icon-hint", primary_action_icon_hint);
    _hash.insert ("primary-action-uri", primary_action_uri);
    _hash.insert ("play-action-uri", play_action_uri);
    _hash.insert ("pause-action-uri", pause_action_uri);
  }

  internal override string get_renderer_name ()
  {
    return "preview-track";
  }

  internal override HashTable<string, Variant> get_properties ()
  {
    return _hash;
  }
}


/*
 * Creates a new AlbumPreview instance
 *
 * primary_action_* corresponds to the main button in the preview
 * play/pause_action_uri is when the play/pause buttons are clicked on each track
 */
public class AlbumPreview : Preview
{
  private HashTable<string, Variant> _hash;

  private Variant[] _tracks;

  public AlbumPreview (string name,
                       string artist,
                       string year,
                       string[] genres,
                       string album_cover,
                       string primary_action_name,
                       string primary_action_icon_hint,
                       string primary_action_uri)
  {
    _hash = new HashTable<string, Variant> (null, null);
    
    _hash.insert ("name", name);
    _hash.insert ("artist", artist);
    _hash.insert ("year", year);
    _hash.insert ("album_cover", album_cover);
    _hash.insert ("primary-action-name", primary_action_name);
    _hash.insert ("primary-action-icon-hint", primary_action_icon_hint);
    _hash.insert ("primary-action-uri", primary_action_uri);

  }

  void add_track (uint number,
                  string title,
                  uint length,
                  string play_action_uri,
                  string pause_action_uri)
  {
    var track = new Variant ("(ususs)", number, title, length, play_action_uri, pause_action_uri);
    _tracks+= track;
  }

  internal override string get_renderer_name ()
  {
    return "preview-album";
  }

  internal override HashTable<string, Variant> get_properties ()
  {
    _hash.insert ("tracks", _tracks);
    return _hash;
  }
}

} /* namespace */
