/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

/*
 * The private implementation of the Scope. This makes sure that none of the 
 * implementation details leak out into the public interface.
 */
private class ScopeImpl : GLib.Object, ScopeService
{
  private Scope _owner;
  private uint _dbus_id;
  private uint _info_changed_id;
  private uint _filters_changed_id;

  public Dee.SerializableModel _results_model;
  public Dee.SerializableModel _global_results_model;
  public Dee.SerializableModel _filters_model;

  /* we need notifications on this property */
  public ViewType view_type { get; set; }

  public ScopeImpl (Scope owner)
  {
    /* NOTE: Vala isn't allowing me to make Owner a construct variable so our
     * construction happens here instead of in construct {}
     */
    _owner = owner;
    _owner.notify["search-in-global"].connect (queue_info_changed);
    _owner.notify["visible"].connect (queue_info_changed);
    _owner.sources.changed.connect (queue_info_changed);

    create_models ();
  }
  
  /* Create usable name prefix for the models */
  private string create_dbus_name ()
  {
    /* We randomize the names to avoid conflicts to ensure that we always
     * have a clean start (no processes hanging around that would cause our models
     * to not be the leaders)
     */
    uint t = (uint)time_t ();
    var dbus_path = _owner.dbus_path;
    var dbus_name = "com.canonical.Unity.Scope";
    dbus_name += "." + Path.get_basename (dbus_path);
    dbus_name += ".T%u".printf (t);
    return dbus_name;
  }

  private void create_models ()
  {
    /* Schema definitions come from the Lens specification */
    _results_model = new Dee.SequenceModel ();
    _results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _global_results_model = new Dee.SequenceModel ();
    _global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _filters_model = new Dee.SequenceModel ();
    _filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");
    _filters_model.row_added.connect (on_filter_added);
    _filters_model.row_changed.connect (on_filter_changed);
    _filters_model.row_removed.connect (on_filter_removed);
  }

  private void create_shared_models (string dbus_name)
  {
    var backend = _results_model;
    _results_model = new Dee.SharedModel.with_back_end (
        dbus_name + ".Results", backend);

    backend = _global_results_model;
    _global_results_model = new Dee.SharedModel.with_back_end (
        dbus_name + ".GlobalResults", backend);

    backend = _filters_model;
    backend.row_added.disconnect (on_filter_added);
    backend.row_changed.disconnect (on_filter_changed);
    backend.row_removed.disconnect (on_filter_removed);
    _filters_model = new Dee.SharedModel.with_back_end (
        dbus_name + ".Filters", backend);
    _filters_model.row_added.connect (on_filter_added);
    _filters_model.row_changed.connect (on_filter_changed);
    _filters_model.row_removed.connect (on_filter_removed);
  }

  public void export () throws IOError
  {
    create_shared_models (create_dbus_name ());
    var conn = Bus.get_sync (BusType.SESSION);
    _dbus_id = conn.register_object (_owner.dbus_path, this as ScopeService);

    queue_info_changed ();
  }

  /* Queue up info-changed requests as we don't want to be spamming Unity with
   * them.
   */
  private void queue_info_changed ()
  {
    if (_info_changed_id == 0)
    {
      _info_changed_id = Idle.add (emit_info_changed);
    }
  }

  private void queue_filters_changed ()
  {
    if (_filters_changed_id == 0)
    {
      _filters_changed_id = Timeout.add(0, emit_filters_changed);
    }
  }

  private bool emit_filters_changed ()
  {
    _owner.filters_changed ();
    _filters_changed_id = 0;

    return false;
  }

  private string get_model_name (Dee.SerializableModel model)
  {
    if (model is Dee.SharedModel)
    {
      var shared_model = model as Dee.SharedModel;
      return shared_model.get_swarm_name ();
    }
    else
      return "<local>";
  }

  private bool emit_info_changed ()
  {
    var info = ScopeInfo ();
    info.dbus_path = _owner.dbus_path;
    info.sources = _owner.sources.get_hints ();
    info.search_in_global = _owner.search_in_global;
    info.private_connection_name = "<not implemented>";
    info.results_model_name = get_model_name (_results_model);
    info.global_results_model_name = get_model_name (_global_results_model);
    info.filters_model_name = get_model_name (_filters_model);
    info.hints = new HashTable<string, Variant> (str_hash, str_equal);

    changed (info);

    _info_changed_id = 0;
    return false;
  }

  private void on_filter_added (Dee.Model model, Dee.ModelIter iter)
  {
    if (model.get_string (iter, FilterColumn.ID) == Lens.SOURCES_FILTER_ID)
    {
      /* just make sure we directly update the properties on scope.sources
       * filter, the options are handled separately */
      _owner.sources.filtering = model.get_bool (iter, FilterColumn.FILTERING);
      return;
    }

    string icon_hint_s = model.get_string (iter, FilterColumn.ICON_HINT);

    Icon? icon_hint = null;
    try
      {
        if (icon_hint_s != "")
          icon_hint = Icon.new_for_string (icon_hint_s);
      }
    catch (Error e)
      {
        warning ("Error parsing GIcon data '%s': %s", icon_hint_s, e.message);
      }
    
    Filter? filter = null;
    FilterRenderer renderer = Filter.renderer_for_name (
        model.get_string (iter, FilterColumn.RENDERER_NAME));

    switch (renderer)
    {
      case FilterRenderer.RATINGS:
        filter = new RatingsFilter (model.get_string (iter, FilterColumn.ID),
                                    model.get_string (iter, FilterColumn.NAME),
                                    icon_hint);
        break;
      case FilterRenderer.RADIO_OPTIONS:
        filter = new RadioOptionFilter (model.get_string (iter, FilterColumn.ID),
                                        model.get_string (iter, FilterColumn.NAME),
                                        icon_hint);
        break;
      case FilterRenderer.CHECK_OPTIONS:
        filter = new CheckOptionFilter (model.get_string (iter, FilterColumn.ID),
                                        model.get_string (iter, FilterColumn.NAME),
                                        icon_hint);
        break;
      case FilterRenderer.CHECK_OPTIONS_COMPACT:
        filter = new CheckOptionFilterCompact (model.get_string (iter, FilterColumn.ID),
                                               model.get_string (iter, FilterColumn.NAME),
                                               icon_hint);
        break;
      case FilterRenderer.MULTIRANGE:
        filter = new MultiRangeFilter (model.get_string (iter, FilterColumn.ID),
                                       model.get_string (iter, FilterColumn.NAME),
                                       icon_hint);
        break;
    }

    if (filter is Filter)
    {
      filter.set_model_and_iter (model, iter);
      _owner._filters.append (filter);

      queue_filters_changed ();
    }
  }

  private void on_filter_changed (Dee.Model model, Dee.ModelIter iter)
  {
    if (model.get_string (iter, FilterColumn.ID) == Lens.SOURCES_FILTER_ID)
    {
      /* just make sure we directly update the properties on scope.sources
       * filter, the options are handled separately */
      _owner.sources.filtering = model.get_bool (iter, FilterColumn.FILTERING);
      return;
    }

    queue_filters_changed ();
  }

  private void on_filter_removed (Dee.Model model, Dee.ModelIter iter)
  {
    bool dirty = false;

    foreach (Filter filter in _owner._filters)
    {
      if (filter.id == model.get_string (iter, FilterColumn.ID))
      {
        dirty = true;
        _owner._filters.remove (filter);
        break;
      }
    }

    if (dirty) queue_filters_changed ();
  }


  /*
   * DBus Interface Implementation
   */
  public async void info_request ()
  {
    queue_info_changed ();
  }

  public async ActivationReplyRaw activate (string uri,
                                            uint action_type) throws IOError
  {
    var reply = ActivationReplyRaw ();
    
    ActivationResponse? response = _owner.activate_uri (uri);
    if (response == null)
      response = new ActivationResponse(HandledType.NOT_HANDLED);

    reply.uri = uri;
    reply.handled = response.handled;
    reply.hints = response.get_hints ();

    return reply;
  }

  private Cancellable[] cancellable_arr = new Cancellable[SearchType.N_TYPES];

  public async void schedule_search_changed (LensSearch search,
                                             SearchType search_type,
                                             bool wait_for_view)
  {
    /* don't use requires() here - valac 0.14 doesn't do the proper thing
     * with async methods */
    if (search_type >= SearchType.N_TYPES) return;

    bool waiting_needed = wait_for_view &&
      ((search_type == SearchType.GLOBAL && view_type != ViewType.HOME_VIEW) ||
       (search_type == SearchType.DEFAULT && view_type != ViewType.LENS_VIEW));

    /* Cancel any previous searches */
    if (cancellable_arr[search_type] != null)
    {
      cancellable_arr[search_type].cancel ();
    }

    var cancellable = new Cancellable ();
    cancellable_arr[search_type] = cancellable;

    if (waiting_needed)
    {
      var view_sig_id = this.notify["view-type"].connect (() =>
      {
        bool resume = (search_type == SearchType.DEFAULT && 
                       view_type == ViewType.LENS_VIEW)
                   || (search_type == SearchType.GLOBAL &&
                       view_type == ViewType.HOME_VIEW);
        if (resume) schedule_search_changed.callback ();
      });

      yield;
      SignalHandler.disconnect (this, view_sig_id);
    }

    /* We'll wait a bit to check if we don't get different search */
    Idle.add (schedule_search_changed.callback);
    yield;

    if (cancellable.is_cancelled ()) return;

    /* careful with the cancellable, the handler should ref it to use it */
    _owner.search_changed (search, search_type, cancellable);
  }

  private string[] search_keys = new string[SearchType.N_TYPES];

  public void update_search_key (LensSearch ls, SearchType search_type)
    requires (search_type < SearchType.N_TYPES)
  {
    string? search_key = get_search_key (ls, search_type);

    search_keys[search_type] = search_key;
    _owner.set_last_search (ls, search_type);
  }

  private string? get_search_key (LensSearch ls, SearchType search_type)
  {
    string? search_key = search_type == SearchType.DEFAULT ?
        _owner.generate_search_key["default"] (ls) :
        _owner.generate_search_key["global"] (ls);

    return search_key;
  }

  public void invalidate_search (SearchType search_type)
    requires (search_type < SearchType.N_TYPES)
  {
    search_keys[search_type] = null;
  }

  private async HashTable<string, Variant> search_internal (
      string search_string, HashTable<string, Variant> hints,
      SearchType search_type) throws ScopeError
  {
    HashTable<string, Variant> result =
        new HashTable<string, Variant> (str_hash, str_equal);
    bool is_global_search = search_type == SearchType.GLOBAL;

    // prepare new LensSearch instance
    var s = new LensSearch (search_string, hints, is_global_search ?
        _global_results_model : _results_model);

    // wait for the finished signal
    ulong sig_id = s.finished.connect ((lens_search) =>
    {
      HashTable<string, Variant>? reply_hints = lens_search.get_reply_hints ();
      if (reply_hints != null)
      {
        HashTableIter<string, Variant> iter = HashTableIter<string, Variant> (reply_hints);
        unowned string key;
        unowned Variant variant;
        while (iter.next (out key, out variant))
        {
          result.insert (key, variant);
        }
      }
      result.insert ("model-seqnum", new Variant.uint64 (lens_search.results_model.get_seqnum ()));
      search_internal.callback ();
    });

    LensSearch? last_search = null;
    ulong ls_sig_id = 0;

    string search_key = get_search_key (s, search_type);
    if ((search_key != null && search_key != search_keys[search_type]) ||
        (search_key == null && !s.equals (_owner.get_last_search (search_type))))
    {
      schedule_search_changed.begin (s, search_type, false);
      search_keys[search_type] = search_key;

      _owner.set_last_search (s, search_type);
    }
    else
    {
      // search key didn't change, but the last search might not have finished
      // yet, let's wait for it if that's the case
      last_search = _owner.get_last_search (search_type);
      if (last_search != null && !last_search.was_finished ())
      {
        // wait for the last search to finish
        ls_sig_id = last_search.finished.connect (() =>
        {
          s.finished ();
        });

        // update view_type if necessary, because schedule_search_changed
        // might be waiting for it
        ViewType current_view = is_global_search ?
          ViewType.HOME_VIEW : ViewType.LENS_VIEW;
        if (view_type != current_view)
        {
          var old_view = view_type;
          // unblocks the suspended search
          set_view_type (current_view);
          // makes sure we don't remain in wrong state
          set_view_type (old_view);
        }
      }
      else
      {
        // returning empty HashTable in this case (search_key didn't change)
        return result;
      }
    }

    // now we have a Cancellable associated with this search
    // (from schedule_search_changed)
    var cancellable = cancellable_arr[search_type];
    ulong canc_id = cancellable.connect (() =>
    {
      SignalHandler.block (s, sig_id);
      // we can't directly call callback (), because we're inside cancel ()
      // which means the Cancellable's lock is held, so we wouldn't be able
      // to disconnect the handler
      Idle.add (search_internal.callback);
    });

    yield;
    // we just got finished signal, or the cancellable was cancelled

    if (ls_sig_id != 0) SignalHandler.disconnect (last_search, ls_sig_id);
    SignalHandler.disconnect (s, sig_id);
    cancellable.disconnect (canc_id);

    if (cancellable.is_cancelled ())
    {
      throw new ScopeError.SEARCH_CANCELLED ("Search '%s' was cancelled",
                                             search_string);
    }

    return result;
  }

  public async HashTable<string, Variant> search (string search_string,
      HashTable<string, Variant> hints) throws IOError, ScopeError
  {
    HashTable<string, Variant> result;

    result = yield search_internal (search_string, hints, SearchType.DEFAULT);

    return result;
  }

  public async HashTable<string, Variant> global_search (string search_string,
      HashTable<string, Variant> hints) throws IOError, ScopeError
  {
    HashTable<string, Variant> result;

    result = yield search_internal (search_string, hints, SearchType.GLOBAL);

    return result;
  }

  public async PreviewReplyRaw preview (string uri) throws IOError
  {
    var reply = PreviewReplyRaw ();
    
    Preview? response = _owner.preview_uri (uri);
    if (response == null)
      response = new NoPreview();

    reply.uri = uri;
    reply.renderer_name = response.get_renderer_name ();
    reply.properties = response.get_properties ();

    return reply;
  }

  public async void set_view_type (uint view_type_id) throws IOError
  {
    ViewType view_type = (ViewType) view_type_id;
    this.view_type = view_type;

    _owner.active = view_type == ViewType.LENS_VIEW ||
        (view_type == ViewType.HOME_VIEW && _owner.search_in_global);
  }

  public async void set_active_sources (string[] sources) throws IOError
  {
    _owner.set_active_sources_internal (sources);
  }
}

} /* namespace */
