/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Michal Hruby <michal.hruby@canonical.com>
 *
 */

public class Main
{
  static bool remote_scope_test = false;
  public static int main (string[] args)
  {
    if ("--with-remote-scope" in args) remote_scope_test = true;

    if (remote_scope_test)
    {
      Environment.set_variable ("LIBUNITY_LENS_DIRECTORY",
                                Config.TESTDIR + "/data", true);
    }
    else
    {
      Environment.set_variable ("LIBUNITY_LENS_DIRECTORY",
                                Config.TESTDIR, true);
    }

    Test.init (ref args);

    Test.add_data_func ("/Unit/Lens/Export", test_lens_export);
    if (remote_scope_test)
    {
      Test.add_data_func ("/Integration/RemoteScope/Initialize", test_remote_scope_init);
    }
    Test.add_data_func ("/Unit/LocalScope/Initialize", test_local_scope_init);
    Test.add_data_func ("/Unit/LocalScope/SearchOnView", test_local_scope_search_on_first_view);
    Test.add_data_func ("/Unit/Lens/Search", test_lens_search);
    Test.add_data_func ("/Unit/LocalScope/SearchChanged", test_local_scope_search);
    Test.add_data_func ("/Unit/LocalScope/MergeStrategy", test_merge_strategy);
    Test.add_data_func ("/Unit/Lens/ReturnAfterScopeFinish", test_lens_return_after_scope_finish);
    Test.add_data_func ("/Unit/Lens/SuccessiveSearches", test_lens_successive_searches);
    Test.add_data_func ("/Unit/Lens/TwoSearches", test_lens_two_searches);
    Test.add_data_func ("/Unit/Lens/ModelSync", test_lens_model_sync);
    Test.add_data_func ("/Unit/Lens/ReplyHint", test_lens_reply_hint);
    Test.add_data_func ("/Unit/Lens/Sources", test_lens_sources);

    Test.run ();

    return 0;
  }

  public static bool run_with_timeout (MainLoop ml, uint timeout_ms)
  {
    bool timeout_reached = false;
    var t_id = Timeout.add (timeout_ms, () => 
    {
      timeout_reached = true;
      debug ("Timeout reached");
      ml.quit ();
      return false;
    });

    ml.run ();

    if (!timeout_reached) Source.remove (t_id);

    return !timeout_reached;
  }

  static Unity.Lens exported_lens;
  static bool name_owned = false;

  public static void test_lens_export ()
  {
    // register us a name on the bus
    Bus.own_name (BusType.SESSION, "com.canonical.Unity.Lens.Test", 0,
                  () => {}, () => { name_owned = true; },
                  () => { debug ("Name lost"); assert_not_reached (); });

    var lens = new Unity.Lens ("/com/canonical/Unity/Lens/Test",
                               "unity_lens_test");
    lens.search_in_global = false;
    lens.search_hint = "Search hint";
    lens.export ();

    exported_lens = lens;
  }

  static Unity.Scope local_scope;

  public static void test_local_scope_init ()
  {
    assert (exported_lens != null);

    var scope = new Unity.Scope ("/com/canonical/Unity/LocalScope/Test");

    local_scope = scope;
    exported_lens.add_local_scope (scope);
  }

  public static void test_remote_scope_init ()
  {
    assert (exported_lens != null);

    bool scope_up = false;

    // the remote scope doesn't have a dbus service file installed, so we
    // expect that something (dbus-test-runner) started it already
    var ml = new MainLoop ();
    uint watch_id = Bus.watch_name (BusType.SESSION, 
                                    "com.canonical.Unity.Scope0.Test", 0,
                                    () => { scope_up = true; ml.quit (); },
                                    () => { scope_up = false; });

    run_with_timeout (ml, 2000);

    assert (scope_up == true);

    flush_bus ();
    // we need to wait a bit more to connect to the proxy
    // FIXME: find a better way to do this
    ml = new MainLoop ();
    Timeout.add (500, () => { ml.quit (); return false; });
    ml.run ();

    Bus.unwatch_name (watch_id);
    // should be still up
    assert (scope_up == true);
  }

  private static void call_lens_search (string search_string, Func<Variant?>? cb = null)
  {
    var bus = Bus.get_sync (BusType.SESSION);

    var vb = new VariantBuilder (new VariantType ("(sa{sv})"));
    vb.add ("s", search_string);
    vb.open (new VariantType ("a{sv}"));
    vb.close ();

    bus.call.begin ("com.canonical.Unity.Lens.Test",
                    "/com/canonical/Unity/Lens/Test",
                    "com.canonical.Unity.Lens",
                    "Search",
                    vb.end (),
                    null,
                    0,
                    -1,
                    null,
                    (obj, res) =>
    {
      try
      {
        var reply = bus.call.end (res);
        if (cb != null) cb (reply);
      }
      catch (Error e)
      {
        warning ("%s", e.message);
      }
    });
  }

  public static void test_lens_search ()
  {
    var ml = new MainLoop ();
    // make sure we got response from own_name, so we can send ourselves
    // a dbus method call
    Idle.add (() =>
    {
      if (name_owned) ml.quit ();
      return !name_owned;
    });

    ml.run ();

    call_lens_search ("foo");
  }

  public static void test_local_scope_search_on_first_view ()
  {
    assert (local_scope != null);

    var ml = new MainLoop ();
    bool got_global_search = false;
    bool got_lens_search = false;
    ulong sig_id = local_scope.search_changed.connect ((lens_search,
                                                        search_type) =>
    {
      assert (lens_search.search_string == "");
      if (search_type == Unity.SearchType.GLOBAL) got_global_search = true;
      else got_lens_search = true;
      lens_search.finished ();
      ml.quit ();
    });

    local_scope.set_view_type_internal (Unity.ViewType.HOME_VIEW);
    // wait for the signal or timeout
    run_with_timeout (ml, 1000);

    assert (got_global_search == true);

    // reset back
    local_scope.set_view_type_internal (Unity.ViewType.HIDDEN);
    SignalHandler.disconnect (local_scope, sig_id);
  }
  
  public static void test_local_scope_search ()
  {
    assert (local_scope != null);

    var ml = new MainLoop ();
    bool got_search_changed = false;
    ulong sig_id = local_scope.search_changed.connect ((lens_search) =>
    {
      assert (lens_search.search_string == "foo");
      got_search_changed = true;
      lens_search.finished ();
      ml.quit ();
    });

    // wait for the signal or timeout
    run_with_timeout (ml, 1000);

    assert (got_search_changed == true);
    SignalHandler.disconnect (local_scope, sig_id);
  }
  
  private class TestMergeStrategy : Unity.MergeStrategy, GLib.Object
  {
    public int n_rows = 0;
    
    public unowned Dee.ModelIter? merge_result (Dee.Model target, Variant[] row)
    {
      n_rows++;
      assert (row.length == 7);
      assert (row[0].get_string().has_suffix ("uri"));
      assert (row[1].get_string() == "icon");
      assert (row[2].get_uint32() == 0);
      assert (row[3].get_string() == "mimetype");
      assert (row[4].get_string() == "display-name");
      assert (row[5].get_string() == "comment");
      assert (row[6].get_string() == "dnd-uri");
      
      /* Since this method returns null,
       * no rows should ever go in the results model*/
      assert (target.get_n_rows () == 0);
      
      return null;
    }
  }
  
  public static void test_merge_strategy ()
  {
    assert (exported_lens != null);
    assert (local_scope != null);
    
    /* Since test cases are not completely isolated we need 
     * to instantate the default merge strategy when done */
    var old_merge_strategy = exported_lens.merge_strategy;
    local_scope.results_model.clear ();
    
    var merge_strategy = new TestMergeStrategy ();
    exported_lens.merge_strategy = merge_strategy;
    
    local_scope.results_model.append ("uri", "icon", 0, "mimetype",
                                      "display-name", "comment", "dnd-uri");
    local_scope.results_model.append ("uri", "icon", 0, "mimetype",
                                      "display-name", "comment", "dnd-uri");

    assert (merge_strategy.n_rows == 2);
    
    exported_lens.merge_strategy = old_merge_strategy;
  }

  private static void flush_bus ()
  {
    var bus = Bus.get_sync (BusType.SESSION);
    bus.flush_sync ();

    var ml = new MainLoop ();
    Idle.add (() => { ml.quit (); return false; });
    ml.run ();
    // this should flush the dbus method calls
  }

  public static void test_lens_return_after_scope_finish ()
  {
    assert (local_scope != null);

    var ml = new MainLoop ();
    bool got_search_changed = false;
    bool finish_called = false;

    Func<Variant?> cb = () =>
    {
      ml.quit ();
    };
    ulong sig_id = local_scope.search_changed.connect ((lens_search) =>
    {
      got_search_changed = true;
      Timeout.add (750, () =>
      {
        finish_called = true;
        lens_search.finished ();
        return false;
      });
    });

    // we want to make sure the Search DBus call doesn't return before we
    // call finished on the LensSearch instance
    call_lens_search ("qoo", cb);
    run_with_timeout (ml, 5000);

    assert (got_search_changed == true);
    assert (finish_called == true);

    SignalHandler.disconnect (local_scope, sig_id);
  }

  public static void test_lens_successive_searches ()
  {
    assert (local_scope != null);

    var ml = new MainLoop ();
    bool got_search_changed = false;
    bool finish_called = false;

    ulong sig_id = local_scope.search_changed.connect ((lens_search) =>
    {
      got_search_changed = true;
      Timeout.add (750, () =>
      {
        lens_search.results_model.clear ();
        lens_search.results_model.append ("", "", 0, "", "", "", "");
        lens_search.finished ();
        finish_called = true;
        return false;
      });
    });

    // we want to make sure the Search DBus call doesn't return before we
    // call finished on the LensSearch instance
    Variant? result1 = null;
    Variant? result2 = null;
    call_lens_search ("successive-searches", (result) =>
    {
      result1 = result;
    });
    // and another search with the same search string, it shouldn't return first
    call_lens_search ("successive-searches", (result) =>
    {
      result2 = result;
      ml.quit ();
    });
    
    run_with_timeout (ml, 5000);

    assert (got_search_changed == true);
    assert (finish_called == true);
    assert (result1.equal (result2));

    SignalHandler.disconnect (local_scope, sig_id);
  }

  public static void test_lens_two_searches ()
  {
    assert (local_scope != null);

    var ml = new MainLoop ();
    Cancellable? canc1 = null;
    Cancellable? canc2 = null;
    bool got_search_changed = false;
    uint finish_calls = 0;

    ulong sig_id = local_scope.search_changed.connect ((lens_search, search_type, cancellable) =>
    {
      got_search_changed = true;
      switch (lens_search.search_string)
      {
        case "foo1": canc1 = cancellable; break;
        case "foo2": canc2 = cancellable; break;
        default: assert_not_reached ();
      }

      var timer_id = Timeout.add (1000, () =>
      {
        finish_calls++;
        lens_search.finished ();

        if (finish_calls == 2) ml.quit ();
        return false;
      });

      ml.quit ();
    });

    string order = "";
    var reply_ml = new MainLoop ();
    int replies = 0;
    Func<Variant?> foo1_finished_cb = () =>
    {
      order += "1";
      if (++replies == 2) reply_ml.quit ();
    };
    Func<Variant?> foo2_finished_cb = () =>
    {
      order += "2";
      if (++replies == 2) reply_ml.quit ();
    };

    // we dont want to wait indefinitely
    var bad_timer = Timeout.add (2000, () => { assert_not_reached (); });
    call_lens_search ("foo1", foo1_finished_cb);
    ml.run ();
    flush_bus ();
    ml = new MainLoop ();
    call_lens_search ("foo2", foo2_finished_cb);
    ml.run ();

    Source.remove (bad_timer);

    assert (canc1 != null);
    assert (canc2 != null);

    assert (canc1.is_cancelled () == true);
    assert (canc2.is_cancelled () == false);

    flush_bus ();

    // the timers are still running and we need to wait for the replies
    reply_ml.run ();

    // make sure the first search finished earlier that the second
    assert (order == "12");

    SignalHandler.disconnect (local_scope, sig_id);
  }

  public static void test_lens_model_sync ()
  {
    assert (local_scope != null);

    bool got_search_changed = false;
    var ml = new MainLoop ();
    ulong sig_id = local_scope.search_changed.connect ((lens_search, search_type, cancellable) =>
    {
      assert (lens_search.search_string == "model-sync");
      got_search_changed = true;

      Timeout.add (300, () =>
      {
        var model = lens_search.results_model;
        model.append ("uri", "icon", 0, "mimetype", "display-name",
                      "comment", "dnd-uri");
        lens_search.finished ();
        return false;
      });
    });

    uint64 seqnum = 0;
    call_lens_search ("model-sync", (reply) =>
    {
      assert (reply != null);
      HashTable<string, Variant> ht = (HashTable<string, Variant>) reply.get_child_value (0);
      unowned Variant? seqnum_v = ht.lookup ("model-seqnum");
      assert (seqnum_v != null);
      seqnum = seqnum_v.get_uint64 ();
      ml.quit ();
    });

    run_with_timeout (ml, 3000);
    // libunity will emit warnings if the models are out-of-sync, those would
    // fail this test

    assert (got_search_changed == true);
    // FIXME: not too great test if previous tests did something with the model
    assert (seqnum > 0);

    SignalHandler.disconnect (local_scope, sig_id);
  }

  public static void test_lens_reply_hint ()
  {
    assert (local_scope != null);

    bool got_search_changed = false;
    var ml = new MainLoop ();
    ulong sig_id = local_scope.search_changed.connect ((lens_search, search_type, cancellable) =>
    {
      assert (lens_search.search_string == "reply-hint");
      got_search_changed = true;

      Timeout.add (10, () =>
      {
        var model = lens_search.results_model;
        model.append ("uri", "icon", 0, "mimetype", "display-name",
                      "comment", "dnd-uri");
        lens_search.set_reply_hint ("test-reply-hint",
                                    new Variant.string ("value"));
        lens_search.finished ();
        return false;
      });
    });

    string? hint_reply = null;
    call_lens_search ("reply-hint", (reply) =>
    {
      assert (reply != null);
      HashTable<string, Variant> ht = (HashTable<string, Variant>) reply.get_child_value (0);
      unowned Variant? hint_v = ht.lookup ("test-reply-hint");
      assert (hint_v != null);
      hint_reply = hint_v.get_string ();
      ml.quit ();
    });

    run_with_timeout (ml, 3000);

    assert (got_search_changed == true);
    // FIXME: not too great test if previous tests did something with the model
    assert (hint_reply == "value");

    SignalHandler.disconnect (local_scope, sig_id);
  }

  public static void test_lens_sources ()
  {
    assert (local_scope != null);

    local_scope.sources.add_option ("id1", "Source1", null);
    local_scope.sources.add_option ("id2", "Source2", null);

    var ml = new MainLoop ();
    // wait a bit for the model to update
    Idle.add (() => { ml.quit (); return false; });
    ml.run ();
    ml = new MainLoop ();

    ulong sig_id = local_scope.search_changed.connect ((lens_search) =>
    {
      assert (lens_search.search_string.has_prefix ("sources-test"));
      lens_search.finished ();
    });
    // do a search, so we can sync with the remote scope
    call_lens_search ("sources-test", () => { ml.quit (); });
    ml.run ();
    ml = new MainLoop ();
    // after this the sources *have* to be updated
    Idle.add (() => { ml.quit (); return false; });
    ml.run ();

    bool found1 = false;
    bool found2 = false;
    bool remote1 = false;
    bool remote2 = false;
    foreach (var filter_option in exported_lens.get_sources_internal ().options)
    {
      if (filter_option.id.has_suffix ("id1") && filter_option.id != "id1")
        found1 = true;
      else if (filter_option.id.has_suffix ("id2") && filter_option.id != "id2")
        found2 = true;
      else if (filter_option.id.has_suffix ("id1-remote"))
        remote1 = true;
      else if (filter_option.id.has_suffix ("id2-remote"))
        remote2 = true;
    }

    assert (found1);
    assert (found2);
    if (remote_scope_test)
    {
      assert (remote1);
      assert (remote2);
    }

    // =============== PART 2 of the test =============== //
    // we'll now remove one source
    local_scope.sources.remove_option ("id1");

    ml = new MainLoop ();
    // wait a bit for the model to update
    Idle.add (() => { ml.quit (); return false; });
    ml.run ();
    ml = new MainLoop ();
    // do another search to synchronize
    call_lens_search ("sources-test-2", () => { ml.quit (); });
    ml.run ();
    ml = new MainLoop ();
    // after this the sources *have* to be updated
    Idle.add (() => { ml.quit (); return false; });
    ml.run ();

    found1 = false;
    found2 = false;
    remote1 = false;
    remote2 = false;
    foreach (var filter_option in exported_lens.get_sources_internal ().options)
    {
      if (filter_option.id.has_suffix ("id1") && filter_option.id != "id1")
        found1 = true;
      else if (filter_option.id.has_suffix ("id2") && filter_option.id != "id2")
        found2 = true;
      else if (filter_option.id.has_suffix ("id1-remote"))
        remote1 = true;
      else if (filter_option.id.has_suffix ("id2-remote"))
        remote2 = true;
    }
    
    // make the the id1 sources are gone
    assert (found1 == false);
    assert (found2);
    if (remote_scope_test)
    {
      assert (remote1 == false);
      assert (remote2);
    }

    SignalHandler.disconnect (local_scope, sig_id);
  }
}
