/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;
using Unity.Protocol;

namespace Unity {

private class ScopeProxyLocal : GLib.Object, ScopeProxy
{
  public Scope scope { get; construct; }
  public OptionsFilter sources {
    get { return scope.sources; }
    set {}
  }
  public bool search_in_global {
    get { return scope.search_in_global; }
    set {}
  }
  public bool provides_personal_content {
    get { return scope.provides_personal_content; }
    set {}
  }
  public Dee.SerializableModel results_model {
    get { return scope.results_model; }
    set { }
  }
  public Dee.SerializableModel global_results_model {
    get { return scope.global_results_model; }
    set { }
  }

  public Dee.SerializableModel filters_model {
    get { return scope.get_filter_model (); }
    set { }
  }

  public ViewType view_type { get; set; }

  public ScopeProxyLocal (Scope scope)
  {
    Object (scope:scope);
  }

  construct
  {
    scope.bind_property ("search-in-global", this, "search-in-global", BindingFlags.DEFAULT);
    scope.sources.changed.connect (() => { this.notify_property ("sources"); });
    scope.bind_property ("provides-personal-content", this, "provides-personal-content", BindingFlags.DEFAULT);

    this.notify["view-type"].connect (() =>
    {
      scope.set_view_type_internal (view_type);
    });
    /* As we've set these in the constructor, let anyone that wants to watch
     * know that they've been set
     */
    Idle.add (()=>
    {
      notify_property ("search-in-global");
      notify_property ("sources");
      notify_property ("provides-personal-content");

      return false;
    });
  }

  /*
   * Implementation of the ScopeService interface
   */
  public async ActivationReplyRaw activate (string uri, ActionType action_type,
                                            HashTable<string, Variant> hints)
  {
    var reply = yield scope.activate (uri, action_type, hints);
    return reply;
  }

  public async HashTable<string, Variant> search (
      string search_string, HashTable<string, Variant> hints)
  {
    try {
      var result = yield scope.search (search_string, hints);
      return result;
    } catch (ScopeError.SEARCH_CANCELLED scope_error) {
      // all good
    } catch (Error e) {
      warning (@"Unable to search scope ('$search_string'): $(e.message)");
    }

    return new HashTable<string, Variant> (str_hash, str_equal);
  }

  public async HashTable<string, Variant> update_preview_property (
      string uri, HashTable<string, Variant> values) throws IOError
  {
      var reply = yield scope.update_preview_property (uri, values);
      return reply;
  }

  public async HashTable<string, Variant> global_search (
      string search_string, HashTable<string, Variant> hints)
  {
    try {
      var result = yield scope.global_search (search_string, hints);
      return result;
    } catch (ScopeError.SEARCH_CANCELLED scope_error) {
      // all good
    } catch (Error e) {
      warning (@"Unable to global_search scope ('$search_string'): $(e.message)");
    }

    return new HashTable<string, Variant> (str_hash, str_equal);
  }

  public async void set_active_sources (string[] sources)
  {
    scope.set_active_sources_internal (sources);
  }
}

} /* Namespace */
