/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "attachmentscache.h"

#include <QFile>
#include <QFileInfo>
#include <QDataStream>

#include "connectionmanager.h"
#include "configdata.h"

using namespace WebIssues;

AttachmentsCache::AttachmentsCache()
{
    openCache();

    flushCache( 0 );

    connect( configData, SIGNAL( settingsChanged() ), this, SLOT( settingsChanged() ) );
}

AttachmentsCache::~AttachmentsCache()
{
}

QString AttachmentsCache::findAttachment( int attachmentId )
{
    for ( int i = 0; i < m_cachedItems.count(); i++ ) {
        if ( m_cachedItems.at( i ).m_attachmentId == attachmentId ) {
            if ( i > 0 ) {
                Item item = m_cachedItems.at( i );
                m_cachedItems.removeAt( i );
                m_cachedItems.prepend( item );
            
                saveCache();
            }
            return connectionManager->locateCacheFile( "attachments/" + m_cachedItems.at( 0 ).m_name );
        }
    }
    return QString();
}

QString AttachmentsCache::allocAttachment( int attachmentId, const QString& name, int size )
{
    m_allocatedItem.m_attachmentId = attachmentId;
    m_allocatedItem.m_name = generateUniqueName( name );
    m_allocatedItem.m_size = size;

    flushCache( size );

    return connectionManager->locateCacheFile( "attachments/" + m_allocatedItem.m_name );
}

void AttachmentsCache::commitAttachment()
{
    m_cachedItems.prepend( m_allocatedItem );

    saveCache();
}

void AttachmentsCache::openCache()
{
    m_cachedItems.clear();

    QString path = connectionManager->locateCacheFile( "attachments/index.cache" );

    QFile file( path );
    if ( !file.open( QIODevice::ReadOnly ) )
        return;

    QDataStream stream( &file );
    stream.setVersion( 5 );

    qint32 version;
    stream >> version;

    if ( version != 1 )
        return;

    qint32 attachmentId;
    while ( !( stream >> attachmentId ).atEnd() && attachmentId != 0 ) {
        QString name;
        stream >> name;
        qint32 size;
        stream >> size;

        Item item;
        item.m_attachmentId = attachmentId;
        item.m_name = name;
        item.m_size = size;

        m_cachedItems.append( item );
    }
}

void AttachmentsCache::saveCache()
{
    QString path = connectionManager->locateCacheFile( "attachments/index.cache" );

    QFile file( path );
    if ( !file.open( QIODevice::WriteOnly ) )
        return;

    QDataStream stream( &file );
    stream.setVersion( 5 );

    stream << (qint32)1; // increment version when adding/changing fields

    for ( int i = 0; i < m_cachedItems.count(); i++ ) {
        stream << (qint32)m_cachedItems.at( i ).m_attachmentId;
        stream << m_cachedItems.at( i ).m_name;
        stream << (qint32)m_cachedItems.at( i ).m_size;
    }
    stream << (qint32)0;
}

void AttachmentsCache::settingsChanged()
{
    flushCache( 0 );
}

void AttachmentsCache::flushCache( int allocated )
{
    int limit = configData->attachmentsCacheSize() * 1024 * 1024;
    int occupied = ( allocated + 4095 ) & ~4096;

    for ( int i = 0; i < m_cachedItems.count(); i++ )
        occupied += ( m_cachedItems.at( i ).m_size + 4095 ) & ~4096;

    bool modified = false;
    int index = m_cachedItems.count() - 1;

    while ( index >= 0 && occupied > limit ) {
        QString path = connectionManager->locateCacheFile( "attachments/" + m_cachedItems.at( index ).m_name );
        if ( !QFile::exists( path ) || QFile::remove( path ) ) {
            occupied -= ( m_cachedItems.at( index ).m_size + 4095 ) & ~4096;
            m_cachedItems.removeAt( index );
            modified = true;
        }
        index--;
    }

    if ( modified )
        saveCache();
}

QString AttachmentsCache::generateUniqueName( const QString& name )
{
    QString path = connectionManager->locateCacheFile( "attachments/" + name );

    QFileInfo info( path );

    if ( !info.exists() )
        return name;

    QString baseName = info.baseName();
    QString suffix = info.completeSuffix();
    if ( !suffix.isEmpty() )
        suffix.prepend( '.' );

    for ( int number = 2; ; number++ ) {
        QString generatedName = QString( "%1(%2)%3" ).arg( baseName ).arg( number ).arg( suffix );
        path = connectionManager->locateCacheFile( "attachments/" + generatedName );
        if ( !QFile::exists( path ) )
            return generatedName;
    }
}

#include "attachmentscache.moc"
