/* LoadWorkspace.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.main.actions;

import java.awt.Window;
import java.io.File;
import java.text.MessageFormat;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import org.grinvin.gui.FileFilters;
import org.grinvin.io.WorkspaceLoader;
import org.grinvin.io.WorkspaceSaver;
import org.grinvin.io.WorkspaceFileView;
import be.ugent.caagt.swirl.actions.FileOpenAction;
import org.grinvin.workspace.Workspace;
import org.grinvin.workspace.WorkspaceException;


/**
 *
 */
public class LoadWorkspace extends FileOpenAction {
    
    // TODO: a lot of this stuff should not be in this file!
    
    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    //
    private static final ResourceBundle BUNDLE
            = ResourceBundle.getBundle("org.grinvin.main.actions.resources");
    
    public LoadWorkspace(Window parent) {
        super(parent, false,
            FileFilters.GWS_FILTER, FileFilters.WORKSPACE_FILTER
        );
        fileChooser.setFileView(new WorkspaceFileView());
    }
    
    // implements performOpen
    public void openFile (File file) {
        loadWorkspace(checkFileForLoad(file));
    }
    
    private void loadWorkspace(File file) {
        if(!file.exists())
            throw new RuntimeException("Workspace file " + file + " does not exist");
        try {
            if(file.getName().toLowerCase().endsWith(".gws")) {
                if(!file.isDirectory()){
                    WorkspaceLoader.load(Workspace.getInstance(), file);
                } else if (JOptionPane.showConfirmDialog(null, BUNDLE.getString("workspace.load.force.wrong.message"), BUNDLE.getString("workspace.load.force.wrong.title"), JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) {
                    loadWorkspaceFromDirectory(file);
                }
            } else if(file.getName().toLowerCase().endsWith(".workspace")) {
                if(file.isDirectory()) {
                    loadWorkspaceFromDirectory(file);
                } else if (JOptionPane.showConfirmDialog(null, BUNDLE.getString("workspace.load.force.wrong.message"), BUNDLE.getString("workspace.load.force.wrong.title"), JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) {
                    WorkspaceLoader.load(Workspace.getInstance(), file);
                }
            } else {
                if (JOptionPane.showConfirmDialog(null, BUNDLE.getString("workspace.load.force.unknown.message"), BUNDLE.getString("workspace.load.force.unknown.title"), JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) {
                    if(file.isDirectory()) {
                        loadWorkspaceFromDirectory(file);
                    } else {
                        WorkspaceLoader.load(Workspace.getInstance(), file);
                    }
                }
            }
        } catch (WorkspaceException ex) {
            LOGGER.log(Level.WARNING, "Failed to load workspace from " + file, ex);
        }
    }
    
    private void loadWorkspaceFromDirectory(File file) throws WorkspaceException {
        assert file.isDirectory() : "loadWorkspaceDirectory called with a file as argument.";
        if(new File(file, "meta.xml").exists()) {
            WorkspaceLoader.loadFromDirectory(Workspace.getInstance(), file);
            saveImportedWorkspace();
        } else {
            //file has wrong structure
            throw new WorkspaceException("meta.xml was not found", file);
        }
    }
    
    /**
     * Ask the user for a workspace location.
     * @return directory which holds a workspace, or null when no directory
     * could be chosen.
     */
    public static File loadNewWorkspace() {
        JFileChooser chooser = new JFileChooser();
        chooser.addChoosableFileFilter(FileFilters.GWS_FILTER);
        chooser.addChoosableFileFilter(FileFilters.WORKSPACE_FILTER);
        chooser.setFileView(new WorkspaceFileView());
        
        if (chooser.showDialog(null, BUNDLE.getString("workspace.save.caption")) == JFileChooser.APPROVE_OPTION) {
            return checkFileForLoad(chooser.getSelectedFile());
        }
        return null;
    }
    
    /**
     * Ask the user for a workspace location to save the imported workspace to.
     */
    public static void saveImportedWorkspace() {
        JOptionPane.showMessageDialog(null, BUNDLE.getString("workspace.imported.message"), BUNDLE.getString("workspace.imported.title"), JOptionPane.INFORMATION_MESSAGE);
        
        JFileChooser chooser = new JFileChooser();
        chooser.setFileFilter(FileFilters.GWS_FILTER);
        chooser.setFileView(new WorkspaceFileView());
        
        boolean workspaceSaved = false;
        while(!workspaceSaved) {
            if (chooser.showDialog(null, BUNDLE.getString("workspace.save.caption")) == JFileChooser.APPROVE_OPTION) {
                try {
                    File file = checkFileForSave(chooser.getSelectedFile());
                    if(file != null) {
                        WorkspaceSaver.saveAs(Workspace.getInstance(), file);
                        workspaceSaved = true;
                    }
                } catch (WorkspaceException e) {
                    LOGGER.log(Level.SEVERE, "Saving workspace failed", e);
                    JOptionPane.showMessageDialog(null, BUNDLE.getString("workspace.savefailed.message"), BUNDLE.getString("workspace.savefailed.title"), JOptionPane.WARNING_MESSAGE);
                }
            } else {
                System.exit(0);
            }
        }
    }
    
    private static File checkFileForLoad(File file) {
        if (!file.exists()) {
            if(new File(file.getParentFile(), file.getName() + ".gws").exists())
                return new File(file.getParentFile(), file.getName() + ".gws");
            else if (new File(file.getParentFile(), file.getName() + ".workspace").exists())
                return new File(file.getParentFile(), file.getName() + ".workspace");
        }
        return file;
    }
    
    private static File checkFileForSave(File file) throws WorkspaceException {
        if (!file.getName().toLowerCase().endsWith(".gws"))
            file = new File(file.getParentFile(), file.getName() + ".gws");
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new WorkspaceException("File is a directory.", file);
            } else if (JOptionPane.showConfirmDialog(null, MessageFormat.format(BUNDLE.getString("workspace.overwrite.message"), file), BUNDLE.getString("workspace.overwrite.title"), JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) {
                return file;
            } else {
                return null;
            }
        } else {
            return file;
        }
    }
}
