/* EdgeConnectivity.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Invariant computer which returns the edge-connectivity.
 */
public class EdgeConnectivity extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        //first calculate minimum degree (i.e. upperbound for edge-connectivity)
        int[][] adjlist = bundle.adjacencyList();
        if(adjlist.length<2)
            return new IntegerValue(0, this);
        int mindeg = Integer.MAX_VALUE;
        for (int[] neighbours : adjlist)
            if (neighbours.length < mindeg)
                mindeg = neighbours.length;
        if(mindeg==adjlist.length-1)
            //complete graph
            return new IntegerValue(mindeg, this);
        
        int[][] cut = new int[mindeg][2];
        int minimumSize = mindeg;
        
        if(!isConnected(adjlist, cut, 0))
            //disconnected graph
            return new IntegerValue(0, this);
        
        
        for(int i=0; i<adjlist.length; i++){
            cut[0][0]=i;
            for(int j : adjlist[i]){
                if(j>i){
                    cut[0][1]=j;
                    minimumSize = findMinimumCutSize(adjlist, cut, minimumSize, 1);
                }
            }
        }
        
        return new IntegerValue(minimumSize, this);
    }
    
    private int findMinimumCutSize(int[][] adjlist, int[][] cut, int minimumSize, int currentSize){
        if(currentSize>=minimumSize)
            return minimumSize;
                
        if(!isConnected(adjlist, cut, currentSize))
            return currentSize;
        
        for(int i = cut[currentSize-1][0]; i<adjlist.length; i++){
            cut[currentSize][0]=i;
            for(int j : adjlist[i]){
                if(j>i && !(cut[currentSize-1][0]==i && cut[currentSize-1][1]==j)){
                    cut[currentSize][1]=j;
                    minimumSize = findMinimumCutSize(adjlist, cut, minimumSize, currentSize+1);
                }
            }
        }
        
        return minimumSize;
    }
    
    private boolean isConnected(int[][] adjlist, int[][] cut, int cutSize){
        int n = adjlist.length;
        int[][] dist = new int[n][n];
        
        // compute the adjacency matrix
        for(int i = 0; i<n; i++)
            for(int vertex : adjlist[i])
                dist[i][vertex] = 1;

        //remove the edges in the cut
        for(int i = 0; i<cutSize; i++){
            dist[cut[i][0]][cut[i][1]]=0;
            dist[cut[i][1]][cut[i][0]]=0;
        }
        
        // compute the distance matrix
        int d = 1;
        int count;
        do {
            int d1 = d++;
            count = 0;
            for (int i = 0; i < n; i++)
                for (int j = i + 1; j < n; j++)
                    if (dist[i][j] == 0)
                        for (int k = 0; k < n; k++)
                            if (dist[i][k] == 1 && dist[j][k] == d1) {
                dist[i][j] = d;
                dist[j][i] = d;
                count++;
                break;
                            }
        } while (count > 0);
        
        //check if any vertices are at distance infinity of each other
        for(int i = 0; i<n; i++)
            for(int j = i+1; j<n; j++)
                if(dist[i][j]==0)
                    return false;
        
        return true;
    }
        
    public String getInvariantId() {
        return "org.grinvin.invariants.EdgeConnectivity";
    }
    
}
