/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * main.h --
 *
 * Shared state used by the Linux kernel module implementation of driver for the
 * VMware Host/Guest filesystem.
 */

#ifndef _HGFS_DRIVER_MAIN_H_
# define _HGFS_DRIVER_MAIN_H_

#include <linux/version.h>
#include <linux/linkage.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include "compat_sched.h"
#include <linux/slab.h>
#include <linux/file.h>
#include <linux/pagemap.h>
#include <asm/uaccess.h>

#include "vm_assert.h"
#include "hgfs.h"

/* Data kept in each superblock in sb->u */

typedef struct HgfsSuperInfo {
   struct super_block *sb;          /* Superblock for this fs */
   struct file *userFile;           /* File that user program opened */
   uid_t uid;                       /* Uid of user who mounted this fs  */
   ino_t nextIno;                   /* Next unused inode number */
   wait_queue_head_t waitQ;         /* Server process(es) wait(s) on this queue */
   struct list_head outQ;           /* List of outgoing (unprocessed) requests  */
} HgfsSuperInfo;



#define HGFS_REQ_STATE_INACTIVE (struct file *)0
#define HGFS_REQ_STATE_FINISHED (struct file *)1

/*
 * A request to be sent to the user process
 */
typedef struct HgfsReq {
   /* Links to place the object on various lists */
   struct list_head list;

   /*
    * Process that is waiting for the reply. If the sending process catches a
    * signal while waiting it sets this to NULL. HgfsDevWrite checks this when
    * a reply packet comes in and if no process is waiting it discards the
    * reply and frees the request object, since the requesting process has
    * already returned.
    */
   struct task_struct *client;

   /* ID of this request, also its requestPool index */
   uint32 id;

   /*
    * Current state of this request, and also authorization check. Its
    * value:meaning pairs are as follows:
    *
    * HGFS_REQ_STATE_INACTIVE:
    *    Inactive. The request is on the free list, or is waiting on the outQ.
    *    We don't need to distinguish between these two states because the
    *    request will also be on one of those two lists.
    *
    * A struct file *:
    *    The request has been sent and is waiting for a reply. The value of
    *    this field should be a file pointer that matches the one in the
    *    HgfsSuperInfo->userFile field for this fs's superblock. When the reply
    *    is received from the server process, we look the request up in the
    *    requestPool using the id field in the reply packet, and check that the
    *    struct file * used by the server to send the packet matches the
    *    req->stateFile field; if it does not, then either the reply is
    *    invalid.
    *
    * HGFS_REQ_STATE_FINISHED:
    *    The request's reply has been received and verified as described above.
    *    After the reply has been processed, it will be returned to the free
    *    list.
    */
   struct file *stateFile;

   /* Used to detect if there was an error sending a request. */
   int error;

   /* Total size of the packet */
   uint32 packetSize;

   /* Packet of data, for both incoming and outgoing messages */
   char packet[HGFS_PACKET_MAX];
} HgfsReq;


/*
 * Global spinlock to protect our data structures. Because
 * our pool of request objects are shared globally among all
 * mounts, this lock is also global.
 *
 * Eventually we could add a per-superblock lock also
 * if we discover that we need/want finer grained
 * locking for per-superblock-only data structures. Same
 * for per-request structures.
 */
extern spinlock_t hgfsBigLock;


/*
 * Proc device f_ops, needed in driver.c and defined in dev.c
 */
extern struct file_operations HgfsProcDevOperations;

/*
 * Hgfs filesystem type, needed in main.c and defined in driver.c
 */
extern struct file_system_type hgfsType;

/*
 * Global state defined in main.c
 */
extern HgfsReq requestPool[];        /* Pool of request objects */
extern struct semaphore requestSem;  /* Limits the max number of outstanding requests */
extern struct list_head reqFreeList; /* Free list of available (but uninitialized) requests */


/*
 * Global request structures
 */

#define HGFS_MAX_OUTSTANDING_REQS 3     // Max _global_ number of simultaneous requests allowed

/* 
 * XXX These go in driver.c, they are here only temporarily until I
 * can separate the per-superblock fs info from the common state.
 */
/* Global filesystem definitions */

#define HGFS_BLOCKSIZE 1024             // Blocksize to be set in superblock (XXX how is this used?)
/* Inode number of the root inode */
#define HGFS_ROOT_INO 0
/*
 * Start giving out inode numbers here, for all inodes except the root inode.
 * Must be > HGFS_ROOT_INO to detect wraparound
 */
#define HGFS_START_INO (HGFS_ROOT_INO + 1)
#define HGFS_IO_MAX 4096                /* Max amount of read/write data per server request.
                                         * Must be smaller than HGFS_PACKET_MAX by a large enough
                                         * margin to allow for headers and other request fields.
                                         */


/*
 * This is used by the proc device operations and by read_super at
 * mount time. When our proc device is opened, we create an
 * HgfsSuperInfo and stick a pointer to it in file->private_data.
 * Reads and Writes on that file then use that pointer to access the
 * SuperInfo, which has our outQ, etc.
 */
#define HGFS_SET_DEV_TO_COMMON(file, common) do { (file)->private_data = (common); } while (0)
#define HGFS_DEV_TO_COMMON(file) ((HgfsSuperInfo *)(file)->private_data)


/*
 * Logging stuff
 */

#ifdef VMX86_DEVEL
extern int LOGLEVEL_THRESHOLD;

#define LOG(level, args) ((void) (LOGLEVEL_THRESHOLD >= (level) ? (printk args) : 0))
#else
#define LOG(level, args)
#endif

void Panic(const char *fmt, ...); // IN
char *vmstrdup(char const *string); // IN

void HgfsFreeRequest(HgfsReq *req); // IN/OUT

void HgfsCleanupProcDevice(void);
int HgfsSetupProcDevice(void);

#endif // _HGFS_DRIVER_MAIN_H_
