# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).

import os
from textwrap import dedent

from xdg import BaseDirectory

from gdp import (
    Config,
    )
from testing import GeditTestCase


class ConfigTestCase(GeditTestCase):

    def make_config(self, text=''):
        # Shadow the base class with a config without a module.
        config_file = self.make_file(text, suffix='.conf')
        test_config = Config()
        test_config.dir_path = os.path.dirname(config_file.name)
        test_config.file_path = config_file.name
        return test_config, config_file

    def test_init(self):
        # The default conf data is loaded.
        test_config = Config()
        self.assertIs(None, test_config._loaded_file_path)
        self.assertNotEqual(0, test_config.sections())
        self.assertIn(BaseDirectory.xdg_config_home, test_config.dir_path)
        self.assertIn('/gdp', test_config.dir_path)
        self.assertIn('/gdp/gdp.conf', test_config.file_path)

    def test_load(self):
        # Read the optional conf file.
        text = dedent("""\
            [completer]
            show_accel = <Control>a
            """)
        test_config, config_file = self.make_config(text)
        test_config.load()
        self.assertEqual([config_file.name], test_config._loaded_file_path)
        self.assertEqual(
            '<Control>a', test_config.get('completer', 'show_accel'))

    def test_load_without_conf(self):
        # Read without optional conf file.
        test_config, config_file = self.make_config()
        config_file.close()
        test_config.load()
        self.assertEqual([], test_config._loaded_file_path)

    def test_dump(self):
        # Write the optional conf file.
        test_config, config_file = self.make_config()
        test_config.set('completer', 'show_accel', '<Control>a')
        test_config.dump()
        text = dedent("""\
            [completer]
            show_accel = <Control>a
            suggest_completions = False
            """)
        self.assertIn(text, config_file.read())

    def test_getlist(self):
        test_config, config_file = self.make_config()
        self.assertEqual(
            ['<Current File>', '<Working Directory>'],
            test_config.getlist('finder', 'paths'))

    def test_setlist(self):
        test_config, config_file = self.make_config()
        test_config.setlist(
            'finder', 'paths',
            ['<Current File>', '<Working Directory>', './', './eg'])
        self.assertEqual(
            ['<Current File>', '<Working Directory>', './', './eg'],
            test_config.getlist('finder', 'paths'))

    def test_dump_list(self):
        # verify that config parser serialises lists as expected.
        test_config, config_file = self.make_config()
        test_config.setlist(
            'finder', 'paths',
            ['<Current File>', '<Working Directory>', '\t./\t'])
        test_config.dump()
        text = dedent("""\
            [finder]
            paths = <Current File>
            \t<Working Directory>
            \t\t./\t
            """)
        self.assertIn(text, config_file.read())

    def test_completer(self):
        # Verify the default completer section.
        test_config = Config()
        self.assertEqual(
            '<Control>space', test_config.get('completer', 'show_accel'))
        self.assertEqual(
            False, test_config.getboolean('completer', 'suggest_completions'))

    def test_format(self):
        # Verify the default completer section.
        test_config = Config()
        self.assertEqual(
            False, test_config.getboolean('formatter', 'report_only_errors'))

    def test_finder(self):
        # Verify the default completer section.
        test_config = Config()
        self.assertEqual(
            ['<Current File>', '<Working Directory>'],
            test_config.get('finder', 'paths').split('\n'))
        self.assertEqual(
            [''],
            test_config.get('finder', 'matches').split('\n'))
        self.assertEqual(
            ['<Any Text File>'],
            test_config.get('finder', 'files').split('\n'))
        self.assertEqual(
            [''],
            test_config.get('finder', 'substitutions').split('\n'))
