# -*- coding: utf-8 -*-
# advanced kde distutils
# Copyright (c) 2007 Sebastian Wiesner
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.


import os
import subprocess
import gzip

from glob import glob

from distutils import core
from distutils import dir_util
from distutils import spawn
from distutils import util

from distutils.version import LooseVersion
from distutils.dist import Distribution
from distutils.cmd import Command, install_misc
from distutils.command.build import build
from distutils.command.install import install
from distutils.command.clean import clean
from distutils.command.install_data import install_data
from distutils.command.install_lib import install_lib


# FIXME: these could be options
PO_DIRECTORY = 'po'
INSTALL_LOG = 'installed_files'

# central function
def setup(**kwargs):
    if 'cmdclass' not in kwargs:
        kwargs['cmdclass'] = {}

    cmdclasses = {
        'messages': Messages,
        'build_messages': BuildMessages,
        'build_ui': BuildUI,
        'build_manpages': BuildManPages,
        'build': KDEBuild,
        'install_links': InstallLinks,
        'install_sources': InstallSources,
        'install_appdata': InstallAppData,
        'install_messages': InstallMessages,
        'install_desktop': InstallDesktop,
        'install_icons': InstallIcons,
        'install_manpages': InstallManPages,
        'install': KDEInstall,
        'uninstall': KDEUninstall,
        'clean': KDEClean
        }

    for key in cmdclasses:
        kwargs['cmdclass'].setdefault(key, cmdclasses[key])

    kwargs.setdefault('distclass', KDEDistribution)

    core.setup(**kwargs)


def has_messages(self):
    return bool(self.distribution.sources)


def has_app_data(self):
    return bool(self.distribution.appdata)


def has_ui(self):
    return bool(self.distribution.ui)


def has_links(self):
    return bool(self.distribution.links) and has_sources(self)


def has_sources(self):
    return bool(self.distribution.sources)


def has_desktop(self):
    return bool(self.distribution.desktop)


def has_icons(self):
    return bool(self.distribution.icons)


def has_manpages(self):
    return bool(self.distribution.manpages)


def is_uifile(item):
    """Checks if a file is a qt designer file"""
    return item.endswith('.ui')


def ui2py(item):
    """Returns the python module for the ui file `item`"""
    return os.path.basename(os.path.splitext(item)[0] + '.py')


class KDEDistribution(Distribution):
    """
    :ivar sources: Python and designer files to be installed
    :ivar appdata: Additional application data
    :ivar links: Executable links to files in sources
    :ivar uiimports: Additional imports used in ui files
    :ivar desktop: Desktop files
    :ivar icons: Icons and icons
    :ivar manpages: man pages
    :ivar ui: All ui files in `sources`
    :ivar py: All py files in `sources`
    """
    
    def __init__(self, attrs):
        self.min_qt_version = None
        self.min_kde_version = None
        self.sources = None
        self.appdata = None
        self.links = None
        self.uiimports = None
        self.desktop = None
        self.icons = None
        self.manpages = None
        Distribution.__init__(self, attrs)

    def finalize_options(self):
        """Finalize options: extract ui files from sources"""
        self.ui = filter(is_uifile, self.sources)
        self.py = filter(lambda item: item.endswith('.py'), self.sources)
        if self.min_qt_version:
            self.min_qt_version = LooseVersion(self.min_qt_version)
        if self.min_kde_version:
            self.min_kde_version = LooseVersion(self.min_kde_version)


class Messages(Command):   
    description = 'Extract messages from source files'

    user_options = [('xgettext-exe=', None,
                     'Full path to the xgetext executable'),
                    ('kde-pot=', None,
                     'Location of the KDE pot file'),
                    ('build-ui=', None,
                     'build directory of ui files'),
                    ('msgmerge-exe=', None,
                     'Full path to the msgmerge executable'),
                    ('skip-build', None, 'skip build steps')]

    boolean_options = ['skip-build']

    def initialize_options(self):
        self.xgettext_exe = None
        self.msgmerge_exe = None
        self.kde_pot = None
        self.skip_build = None
        self.build_ui = None

    def finalize_options(self):
        self.set_undefined_options('build', ('build_ui', 'build_ui'))
        self.set_undefined_options('install', ('skip_build', 'skip_build'))
        
        if self.xgettext_exe is None:
            self.announce('Searching xgettext...')
            self.xgettext_exe = spawn.find_executable('xgettext')
            if self.xgettext_exe is None:
                raise SystemExit('Couldn\'t find "xgettext".')
            self.announce('  ...xgettext found at %s' % self.xgettext_exe)

        if self.msgmerge_exe is None:
            self.announce('Searching msgmerge...')
            self.msgmerge_exe = spawn.find_executable('msgmerge')
            if self.msgmerge_exe is None:
                raise SystemExit('Couldn\'t find "msgmerge".')
            self.announce('  ...msgmerge found at %s' % self.msgmerge_exe)

        if self.kde_pot is None:
            install = self.get_finalized_command('install')
            includes = install.query_kde_directory('include')
            self.kde_pot = os.path.join(includes, 'kde.pot')

    def run(self):
        if not self.skip_build:
            self.run_command('build_ui')
        if not os.path.exists(PO_DIRECTORY):
            os.mkdir(PO_DIRECTORY)
        name = self.distribution.get_name() + '.pot'
        target = os.path.join(PO_DIRECTORY, name)
        cmd = [self.xgettext_exe, '-o', target, '-ki18n', '-ktr2i18n', 
               '-kI18N_NOOP', '-ktranslate', '-kaliasLocale', '-x',
               self.kde_pot]
        cmd.extend(self.distribution.py)
        cmd.extend([os.path.join(self.build_ui, ui2py(item)) for item in
                    self.distribution.ui])
        self.spawn(cmd)

        for po_file in glob(os.path.join(PO_DIRECTORY, '*.po')):
            cmd = [self.msgmerge_exe, '-q', '-o', po_file, po_file, target]
            self.spawn(cmd)


class BuildUI(Command):
    description = 'Compile ui files'

    user_options = [('build-dir=', 'd', 'directory to build ui files in'),
                    ('pyuic-exe=', None, 'Path to the pyuic executable')]

    import_pattern = 'from %s import *\n'

    def initialize_options(self):
        self.build_dir = None
        self.pyuic_exe = None

    def finalize_options(self):
        self.set_undefined_options('build',
                                   ('build_ui', 'build_dir'),
                                   ('pyuic_exe', 'pyuic_exe'))

    def run(self):
        self.announce('Building ui files...')
        self.mkpath(self.build_dir)

        for uifile in self.distribution.ui:
            self.uicompile(uifile)

    def uicompile(self, uifile):
        """Compiles `uifile`"""
        target = os.path.join(self.build_dir, ui2py(uifile))
        cmd = [self.pyuic_exe, '-o', target, '-p' , '4', '-tr', 'i18n',
               uifile]
        # note that we need cannot rely on distutils.spawn here, because
        # we need the stdout of pyuic_exe
        process = subprocess.Popen(cmd, stdout=subprocess.PIPE)
        process.wait()


class BuildMessages(Command):
    description = 'Compile message catalogs'

    user_options = [('build-dir=', 'd',
                     'directory to build message catalogs in'),
                    ('msgfmt-exe=', None, 'Path to the msgfmt executable')]

    def initialize_options(self):
        self.build_dir = None
        self.msgfmt_exe = None

    def finalize_options(self):
        self.set_undefined_options('build',
                                   ('msgfmt_exe', 'msgfmt_exe'),
                                   ('build_messages', 'build_dir'))

    def run(self):
        self.announce('Building po catalogs...')
        self.mkpath(self.build_dir)
        
        po_files = os.path.join('po', '*.po')
        po_files = glob(po_files)
        for po_file in po_files:
            language = os.path.splitext(os.path.basename(po_file))[0]
            self.announce('Building catalog %s...' % language)
            target = os.path.join(self.build_dir, language + '.gmo')
            cmd = [self.msgfmt_exe, '-o', target, po_file]
            self.spawn(cmd)
        self.announce('Done building message catalogs.')


class BuildManPages(Command):
    description = 'Gzip manpages'

    user_options = [('build-dir=', 'd',
                     'directory to put gzipped manpages in in')]

    def initialize_options(self):
        self.build_dir = None

    def finalize_options(self):
        self.set_undefined_options('build', ('build_manpages', 'build_dir'))

    def run(self):
        self.announce('Gzipping manpages...')
        self.mkpath(self.build_dir)

        for man in self.distribution.manpages:
            if not isinstance(man, basestring):
                man = man[0]
            target = os.path.join(self.build_dir, man + '.gz')
            self.announce('Gzipping manpage %s...' % man)
            stream = open(man)
            gzipped_stream = gzip.open(target, 'w')
            data = stream.read()
            gzipped_stream.write(data)
            stream.close()
            gzipped_stream.close()
        self.announce('Done gzipping manpages')


class KDEBuild(build):
    user_options = build.user_options[:]
    user_options.extend([
        ('build-ui=', None, 'build directory for ui files'),
        ('build-messages=', None, 'build directory for messages'),
        ('build-manpages=', None, 'build directory for manpages'),
        ('msgfmt-exe=', None, 'Path to the msgfmt executable'),
        ('pyuic-exe=', None, 'Path to the pyuic executable')])

    sub_commands = build.sub_commands[:]
    sub_commands.append(('build_ui', has_ui))
    sub_commands.append(('build_messages', has_messages))
    sub_commands.append(('build_manpages', has_manpages))

    def initialize_options(self):
        self.build_messages = None
        self.build_ui = None
        self.build_manpages = None
        self.msgfmt_exe = None
        self.pyuic_exe = None
        build.initialize_options(self)

    def finalize_options(self):
        if self.build_messages is None:
            self.build_messages = os.path.join(self.build_base,
                                               'po')
        if self.build_ui is None:
            self.build_ui = os.path.join(self.build_base, 'ui')

        if self.build_manpages is None:
            self.build_manpages = os.path.join(self.build_base, 'man')
            
        if self.msgfmt_exe is None:
            self.announce('Searching msgfmt...')
            self.msgfmt_exe = spawn.find_executable('msgfmt')
            if self.msgfmt_exe is None:
                raise SystemExit('Couldn\'t find "msgfmt".')
            self.announce('  ...msgfmt found at %s' % self.msgfmt_exe)
            
        if self.pyuic_exe is None:
            self.announce('Searching pyuic...')
            self.pyuic_exe = spawn.find_executable('pyuic')
            if self.pyuic_exe is None:
                raise SystemExit('Couldn\'t find "pyuic".')
            self.announce('  ...pyuic found at %s' % self.pyuic_exe)

        build.finalize_options(self)
            

class InstallStuff(install_misc):
    """Base class for some install commands"""
    
    def mkpath (self, name, mode=0777):
        return dir_util.mkpath(name, mode, dry_run=self.dry_run)


class InstallSources(install_lib):
    description = 'Install python and qt designer source files'
    
    user_options = install_lib.user_options[:]
    user_options.append(('build-ui=', None,
                         'ui directory (where to get ui files from)'))
    
    def initialize_options(self):
        self.build_ui = None
        install_lib.initialize_options(self)

    def finalize_options(self):
        self.set_undefined_options('install',
                                   ('install_sources', 'install_dir'))
        self.set_undefined_options('build',
                                   ('build_ui', 'build_ui'))
        install_lib.finalize_options(self)
        self.sources = self.distribution.sources
        self.outfiles = []

    def run(self):
        if not os.path.exists(self.install_dir):
            self.outfiles.extend(self.mkpath(self.install_dir))

        for item in self.sources:
            if is_uifile(item):
                # FIXME: this doesn't directory structure => no packages
                # copy the compiled file
                item = os.path.join(self.build_ui, ui2py(item))
            if os.path.isfile(item):
                f, copied = self.copy_file(item, self.install_dir)
                self.outfiles.append(f)
            else:
                self.warn('Unable to find %s...' % item)
                
        self.byte_compile(self.outfiles)
        self.outfiles.extend(self._bytecode_filenames(self.outfiles))

    def mkpath (self, name, mode=0777):
        return dir_util.mkpath(name, mode, dry_run=self.dry_run)

    def get_outputs(self):
        return self.outfiles


class InstallAppData(InstallStuff):
    description = 'Installs additional application data'
    
    def finalize_options(self):
        self._install_dir_from('install_app_data')
        self.appdata = self.distribution.appdata

    def run(self):
        if not os.path.exists(self.install_dir):
            self.outfiles.extend(self.mkpath(self.install_dir))

        for item in self.appdata:
            if isinstance(item, basestring):
                # put it right into the installation directory
                if os.path.isfile(item):
                    f, copied = self.copy_file(item, self.install_dir)
                    self.outfiles.append(f)
                elif os.path.isdir(item):
                    target =  os.path.join(self.install_dir, item)
                    files = self.copy_tree(item, target)
                    self.outfiles.extend(files)
                else:
                    self.warn('Unable to find %s...' % item)
            else:
                # assume we have a tupel-like thing here. target directory
                # relative to install_dir is in first element
                target_dir = item[0]
                if self.root:
                    target_dir = util.change_root(self.root, target_dir)
                else:
                    target_dir = os.path.join(self.install_dir, target_dir)
            
                for fso in item[1]:
                    if os.path.isdir(fso):
                        files = self.copy_tree(fso, target_dir)
                        self.outfiles.extend(files)
                    elif os.path.isfile(fso):
                        f, copied = self.copy_file(fso, target_dir)
                        self.outfiles.append(f)
                    else:
                        self.warn('Unable to find %s...' % fso)


class InstallMessages(InstallStuff):
    description = 'Installs message catalogs'

    user_options = install_misc.user_options[:]
    user_options.extend([
        ('build-dir=', 'b', 'build directory (where to install from)'),
        ('skip-build', None, "skip the build steps")
        ])

    boolean_options = ['skip-build']

    def initialize_options(self):
        self.install_dir = None
        self.build_dir = None
        self.skip_build = None
        install_misc.initialize_options(self)

    def finalize_options(self):
        self._install_dir_from('install_messages')
        self.set_undefined_options('build', ('build_messages', 'build_dir'))
        self.set_undefined_options('install', ('skip_build', 'skip_build'))

    def run(self):
        if not self.skip_build:
            self.run_command('build_messages')
        self.announce('Installing message catalogs...')
        po_files = glob(os.path.join(self.build_dir, '*.gmo'))
        for po_file in po_files:
            language = os.path.splitext(os.path.basename(po_file))[0]
            target_dir = os.path.join(self.install_dir, language,
                                      'LC_MESSAGES')
            self.outfiles.extend(self.mkpath(target_dir))
            target = os.path.join(target_dir,
                                  self.distribution.get_name() + '.mo')
            f, copied = self.copy_file(po_file, target)
            self.outfiles.append(f)


class InstallLinks(InstallStuff):
    description = ('Installs executable links')

    def finalize_options(self):
        self._install_dir_from('install_links')
        
    def run(self):
        sources = self.get_finalized_command('install_sources')

        target_directory = sources.install_dir

        if not os.path.exists(self.install_dir):
            self.outfiles.extend(self.mkpath(self.install_dir))

        for link in self.distribution.links:
            dest = os.path.join(self.install_dir, link[0])
            target = os.path.join(target_directory, link[1])
            # make sure, target is executable (link would be vain otherwise)
            mode = int('755', 8)
            self.announce('Changing mode of %s to %o' % (target, mode))
            os.chmod(target, mode)
            self.announce('linking %s to %s' % (dest, target))
            if not self.dry_run:
                if os.path.islink(dest):
                    os.remove(dest)
                os.symlink(target, dest)
                self.outfiles.append(dest)
                    

class InstallDesktop(InstallStuff):
    description = ('Installs desktop files')

    def finalize_options(self):
        self._install_dir_from('install_desktop')
        
    def run(self):
        if not os.path.exists(self.install_dir):
            self.outfiles.extend(self.mkpath(self.install_dir))

        for desktop in self.distribution.desktop:
            if os.path.isfile(desktop):
                name = os.path.basename(desktop)
                target = os.path.join(self.install_dir, name)
                f, copied = self.copy_file(desktop, self.install_dir)
                self.outfiles.append(f)
            else:
                self.warn('Unable to find %s...' % item)


class InstallIcons(InstallStuff):
    description = 'Install icons'

    def finalize_options(self):
        self._install_dir_from('install_icons')
        
    def run(self):
        if not os.path.exists(self.install_dir):
            self.outfiles.extend(self.mkpath(self.install_dir))

        for item in self.distribution.icons:
            theme = item[0]
            categories = item[1]
            for category in categories:
                items = categories[category]
                for item in items:
                    icon, geometry, name = item
                    if os.path.isfile(icon):
                        target_path = os.path.join(self.install_dir, theme,
                                                   geometry, category)
                        self.outfiles.extend(self.mkpath(target_path))
                        target = os.path.join(target_path, name)
                        f, copied = self.copy_file(icon, target)
                        self.outfiles.append(f)
                    else:
                        self.warn('Unable to find %s...' % item)


class InstallManPages(InstallStuff):
    description = 'Install manpages'

    user_options = InstallStuff.user_options[:]
    user_options.append(('build-dir=', None,
                         'build directory of manpages'))

    def initialize_options(self):
        InstallStuff.initialize_options(self)
        self.build_dir = None

    def finalize_options(self):
        self._install_dir_from('install_manpages')
        self.set_undefined_options('build', ('build_manpages', 'build_dir'))

    def run(self):
        for man in self.distribution.manpages:
            if isinstance(man, basestring):
                manpage = man
                locale = ''
            else:
                manpage = man[0]
                locale = man[1]
            category = 'man' + manpage[-1]
            target_dir = os.path.join(self.install_dir, locale, category)
            self.outfiles.extend(self.mkpath(target_dir))
            gzipped_manpage = os.path.join(self.build_dir, manpage + '.gz')
            f, copied = self.copy_file(gzipped_manpage, target_dir)
            self.outfiles.append(f)
            

class KDEInstall(install):
    user_options = install.user_options[:]
    user_options.extend(
        [('install-messages=', None,
          'Installation directory of message catalogs'),
         ('install-app-data=', None,
          'Installation directory for application data'),
         ('install-sources=', None,
          'Installation directory for python sources'),
         ('install-desktop=', None,
          'Installation directory for desktop files'),
         ('install-manpages=', None, 'Installation directory for manpages'),
         ('install-icons=', None, 'Installation directory for icons'),
         ('install-links=', None,
          'Installation directory for executable links'),
         ('kde-config-exe=', None, 'Path to the kde config executable')])

    sub_commands = install.sub_commands[:]
    sub_commands.append(('install_messages', has_messages))
    sub_commands.append(('install_sources', has_sources))
    sub_commands.append(('install_app_data', has_app_data))
    sub_commands.append(('install_links', has_links))
    sub_commands.append(('install_desktop', has_desktop))
    sub_commands.append(('install_icons', has_icons))
    sub_commands.append(('install_manpages', has_manpages))

    def initialize_options(self):
        self.install_messages = None
        self.install_app_data = None
        self.install_sources = None
        self.install_links = None
        self.install_desktop = None
        self.install_icons = None
        self.install_manpages = None
        self.kde_config_exe = None
        install.initialize_options(self)

    def finalize_options(self):
        install.finalize_options(self)

        if self.kde_config_exe is None:
            self.announce('Searching kde-config...')
            self.kde_config_exe = spawn.find_executable('kde-config')
            if self.kde_config_exe is None:
                raise SystemExit('Couldn\'t find "kde-config".')
            self.announce('  ...kde-config found at %s' %
                          self.kde_config_exe)
            
        if self.install_messages is None:
            self.install_messages = self.query_kde_directory('locale')
            
        if self.install_app_data is None:
            name = self.distribution.get_name()
            self.install_app_data = self.query_kde_directory('data')
            self.install_app_data = os.path.join(self.install_app_data,
                                                 name)
            
        if self.install_sources is None:
            self.install_sources = os.path.join(self.install_app_data, 'py')
            
        if self.install_links is None:
            self.install_links = self.query_kde_directory('exe')
            
        if self.install_desktop is None:
            self.install_desktop = self.query_kde_directory('xdgdata-apps')

        if self.install_icons is None:
            self.install_icons = self.query_kde_directory('icon')

        if self.install_manpages is None:
            # FIXME: weak guess, not sure if it works everywhere
            self.install_manpages = os.path.join(self.install_data,
                                                 'share', 'man')

        if self.root is not None:
            self.change_roots('app_data', 'sources', 'links', 'messages',
                              'desktop', 'icons')

        self.min_qt_version = self.distribution.min_qt_version
        self.min_kde_version = self.distribution.min_kde_version 

    def query_kde_directory(self, resource):
        """Returns the kde directory for `resource`"""
        if self.kde_config_exe:
            cmd = [self.kde_config_exe, '--install', resource,
                   '--expandvars']
            process = subprocess.Popen(cmd, stdout=subprocess.PIPE)
            return process.communicate()[0].strip('\n')
        else:
            return None

    def get_kde_qt_versions(self):
        cmd = [self.kde_config_exe, '--version']
        process = subprocess.Popen(cmd, stdout=subprocess.PIPE)
        output = process.communicate()[0]
        for line in output.splitlines():
            if line.startswith('Qt:'):
                qt_version = LooseVersion(line.split(':')[1].strip())
            elif line.startswith('KDE:'):
                kde_version = LooseVersion(line.split(':')[1].strip())
        return qt_version, kde_version


    def verify_kde(self):
        # FIXME: move into a subcommands
        # verify qt
        qt_version, kde_version = self.get_kde_qt_versions()
        if (self.min_qt_version is not None and
            qt_version < self.min_qt_version):
            msg = 'QT Version %s or higher is required, found %s.' 
            raise SystemExit(msg  % (self.min_qt_version, qt_version))
        self.announce('Found Qt version %s.' % qt_version)
        # verify kde
        if (self.min_kde_version is not None and
            kde_version < self.min_kde_version):
            msg = 'KDE Version %s or higher is required, found %s.'
            raise SystemExit(msg % (self.min_kde_version, kde_version))
        self.announce('Found KDE version %s.' % kde_version)

        # verify pyqt
        self.announce('Checking for a working PyQt...')
        try:
            import qt
            self.announce('  ...PyQt is working')
        except:
            raise SystemExit, '  ...PyQT not found.'

        # verify pykde
        self.announce('Checking for a working PyKDE...')
        for k_module in ('dcop', 'kdecore', 'kdeui', 'kio', 'kfile',
                         'kparts', 'khtml', 'kspell'):
            self.announce(k_module)
            try:
                exec('import ' + k_module)
            except:
                raise SystemExit('  ...PyKDE not found')
        self.announce("  ...PyKDE is working")
        

    def run(self):
        self.verify_kde()
        
        install.run(self)
        stream = open(INSTALL_LOG, 'w')
        outputs = self.get_outputs()
        stream.write('\n'.join(outputs))
        stream.write('\n')
        stream.close()


class KDEUninstall(Command):
    description = 'Whipes the program from this computer'

    user_options = []

    def initialize_options(self):
        pass

    def finalize_options(self):
        pass

    def run(self):
        if not os.path.isfile(INSTALL_LOG):
            msg = 'Cannot find the list file "%s".' % INSTALL_LOG
            raise SystemExit(msg)

        stream = open(INSTALL_LOG)
        files = stream.readlines()
        stream.close()

        # sort and reverse the file list. This puts the directories after
        # the files
        files.sort()
        files.reverse()
        
        for fso in files:
            fso = fso.strip()
            self.announce('Removing %s...')
            try:
                if not self.dry_run:
                    if os.path.isfile(fso) or os.path.islink(fso):
                        os.remove(fso)
                    elif os.path.isdir(fso):
                        os.rmdir(fso)
            except OSError, e:
                self.warn('Could not remove %s: %s' % (fso, e))


class KDEClean(clean):
    user_options = clean.user_options[:]
    user_options.extend([
        ('build-messages=', None, 'build directory for messages'),
        ('build-ui=', None, 'build directory for ui files'),
        ('build-manpages=', None, 'build directory for manpages')])

    def initialize_options(self):
        self.build_messages = None
        self.build_ui = None
        self.build_manpages = None
        clean.initialize_options(self)

    def finalize_options(self):
        self.set_undefined_options('build',
                                   ('build_messages', 'build_messages'),
                                   ('build_ui', 'build_ui'),
                                   ('build_manpages', 'build_manpages'))
        clean.finalize_options(self)

    def run(self):
        if self.all:
            for directory in (self.build_ui, self.build_messages,
                              self.build_manpages):
                if os.path.exists(directory):
                    dir_util.remove_tree(directory)
                else:
                    self.warn("'%s' does not exist -- can't clean it" %
                              directory)
        clean.run(self)
