#!/usr/bin/env python
# -*- coding: utf-8 -*-
# KeyJnoteGUI: A frontend to KeyJnote, an eye-candy presentation programm
# Copyright (C) 2006  Sebastian Wiesner <basti.wiesner@gmx.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""A frontend to KeyJnote"""

__revision__ = '$Id: keyjnotegui.py 119 2007-04-12 10:01:30Z lunar $'
__version__ = "0.4.3.1"

import sys
import signal
signal.signal(signal.SIGINT, lambda sig, frame: sys.exit('Interrupted'))
import gettext

from qt import QString, QPopupMenu, QObject, SIGNAL
from kdecore import (KApplication, KCmdLineArgs, KAboutData, KShortcut,
                     i18n)
from kdeui import (KMainWindow, KAction, KMessageBox, KStdAction,
                   KKeyDialog)

# modules of KeyJnoteGUI
import kjnwrapper
from kjnguiwidgets import SettingsDialog, PresentationUI
from settings import Settings


_ = gettext.gettext


class KeyJnoteGUI(KMainWindow):
    """The main window of KeyJnoteGUI."""
    
    def __init__(self, settings, wrapper):
        """Creates a new instance. `settings` is a Settings instance
        containting the configuration of this widget. `wrapper` is used to
        control KeyJnote.

        :ivar wrapper: The wrapper used to access KeyJnote
        :ivar settings: Settings instance"""
        KMainWindow.__init__(self)
        self._presentation_widget = None
        self.settings = settings
        self.wrapper = wrapper
        self.create_gui()
        self.load_settings()

    def create_gui(self):
        """Creates the graphical user interface"""
        # create the status bar
        self.statusBar()
        # create menu bar
        presentation = QPopupMenu(self.menuBar())
        self.execute = KAction(i18n('Run'), 'run', KShortcut('Ctrl+R'),
                               self.run_KeyJnote, self.actionCollection(),
                               'run')
        self.execute.plug(presentation)
        self.execute.plug(self.toolBar())
        # TODO: add actions to import and export KeyJnote description files
        quitact = KStdAction.quit(self.close, self.actionCollection())
        quitact.plug(presentation)
        quitact.plug(self.toolBar())
        self.menuBar().insertItem(i18n('Presentation'), presentation)
        settings = QPopupMenu(self.menuBar())
        self.menuBar().insertItem(i18n('Settings'), settings)
        preferences = KStdAction.preferences(self.configure_preferences,
                                             self.actionCollection())
        preferences.plug(settings)
        keybindings = KStdAction.keyBindings(self.configure_keybindings,
                                             self.actionCollection())
        keybindings.plug(settings)
        helpmenu = self.helpMenu(QString.null, False)
        self.menuBar().insertItem(i18n('Help'), helpmenu)
        # central widget
        self._presentation_widget = PresentationUI(self)
        self.setCentralWidget(self._presentation_widget)
        QObject.connect(self._presentation_widget.transitions,
                        SIGNAL('removed(QListBoxItem *)'),
                        self.check_transitions)
        QObject.connect(self._presentation_widget.transitions,
                        SIGNAL('added(QListBoxItem *)'),
                        self.check_transitions)


    def check_transitions(self):
        """Checks, if transitions are selected... Disables execute action,
        if there are no transitions"""
        listbox = self._presentation_widget.transitions.selectedListBox()
        if listbox.count() == 0:
            self.execute.setEnabled(False)
        else:
            self.execute.setEnabled(True)

    def configure_preferences(self):
        """Shows a preferences dialog"""
        SettingsDialog(self, self.settings).exec_loop()
        # create a new wrapper to reflect the changes
        try:
            self.wrapper = kjnwrapper.create_wrapper(self.settings)
            self.execute.setEnabled(True)
        except kjnwrapper.KJnException, e:
            self.execute.setEnabled(False)
            msg = i18n('The configuration is not correct:\n%1'
                    '\nExecuting KeyJnote is disabled until you correct '
                    'your configuration.').arg(unicode(e))
            KMessageBox.sorry(self, msg)

    def configure_keybindings(self):
        """Shows a dialog to configure keybindings"""
        KKeyDialog.configure(self.actionCollection(), self)
               
    def load_settings(self):
        """Applies settings to gui. This method is invoked by __init__."""
        self._presentation_widget.load(self.settings)
        if self.settings.window_geometry:
            self.setGeometry(self.settings.window_geometry)
        if not self.settings.selected:
            self.execute.setEnabled(False)

    def save_settings(self):
        """Save settings. This method is invoked when the window closes."""
        self._presentation_widget.save(self.settings)
        self.settings.window_geometry = self.geometry()
        self.settings.save()

    def queryClose(self):
        """Invoked, when the window closes. Saves settings."""
        try:
            self.save_settings()
        except Exception, e:
            # silently dump all raised exceptions to stderr.  This makes
            # sure, that there is no hanging window left behind, even if
            # things go wrong here
            print >> sys.stderr, 'Exception occured:', e
        return True
    
    def run_KeyJnote(self):
        """runs the main event loop of keyjnote"""
        self.hide()
        try:
            try:
                self.save_settings()
                self.wrapper.execute()
            finally:
                self.show()
        except kjnwrapper.KJnException, e:
            KMessageBox.sorry(self, unicode(e))


def main():
    data = KAboutData('keyjnotegui', _('KeyJnoteGUI'), __version__,
                      _('A nice GUI for the presentation programm '
                        'KeyJnote'), KAboutData.License_GPL,
                      '(c) 2006, Sebastian Wiesner', None,
                      'http://keyjnotegui.berlios.de/',
                      'basti.wiesner@gmx.net')
    data.addAuthor('Sebastian Wiesner', _('Everything'),
                   'basti.wiesner@gmx.net')
    data.addCredit('Martin Fiedler', _('For KeyJnote'),
                   'martin.fiedler@gmx.net')
    data.addCredit('Yaroslav Halchenko', _('For supporting KeyJnoteGUI and '
                                           'creating the debian package'),
                   'debian@onerussian.com')
    data.addCredit('Miriam Wiesner', _('For suggestion and '
                                       '(unvoluntary) testing'))

    KCmdLineArgs.init(sys.argv, data)
    app = KApplication()
    settings = Settings()
    settings.load()
    wrapper = None
    while not wrapper:
        try:
            wrapper = kjnwrapper.create_wrapper(settings)
            if not settings.selected:
                settings.selected = wrapper.transitions
        except kjnwrapper.KJnException, e:
            msg = i18n('%1\nPlease correct your settings!').arg(unicode(e))
            resp = KMessageBox.warningContinueCancel(None, msg)
            if resp == KMessageBox.Cancel:
                msg = i18n('KeyJnoteGUI cannot be executed without an '
                           'KeyJnote executable!')
                KMessageBox.sorry(None, msg)
                return
            else:
                SettingsDialog(None, settings).exec_loop()
    
    gui = KeyJnoteGUI(settings, wrapper)
    app.setMainWidget(gui)
    gui.show()       
    app.exec_loop()

if __name__ == '__main__':
    main()
