/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_localaccount.mod localaccount plugin

    \section localaccountsection1 SYNOPSIS
    \b lcmaps_localaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]

    \section localaccountsection2 DESCRIPTION
 
    This plugin is an Acquisition Plugin and will provide the LCMAPS system with Local Account
    credential information.
    To do this it needs to look up the Distinghuished Name (DN) from a user's certificate in the gridmapfile.
    If this DN is found in the gridmapfile the plugin knows the mapped local (system) account username.
    By knowing the username of the local account the plugin can gather additional information about this account.
    The plugin will resolve the UID, GID and all the secondary GIDs.
    When this has been done and there weren't any problems detected,
    the plugin will add this information to a datastructure in the Plugin Manager.
    The plugin will finish its run with a LCMAPS_MOD_SUCCESS.
    This result will be reported to the Plugin Manager which started this plugin and
    it will forward this result to the Evaluation Manager, which will take appropriate actions
    for the next plugin to run.
    Normally this plugin would be followed by an Enforcement plugin that can apply these gathered
    credentials in a way that is appropriate to a system administration's needs.
                                      

    \section localaccountoptions OPTIONS
    \subsection localaccountoptie1 -GRIDMAPFILE <gridmapfile>
        See \ref localaccountoptie4 "-gridmap"

    \subsection localaccountoptie2 -gridmapfile <gridmapfile>
        See \ref localaccountoptie4 "-gridmap"

    \subsection localaccountoptie3 -GRIDMAP <gridmapfile>
        See \ref localaccountoptie4 "-gridmap"

    \subsection localaccountoptie4 -gridmap <gridmapfile>
        When this option is set it will override the default path of the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

\section localaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure


\section localaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/) 

\section localaccountSeeAlso SEE ALSO
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_localaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code for localaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>

#include "lcmaps_plugins_basic_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *gridmapfile = NULL;


/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char * logstr = "lcmaps_plugin_localaccount-plugin_initialize()";
    int i;

    lcmaps_log_debug(5,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (gridmap(file))
     * the second will be the path && filename of the gridmapfile
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-gridmap") == 0) ||
              (strcmp(argv[i], "-GRIDMAP") == 0) ||
              (strcmp(argv[i], "-gridmapfile") == 0) ||
              (strcmp(argv[i], "-GRIDMAPFILE") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 gridmapfile = strdup(argv[i + 1]);
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    return LCMAPS_MOD_SUCCESS;
} 

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char * logstr = "lcmaps_plugin_localaccount-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"      ,       "char *"        , 1,   NULL},
        {NULL           ,       NULL            , -1,   NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *              logstr = "lcmaps_plugin_localaccount-plugin_run()";
    char *              dn          = NULL; 
    char *              username    = NULL;
    struct passwd       *user_info  = NULL;
    int                 i           = 0;
    int                 cnt_sec_gid = 0;
    gid_t *             sec_gid     = NULL;
    int                 rc          = 0;
    int                 tmp_cnt     = 0;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_localaccount-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_localaccount-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_localaccount-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_localaccount;
    }

    /*
     * Try to get the ordered values:
     */
    if ( ( dn = *(char **) lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )
    {
        lcmaps_log_debug(1,"%s: found dn: %s\n", logstr, dn);

        /* Do I already have a DN registered in LCMAPS? If not registered, push it to the end-result registry */
        getCredentialData (DN, &tmp_cnt);
        if (tmp_cnt == 0)
        {
            addCredentialData(DN, &dn);
        }
    }
    else
        lcmaps_log(LOG_ERR,"%s: could not get value of dn !\n", logstr);


    /*
     * Check the gridmapfile
     */

    if ((gridmapfile != NULL) && (strlen(gridmapfile) > 0))
        lcmaps_log_debug(1,"%s: gridmapfile is: %s\n", logstr, gridmapfile);
    else
    {
        if (gridmapfile) free(gridmapfile);
        gridmapfile = NULL;
        lcmaps_log_debug(1,"%s: No gridmapfile assigned, so function must find out for it self\n", logstr);
    }

    /*
     * Try to find the dn in the gridmapfile
     */


    if ( (rc = lcmaps_gridlist(dn, &username, gridmapfile, MATCH_EXCLUDE|MATCH_NO_WILD_CHARS, ".", NULL)) == LCMAPS_MOD_SUCCESS)
        lcmaps_log_debug(1,"%s: found username: %s\n", logstr, username);
    else if (rc == LCMAPS_MOD_NOFILE)
    {
        lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
        goto fail_localaccount;
    }
    else if (rc == LCMAPS_MOD_NOENTRY)
    {
        lcmaps_log(LOG_NOTICE, "%s: No entry found for %s in %s\n", logstr, dn, gridmapfile);
        goto fail_localaccount;
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: could not get value of username !\n", logstr);
        goto fail_localaccount;
    }

    /*
     * Get userid to pwd_t structure
     */


    if (username && (strlen(username) > 0))
    {

        if ( ( user_info = getpwnam(username) ) )
        {
            lcmaps_log_debug(2,"%s: username : %s\n", logstr, user_info->pw_name);
            lcmaps_log_debug(2,"%s: user_id  : %d\n", logstr, user_info->pw_uid);
            lcmaps_log_debug(2,"%s: group_id : %d\n", logstr, user_info->pw_gid);
            lcmaps_log_debug(2,"%s: home dir : %s\n", logstr, user_info->pw_dir);

            /* 
             * Add this credential data to the credential data repository in the plugin manager
             */
            addCredentialData(UID, &(user_info->pw_uid));
            addCredentialData(PRI_GID, &(user_info->pw_gid));

            /*
             * Retrieve secondary group id's
             */
            if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
            {
                for (i = 0; i < cnt_sec_gid; i++)
                {
                    addCredentialData(SEC_GID, &(sec_gid[i]));
                }
                free(sec_gid);
            }
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: no user account found name \"%s\"\n", logstr,username);
            goto fail_localaccount;
        }
    }
    else
    {   // error (msg is already given)
        goto fail_localaccount;
    }

    /* succes */
 success_localaccount:
    if (username) free(username);
    lcmaps_log(LOG_INFO,"%s: localaccount plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_localaccount:
    if (username) free(username);
    lcmaps_log(LOG_INFO,"%s: localaccount plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char *              logstr = "lcmaps_plugin_localaccount-plugin_terminate()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (gridmapfile) free(gridmapfile);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-basic/src/localaccount/lcmaps_localaccount.c,v $
    $Date: 2010-02-19 06:08:47 $
    $Revision: 1.10 $
    $Author: okoeroo $
******************************************************************************/
