/* This file is part of libccc
 *
 * AUTHORS
 *     Mirco "MacSlow" Müller <macslow@bangang.de>
 *     Sven Herzberg <herzi@gnome-de.org>
 *
 * Copyright (C) 2007 Mirco "MacSlow" Müller
 * Copyright (C) 2007 Sven Herzberg
 *
 * This work is provided "as is"; redistribution and modification
 * in whole or in part, in any medium, physical or electronic is
 * permitted without restriction.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * In no event shall the authors or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 */

#include "main.h"

#include <math.h>
#include <ccc.h>
#include <cc-config.h>
#include <glib/gi18n-lib.h>

#define TOP      0
#define LEFT     0
#define WIDTH  200
#define HEIGHT 200

static void
paint (CcSimpleItem* pItem,
       CcView*	     pView,
       cairo_t*	     pContext)
{
	unsigned long ulMilliSeconds;
	double fOffsetX = 0.0f;
	double fOffsetY = 0.0f;
	double fLength = 1.0f / 25.0f;
	double fY;
	gint i;
	static GTimer* pTimer = NULL;

	if (!pTimer)
	{
		pTimer = g_timer_new ();
		g_signal_connect_swapped (GTK_WIDGET (pView),
					  "destroy",
					  G_CALLBACK (g_timer_destroy),
					  (gpointer) pTimer);
	}

	cc_view_world_to_window (pView, &fOffsetX, &fOffsetY);
	cairo_translate (pContext, fOffsetX, fOffsetY);

	g_timer_elapsed (pTimer, &ulMilliSeconds);
	ulMilliSeconds /= 1000;

	cairo_save (pContext);
	cairo_scale (pContext, WIDTH, HEIGHT);
	cairo_set_line_cap (pContext, CAIRO_LINE_CAP_ROUND);
	cairo_set_operator (pContext, CAIRO_OPERATOR_OVER);
	cairo_set_line_width (pContext, fLength);
	for (i = 0; i < 60; i++)
	{
		cairo_save (pContext);
		cairo_translate (pContext, 0.5f, 0.5f);
		cairo_rotate (pContext,
			      G_PI / 180.0f *
			      (ulMilliSeconds + 10.0f * i) * 0.36f);
		fY = 0.33f + 0.0825f *
		     sinf ((ulMilliSeconds + 10.0f * i) / 1000 * 10 * G_PI);
		cairo_translate (pContext, 0.0f, fY);
		cairo_rotate (pContext, G_PI / 180.0f * 6.0f * i);
		cairo_set_source_rgba (pContext, 0.25f, 0.5f, 1.0f, i * 0.01f);
		cairo_move_to (pContext, -fLength, 0.0f);
		cairo_line_to (pContext, fLength, 0.0f);
		cairo_stroke (pContext);
		cairo_restore (pContext);
	}
	cairo_restore (pContext);

	cc_item_dirty (pItem, pView, *cc_item_get_all_bounds (pItem, pView));
}

static void
recalc_bbox (CcSimpleItem* pItem,
	     CcView const* pView,
	     gpointer	   data)
{	
	CcDRect drect = {TOP, LEFT, TOP + WIDTH, LEFT + HEIGHT};
	cc_hash_map_insert (CC_ITEM (pItem)->bounds,
			    (CcView*)pView,
			    cc_d_rect_copy (&drect));
	cc_item_bounds_changed (pItem, pView);
}

DemoPage*
flowers (void)
{
	CcSimpleItem* pFlower = cc_simple_item_new ();

	g_signal_connect (pFlower,
			  "render",
			  G_CALLBACK (paint),
			  NULL);

	g_signal_connect (pFlower,
			  "update-bounds",
			  G_CALLBACK (recalc_bbox),
			  NULL);

	return demo_page_new (cc_view_widget_new_root (pFlower), _("Simple Item"));
}

