/* This file is part of libccc
 *
 * AUTHORS
 *     Sven Herzberg  <herzi@gnome-de.org>
 *
 * Copyright (C) 2007  Sven Herzberg
 *
 * This work is provided "as is"; redistribution and modification
 * in whole or in part, in any medium, physical or electronic is
 * permitted without restriction.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * In no event shall the authors or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 */

#include "s-text.h"

#include <ccc/cc-text.h>
#include "cc-test-view.h"

static CcView* view = NULL;
static CcItem* item = NULL;

static void
setup (void)
{
	view = cc_test_view_new ();
	item = g_object_ref_sink (cc_text_new (""));
	cc_view_set_root (view, item);
}

static void
teardown (void)
{
	g_object_unref (view);
	view = NULL;
	g_object_unref (item);
	item = NULL;
}

static void
test_dirty_emission_cb (CcItem       * item,
			CcView       * view,
			CcDRect const* dirty,
			gboolean     * passed)
{
	*passed = TRUE;
}

START_TEST(test_dirty_emission)
{
	/* prepare */
	gboolean  passed;
	gchar   * text = g_strdup_printf ("%d", _i);

	g_signal_connect (item, "dirty",
			  G_CALLBACK (test_dirty_emission_cb), &passed);

	/* check */
	passed = FALSE;
	cc_text_set_text (CC_TEXT (item), text);
	fail_unless (passed);

	/* cleanup */
	g_signal_handlers_disconnect_by_func (item, test_dirty_emission_cb, &passed);
	g_free (text);
}
END_TEST

static void
test_dirty_bounds_cb (CcItem       * item,
		      CcView       * view,
		      CcDRect const* dirty,
		      gboolean     * passed)
{
	CcDRect const* bbox = cc_item_get_all_bounds (item, view);

	if (!CC_IS_VIEW (view)) {
		// FIXME: get rid of this check
		return;
	}

#if 1
	// FIXME: use the other check
	*passed = bbox != NULL;
#else
	*passed &= bbox != NULL;
#endif
}

START_TEST(test_dirty_bounds)
{
	/* prepare */
	gboolean  passed;
	gchar   * text = g_strdup_printf ("%d", _i);

	g_signal_connect (item, "dirty",
			  G_CALLBACK (test_dirty_bounds_cb), &passed);

	/* check */
	passed = FALSE;
	cc_text_set_text (CC_TEXT (item), text);
	fail_unless (passed);

	/* cleanup */
	g_signal_handlers_disconnect_by_func (item, test_dirty_bounds_cb, &passed);
	g_free (text);
}
END_TEST

static void
test_dirty_region_cb (CcItem       * item,
		      CcView       * view,
		      CcDRect const* dirty,
		      gboolean     * passed)
{
	CcDRect const* bbox = cc_item_get_all_bounds (item, view);

	g_print ("test_dirty_region_cb(%s, %s)\n",
		 G_OBJECT_TYPE_NAME (item),
		 G_OBJECT_TYPE_NAME (view));

	if (!CC_IS_VIEW (view)) {
		// FIXME: get rid of this check
		return;
	}

	if (!bbox) {
		// FIXME: get rid of this check
		return;
	}

	if ((bbox->x1 > dirty->x1) ||
	    (bbox->y1 > dirty->y1) ||
	    (bbox->x2 < dirty->x2) ||
	    (bbox->y2 < dirty->y2))
	{
		g_print ("3:bbox:%p\n", bbox),
		*passed = FALSE;
	}
}

START_TEST(test_dirty_region)
{
	/* prepare */
	gboolean  passed;
	gchar   * text = g_strdup_printf ("%d", _i);

	g_signal_connect (item, "dirty",
			  G_CALLBACK (test_dirty_region_cb), &passed);

	/* check */
	passed = TRUE;
	cc_text_set_text (CC_TEXT (item), text);
	fail_unless (passed);

	/* cleanup */
	g_signal_handlers_disconnect_by_func (item, test_dirty_region_cb, &passed);
	g_free (text);
}
END_TEST

TCase*
tcase_text_dirty (void)
{
	TCase* self = tcase_create ("dirty region management");
	tcase_add_checked_fixture (self, setup, teardown);
	tcase_add_loop_test (self, test_dirty_emission, 0, 11);
	tcase_add_loop_test (self, test_dirty_bounds,   0, 11);
	tcase_add_loop_test (self, test_dirty_region,   0, 11);
	return self;
}

