require 'set'

class CKQualifierParser
	class ParseError < StandardError; end #:nodoc:

	attr_reader :qualifier

	def initialize( format )
		tokens     = _analyze format.dup
		@qualifier = _parse tokens
	end

	private

	def _analyze( format )
		format    = "(#{format})"
		scanner   = StringScanner.new format
		qualifier = nil
		tokens    = []
		op_reg    = /\A(==|!=|>=|<=|>|<|=~)/im

		until scanner.eos? do
			scanner.skip /\A[\s]+/

			if str = scanner.scan(/\A(\(|\))/) then
				tokens << str
			elsif str = scanner.scan(op_reg) then
				tokens << CKQualifier.operator_symbol(str)
			elsif str = scanner.scan(/\A\d+\.\d+/) then
				tokens << str.to_f
			elsif str = scanner.scan(/\A\d+/) then
				tokens << str.to_i
			elsif scanner.match?(/\Atrue\W/) then
				scanner.scan /\Atrue/
				tokens << true
			elsif scanner.match?(/\Afalse\W/) then
				scanner.scan /\Afalse/
				tokens << false
			elsif scanner.match?(/\Anil\W/) then
				scanner.scan /\Anil/
				tokens << nil
			elsif scanner.scan(/\A\//) then
				tokens << _parse_regexp(scanner)
			elsif str = scanner.scan(/\A'(([^'\\]|\\.)*)'/) then
				tokens << scanner[0]
			elsif str = scanner.scan(/\A"(([^"\\]|\\.)*)"/) then
				tokens << scanner[0]
			else
				str = scanner.scan /\A[^\s\(\)]+/
				tokens << str
			end
		end

		tokens
	end

	def _parse_regexp( scanner )
		regstr = ''
		char = nil
		pre_char = nil
		option = nil

		until scanner.eos? do
			char = scanner.getch

			if (pre_char != '\\') and (char == '/') then
				option = scanner.scan(/\A\w+/)
				break
			end

			regstr << char
			pre_char = char
		end

		Regexp.new(regstr, option)
	end

	def _parse( tokens )
		op_stack  = []
		out_stack = []
		op = left = right = q = nil

		reg_and = /\Aand\Z/mi
		reg_or  = /\Aor\Z/mi
		reg_not = /\Anot\Z/mi

		tokens.each do |token|
			case token
			when '('
				op_stack << token
			when ')'
				until op_stack.last == '(' do
					op    = op_stack.pop
					right = out_stack.pop
					left  = out_stack.pop

					case op
					when Symbol
						if Regexp === right then
							q = CKKeyValueQualifier.new(left, op, right)
						elsif right =~ /\A'(([^'\\]|\\.)*)'/ then
							q = CKKeyValueQualifier.new(left, op, $1)
						elsif right =~ /\A"(([^"\\]|\\.)*)"/ then
							q = CKKeyValueQualifier.new(left, op, $1)
						elsif (Numeric === right) or (right == true) or \
							(right == false) or right.nil? then
							q = CKKeyValueQualifier.new(left, op, right)
						else
							q = CKKeyComparisonQualifier.new(left, op, right)
						end
					when reg_and
						if CKAndQualifier === right then
							right.qualifiers.unshift left
							q = right
						else
							q = CKAndQualifier.new [left, right]
						end
					when reg_or
						if CKOrQualifier === right then
							right.qualifiers.unshift left
							q = right
						else
							q = CKOrQualifier.new [left, right]
						end
					when reg_not
						q = CKNotQualifier.new right
					end
					out_stack << q
				end
				op_stack.pop		
			when reg_and
				op_stack << token
			when reg_or
				op_stack << token
			when reg_not
				op_stack << token
			when Symbol
				op_stack << token
			else
				out_stack << token
			end
		end

		result = out_stack.pop
		unless out_stack.empty? and op_stack.empty? then
			raise ParseError, 'parse error'
		end

		result
	end
end


class CKQualifier
	module ComparisonSupport
		def coerce( left, right )
			coerced = left

			case right
			when String
				coerced = left.to_s
			when Integer
				coerced = left.to_i
			when Float
				coerced = left.to_f
			when NilClass
				if left == '' then
					coerced = nil
				end
			when Regexp
				coerced = left.to_s
			end

			coerced
		end

		def compare( left, right, symbol )
			coerced = coerce(left, right)
			__send__(symbol, coerced, right)
		end

		def equal?( left, right )
			left == right
		end

		def not_equal?( left, right )
			left != right
		end

		def greater?( left, right )
			left > right
		end

		def greater_or_equal?( left, right )
			left >= right
		end

		def less?( left, right )
			left < right
		end

		def less_or_equal?( left, right )
			left <= right
		end

		def match?( left, right )
			if left =~ right then
				true
			else
				false
			end
		end
	end

	class UnknownKeyError < StandardError; end #:nodoc:

	extend ComparisonSupport

	EQUAL            = :'equal?'
	NOT_EQUAL        = :'not_equal?'
	GREATER          = :'greater?'
	GREATER_OR_EQUAL = :'greater_or_equal?'
	LESS             = :'less?'
	LESS_OR_EQUAL    = :'less_or_equal?'
	MATCH            = :'match?'

	class << self
		OPERATORS   = ['==', '!=', '<', '<=', '>', '>=', '=~']

		def new_with_format( format )
			parser = CKQualifierParser.new(format)
			parser.qualifier
		end

		alias format new_with_format

		def operator_symbol( string )
			case string.upcase
			when '==' then EQUAL
			when '!=' then NOT_EQUAL
			when '>'  then GREATER
			when '>=' then GREATER_OR_EQUAL
			when '<'  then LESS
			when '<=' then LESS_OR_EQUAL
			when '=~' then MATCH
			end
		end

		def operator_string( operator )
			case operator
			when EQUAL            then '=='
			when NOT_EQUAL        then '!='
			when GREATER          then '>'
			when GREATER_OR_EQUAL then '>='
			when LESS             then '<'
			when LESS_OR_EQUAL    then '<='
			when MATCH            then '=~'
			end
		end

		def operators
			OPERATORS
		end

	end

	def qualifier_keys
		set = Set.new
		add_qualifier_keys set
		set
	end

	# abstract - subclasses must override it
	def add_qualifier_keys( set ); end

	alias inspect to_s
end


class CKKeyValueQualifier < CKQualifier
	attr_reader :key, :value, :symbol

	def initialize( key, symbol, value )
		super()
		@key = key
		@symbol = symbol
		@value = value
	end

	def add_qualifier_keys( set )
		set << @key
	end

	def ==( other )
		bool = false
		if CKKeyValueQualifier === other then
			if (@key == other.key) and (@symbol == other.symbol) and \
				(@value == other.value) then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		CKQualifier.compare(object[@key], @value, @symbol)
	end

	public

	def to_s
		op = CKQualifier.operator_string @symbol
		if String === @value then
			value_s = "'#@value'"
		else
			value_s = @value
		end

		"(#@key #{op} #{value_s})"
	end
end


class CKKeyComparisonQualifier < CKQualifier
	attr_reader :left, :symbol, :right

	def initialize( left, symbol, right )
		super()
		@left   = left
		@symbol = symbol
		@right  = right
	end

	def add_qualifier_keys( set )
		set << @left
	end

	def ==( other )
		bool = false
		if CKKeyComparisonQualifier === other then
			if (@left == other.left) and (@symbol == other.symbol) and \
				(@right == other.right) then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		CKQualifier.compare(object[@left], object[@right], @symbol)
	end

	def to_s
		op = CKQualifier.operator_string @symbol
		"(#@left #{op} #@right)"
	end
end


class CKAndQualifier < CKQualifier
	attr_reader :qualifiers

	def initialize( qualifiers )
		super()
		@qualifiers = qualifiers
	end

	def each
		qualifiers.each do |qualifier|
			yield qualifier
		end
	end

	def add_qualifier_keys( set )
		@qualifiers.each do |qualifier|
			qualifier.add_qualifier_keys set
		end
	end

	def ==( other )
		bool = false
		if CKAndQualifier === other then
			if @qualifiers == other.qualifiers then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		@qualifiers.each do |qualifier|
			unless qualifier.eval? object then
				return false
			end
		end
		true
	end

	def to_s
		str = '('
		@qualifiers.each do |q|
			str << q.to_s
			unless @qualifiers.last == q then
				str << " AND "
			end
		end
		str << ')'
		str
	end
end


class CKOrQualifier < CKQualifier
	attr_reader :qualifiers

	def initialize( qualifiers )
		super()
		@qualifiers = qualifiers
	end

	def each
		qualifiers.each do |qualifier|
			yield qualifier
		end
	end

	def add_qualifier_keys( set )
		@qualifiers.each do |qualifier|
			qualifier.add_qualifier_keys set
		end
	end

	def ==( other )
		bool = false
		if CKOrQualifier === other then
			if @qualifiers == other.qualifiers then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		@qualifiers.each do |qualifier|
			if qualifier.eval? object then
				return true
			end
		end
		false
	end

	def to_s
		str = '('
		@qualifiers.each do |q|
			str << q.to_s
			unless @qualifiers.last == q then
				str << " OR "
			end
		end
		str << ')'
		str
	end
end


class CKNotQualifier < CKQualifier
	attr_reader :qualifier

	def initialize( qualifier )
		super()
		@qualifier = qualifier
	end

	def add_qualifier_keys( set )
		qualifier.add_qualifier_keys set
	end

	def ==( other )
		bool = false
		if CKNotQualifier === other then
			if @qualifier == other.qualifier then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		unless @qualifier.eval? object then
			true
		else
			false
		end
	end

	def to_s
		"(NOT #{qualifier})"
	end
end

