package org.apache.commons.openpgp;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.io.InputStream;

/**
 * Interface for verifying data signed with OpenPGP.
 *
 * @author <a href="mailto:brett@apache.org">Brett Porter</a>
 * @todo perhaps should have different interface methods for the default key
 * @todo should the exception be a verification exception instead of a common one?
 */
public interface OpenPgpSignatureVerifier
{
    String ROLE = OpenPgpSignatureVerifier.class.getName();

    /**
     * Verify a piece of data that was signed with OpenPGP.
     *
     * @param data    the data that was signed
     * @param keyRing the keyring containing the key used to sign the data
     */
    SignatureStatus verifySignature( InputStream data, KeyRing keyRing )
        throws OpenPgpException, UnknownKeyException;

    /**
     * Verify a piece of data against a detached signature.
     *
     * @param data         the data to that was signed
     * @param signature    the detached signature to verify against the data
     * @param keyRing      the keyring containing the key used to sign the data
     * @param asciiArmored whether the signature is ascii armored
     */
    SignatureStatus verifyDetachedSignature( InputStream data, InputStream signature, KeyRing keyRing,
                                             boolean asciiArmored )
        throws OpenPgpException, UnknownKeyException, IOException;
}
