/*--------------------------------------------------------------------------+
$Id: UniqueIdManagerTest.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import junit.framework.TestCase;

/**
 * Test for <code>UniqueIdManager</code>-class.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * 
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: BA24B8313EB5F87F3DBA11DC98E250C9
 */
public class UniqueIdManagerTest extends TestCase {

    /**
     * Manager under test.
     * <p>
     * This intentionally doesn't use String objects since Java has a bit tricky
     * implementation regarding String literals, i.e. two equal String objects
     * created through literals have referential equality.
     * <p>
     * Example: <code>System.out.println("test1"=="test1");</code> yields
     * <code>true</code>
     * <p>
     * This is due to String intering. See {@link String#intern()} and <a
     * href="http://java.sun.com/docs/books/jls/third_edition/html/lexical.html#3.10.5">Java
     * Language Specification 3.10.5 String Literals</a> for an explanation.
     */
    private final UniqueIdManager<Integer> idManager = new UniqueIdManager<Integer>();

    /** A test object. */
    private final Integer test1 = new Integer(1);

    /** Another test object. */
    private final Integer test2 = new Integer(2);

    /**
     * Test method code <code>getId()</code>
     * 
     */
    public void testObtainId() {
        idManager.obtainId(test1);
        idManager.obtainId(test2);
        idManager.obtainId(new Integer(3));

        assertEquals(0, idManager.obtainId(test1));
        assertEquals(1, idManager.obtainId(test2));

        // id 2 is assigned to another Object
        assertEquals(3, idManager.obtainId(new Integer(4)));

    }

    /**
     * Test if the implementation really relies on referential equality.
     * 
     */
    public void testReferenceEqualitySemantics() {
        idManager.obtainId(new Integer(3));
        assertFalse(0 == idManager.obtainId(new Integer(3)));
    }

    /**
     * Test method <code>clear()</code>.
     * 
     */
    public void testClear() {
        idManager.obtainId(test1);
        assertEquals(0, idManager.obtainId(test1));
        assertEquals(1, idManager.obtainId(test2));

        idManager.clear();
        //changed order
        assertEquals(0, idManager.obtainId(test2));
        assertEquals(1, idManager.obtainId(test1));
    }
}