/*--------------------------------------------------------------------------+
$Id: HashedListMapTest.java 29486 2010-08-03 17:01:46Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import junit.framework.TestCase;

/**
 * JUnit test for <code>Options</code> class.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * 
 * @version $Rev: 29486 $
 * @levd.rating GREEN Hash: 4EDDF1E3F4A35FC8657E9D8234731A56
 */
public class HashedListMapTest extends TestCase {

	/** Map under test. */
	private HashedListMap<String, String> map;

	/** Create fixture. */
	@Override
	protected void setUp() {
		map = new HashedListMap<String, String>();
		map.add("test1", "test1_0");
		map.add("test1", "test1_1");
		map.add("test2", "test2_0");
		map.add("test2", "test2_1");
		map.add("test2", "test2_2");
	}

	/**
	 * Test method for {@link HashedListMap#createList(Object)}.
	 */
	public void testCreateList() {
		List<String> list = map.createList("test");
		assertEquals(0, list.size());

		// overwrite existing list
		list = map.createList("test1");
		assertEquals(0, list.size());
	}

	/**
	 * Test method for {@link HashedListMap#getList(Object)}.
	 */
	public void testGetList() {
		List<String> list = map.getList("test2");
		assertEquals("test2_0", list.get(0));
		assertEquals("test2_1", list.get(1));
		assertEquals("test2_2", list.get(2));
	}

	/**
	 * Test method for {@link HashedListMap#add(Object, Object)}.
	 */
	public void testAdd() {
		map.add("test1", "x");
		List<String> list = map.getList("test1");
		assertEquals("x", list.get(2));

		// previously non-existent list
		map.add("test3", "x");
		list = map.getList("test3");
		assertEquals("x", list.get(0));
	}

	/** Tests the {@link HashedListMap#addAll(HashedListMap)} method. */
	public void testAddAll() {
		HashedListMap<String, String> other = new HashedListMap<String, String>();
		other.add("test1", "x");
		other.add("test3", "x");

		map.addAll(other);

		assertEquals("x", map.getList("test1").get(2));
		assertEquals("x", map.getList("test3").get(0));
	}

	/**
	 * Test method for {@link HashedListMap#containsList(Object)}.
	 */
	public void testContainsList() {
		assertTrue(map.containsList("test1"));
		assertTrue(map.containsList("test2"));
		assertFalse(map.containsList("test3"));
	}

	/**
	 * Test method for {@link HashedListMap#removeList(Object)}.
	 */
	public void testRemoveList() {
		assertTrue(map.containsList("test1"));
		map.removeList("test1");
		assertFalse(map.containsList("test1"));
	}

	/**
	 * Test method for
	 * {@link HashedListMap#removeAllLists(java.util.Collection)}
	 */
	public void testRemoveAllLists() {
		assertTrue(map.containsList("test1"));
		assertTrue(map.containsList("test2"));
		map.removeAllLists(Arrays.asList(new String[] { "test1", "test2" }));
		assertFalse(map.containsList("test1"));
		assertFalse(map.containsList("test2"));
	}

	/**
	 * Test method for {@link HashedListMap#getKeys()}.
	 */
	public void testGetKeys() {
		Set<String> keys = map.getKeys();
		assertEquals(2, keys.size());
		assertTrue(keys.contains("test1"));
		assertTrue(keys.contains("test2"));
	}

	/**
	 * Test method for {@link HashedListMap#getValues()}.
	 */
	public void testGetValues() {
		List<String> values = map.getValues();
		assertEquals(5, values.size());
		assertTrue(values.contains("test1_0"));
		assertTrue(values.contains("test1_1"));
		assertTrue(values.contains("test2_0"));
		assertTrue(values.contains("test2_1"));
		assertTrue(values.contains("test2_2"));
	}

	/** Test for {@link HashedListMap#areAllListsEmpty()}. */
	public void testAreAllListsEmpty() {
		assertFalse(map.areAllListsEmpty());
		map.clear();
		assertTrue(map.areAllListsEmpty());
		setUp();

		assertFalse(map.areAllListsEmpty());
		for (String key : map.getKeys()) {
			map.getList(key).clear();
		}
		assertTrue(map.areAllListsEmpty());
	}

	/** Test for {@link HashedListMap#listsToArrays(Class)}. */
	public void testListToArrays() {
		Map<String, String[]> arrayMap = map.listsToArrays(String.class);
		assertTrue(arrayMap.keySet().size() == 2);
		assertTrue(arrayMap.get("test1").length == 2);
		assertTrue(arrayMap.get("test2").length == 3);
	}
}