/*--------------------------------------------------------------------------+
$Id: GraphvizGeneratorTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.graph;

import java.awt.HeadlessException;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;

import junit.framework.TestCase;
import edu.tum.cs.commons.filesystem.FileSystemUtils;
import edu.tum.cs.commons.string.StringUtils;

/**
 * JUnit test for <code>GraphvizGenerator</code> class.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * 
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: B522D1BE5C285DDA333DDA60E45BC2D1
 */
public class GraphvizGeneratorTest extends TestCase {

	/** Graph description with error. */
	private static final String ERR_TEST_DESCRIPTION = "digraph G { A -> B;   C -> /*; B -> E}";

	/** Test graph description. */
	private static final String TEST_DESCRIPTION = "digraph G { A -> B;   C -> B; B -> E}";

	/** Graph description with unknown font. */
	private static final String UNKNOWN_FONT_DESCRIPTION = "digraph G { node[fontname=\"unknownFont\"]; A -> B;   C -> B; B -> E}";

	/**
	 * This is for debugging purposes. It generates an image and displays it in
	 * a window.
	 */
	public static void main(String[] args) throws HeadlessException,
			IOException, GraphvizException {
		GraphvizGenerator generator = new GraphvizGenerator();
		BufferedImage image = generator.generateImage(TEST_DESCRIPTION);
		JFrame frame = new JFrame("GraphvizGeneratorTest");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		JLabel label = new JLabel(new ImageIcon(image));
		frame.getContentPane().add(label);

		frame.pack();
		frame.setVisible(true);
	}

	/** Standard generator. */
	private final GraphvizGenerator generator = new GraphvizGenerator();

	/** Test with a big description. */
	public void testGenerateFileWithBigDescription() throws IOException,
			GraphvizException {
		File file = new File("test-tmp/test.png");
		FileSystemUtils.ensureParentDirectoryExists(file);

		String description = createDescription(StringUtils.EMPTY_STRING);

		generator.generateFile(description, file, EGraphvizOutputFormat.PNG);
		assertTrue(file.exists());
		assertTrue(file.length() > 0);
		file.delete();
	}

	/** Test with a big description. */
	public void testGenerateImageWithBigDescription() throws IOException,
			GraphvizException {
		String description = createDescription(StringUtils.EMPTY_STRING);
		BufferedImage image = generator.generateImage(description);
		assertNotNull(image);
		assertTrue(image.getWidth() > 0);
		assertTrue(image.getHeight() > 0);
	}

	/** Test with a description that raises a lot of warnings. */
	public void testGenerateFileBigDescriptionWithWarnings()
			throws IOException, GraphvizException {
		File file = new File("test-tmp/test.png");
		FileSystemUtils.ensureParentDirectoryExists(file);

		String description = createDescription(":");

		generator.generateFile(description, file, EGraphvizOutputFormat.PNG);
		assertTrue(file.exists());
		assertTrue(file.length() > 0);
		file.delete();
	}

	/** Test with a description that raises a lot of warnings. */
	public void testGenerateImageWithBigDescriptionWithWarnings()
			throws IOException, GraphvizException {
		String description = createDescription(":");
		BufferedImage image = generator.generateImage(description);
		assertNotNull(image);
		assertTrue(image.getWidth() > 0);
		assertTrue(image.getHeight() > 0);
	}

	/**
	 * Test method <code>generateFile</code>. This checks if a file was
	 * created and if its size greater zero.
	 */
	public void testGenerateFile() throws IOException, GraphvizException {
		File file = new File("test-tmp/test1.png");
		FileSystemUtils.ensureParentDirectoryExists(file);
		generator.generateFile(TEST_DESCRIPTION, file,
				EGraphvizOutputFormat.PNG);
		assertTrue(file.exists());
		assertTrue(file.length() > 0);
		file.delete();
	}

	/**
	 * Test method <code>generateFile</code> with an erroneous graph
	 * description.
	 */
	public void testGenerateFileError() throws IOException {
		File file = new File("test-tmp/test1.png");
		FileSystemUtils.ensureParentDirectoryExists(file);
		try {
			generator.generateFile(ERR_TEST_DESCRIPTION, file,
					EGraphvizOutputFormat.PNG);
		} catch (GraphvizException e) {
			file.delete();
			return;
		}
		fail("Should create exception");
	}

	/**
	 * Test method <code>generateFile</code> with a graph description that
	 * specifies an unknown font. Graph should be generated even with unknown
	 * font. This documents bug #495.
	 * 
	 * @throws GraphvizException
	 */
	public void testGenerateFileUnknownFont() throws IOException,
			GraphvizException {
		File file = new File("test-tmp/test1.png");
		FileSystemUtils.ensureParentDirectoryExists(file);
		generator.generateFile(UNKNOWN_FONT_DESCRIPTION, file,
				EGraphvizOutputFormat.PNG);
		assertTrue(file.exists());
		assertTrue(file.length() > 0);
		file.delete();
	}

	/**
	 * Test method <code>generateImage</code>. This only checks if an images
	 * was created and if its width and height are greater zero.
	 */
	public void testGenerateImage() throws IOException, GraphvizException {
		BufferedImage image = generator.generateImage(TEST_DESCRIPTION);
		assertNotNull(image);
		assertTrue(image.getWidth() > 0);
		assertTrue(image.getHeight() > 0);
	}

	/**
	 * Test method <code>generateImage</code> with an erroneous graph
	 * description.
	 */
	public void testGenerateImageError() throws IOException {
		try {
			generator.generateImage(ERR_TEST_DESCRIPTION);
		} catch (GraphvizException e) {
			return;
		}
		fail("Should create exception");
	}

	/**
	 * Test method <code>generateImage</code> with a graph description that
	 * specifies an unknown font. Graph should be generated even with unknown
	 * font. This documents bug #495.
	 * 
	 * @throws GraphvizException
	 */
	public void testGenerateImageUnknownFont() throws IOException,
			GraphvizException {
		BufferedImage image = generator.generateImage(UNKNOWN_FONT_DESCRIPTION);
		assertNotNull(image);
		assertTrue(image.getWidth() > 0);
		assertTrue(image.getHeight() > 0);
	}

	/** Create a lenghty description. */
	private String createDescription(String prefix) {
		StringBuilder description = new StringBuilder();

		description.append("digraph G {");

		for (int i = 0; i < 100; i++) {
			description.append("A");
			description.append(prefix);
			description.append(i);
			description.append(" -> ");
			description.append("B");
			description.append(i);
			description.append(";");
			description.append(StringUtils.CR);
		}

		description.append("}");

		return description.toString();
	}
}