/*
 * libexplain - Explain errno values returned by libc functions
 * Copyright (C) 2009 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful,but
 * WITHOUT ANY WARRANTY; without even the implied warranty
 * ofMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNULesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <libexplain/ac/sys/types.h> /* must be first */
#include <libexplain/ac/linux/hayesesp.h>
#include <libexplain/ac/linux/serial.h>
#include <libexplain/ac/linux/termios.h>
#ifndef HAVE_LINUX_TERMIOS_H
#include <libexplain/ac/termios.h>
#include <libexplain/ac/sys/ioctl.h>
#endif

#include <libexplain/buffer/hayes_esp_config.h>
#include <libexplain/buffer/line_discipline.h>
#include <libexplain/buffer/serial_icounter_struct.h>
#include <libexplain/buffer/serial_struct.h>
#include <libexplain/buffer/termio.h>
#include <libexplain/buffer/termios.h>
#include <libexplain/buffer/termios2.h>
#include <libexplain/buffer/termiox.h>
#include <libexplain/buffer/winsize.h>
#include <libexplain/buffer/modem_flags.h>

#include <ioctl-scan/iocontrol.h>
#include <ioctl-scan/report.h>
#include <ioctl-scan/scan/asm_ioctls.h>


#ifdef TCGETA

static void
tcgeta(int fildes)
{
    struct termio   data;

    if (iocontrol(fildes, TCGETA, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_termio(&sb, &data);
        report("TCGETA", "Terminal attributes", text);
    }
}

#endif

#ifdef TCGETS

static void
tcgets(int fildes)
{
    struct termios  data;

    if (iocontrol(fildes, TCGETS, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_termios(&sb, &data);
        report("TCGETS", "Terminal attributes", text);
    }
}

#endif

#ifdef TCGETS2
#ifdef HAVE_LINUX_TERMIOS_H

static void
tcgets2(int fildes)
{
    struct termios2 data;

    if (iocontrol(fildes, TCGETS2, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_termios2(&sb, &data);
        report("TCGETS2", "Terminal attributes", text);
    }
}

#endif
#endif

#ifdef TCGETX
#ifdef HAVE_LINUX_TERMIOS_H

static void
tcgetx(int fildes)
{
    struct termiox  data;

    if (iocontrol(fildes, TCGETX, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_termiox(&sb, &data);
        report("TCGETX", "Terminal attributes", text);
    }
}

#endif
#endif

#ifdef TIOCGETD

static void
tiocgetd(int fildes)
{
    int             data;

    if (iocontrol(fildes, TIOCGETD, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_line_discipline(&sb, data);
        report("TIOCGETD", "Line discipline", text);
    }
}

#endif

#ifdef TIOCGHAYESESP
#ifdef HAVE_LINUX_TERMIOS_H

static void
tiocghayesesp(int fildes)
{
    struct hayes_esp_config data;

    if (iocontrol(fildes, TIOCGHAYESESP, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_hayes_esp_config(&sb, &data);
        report("TIOCGHAYESESP", "Hayes ESP Config", text);
    }
}

#endif
#endif

#ifdef TIOCGICOUNT
#ifdef HAVE_LINUX_SERIAL_H

static void
tiocgicount(int fildes)
{
    struct serial_icounter_struct data;

    if (iocontrol(fildes, TIOCGICOUNT, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_serial_icounter_struct(&sb, &data);
        report("TIOCGICOUNT", "Interrupt count", text);
    }
}

#endif
#endif

#ifdef TIOCGLCKTRMIOS

static void
tiocglcktrmios(int fildes)
{
    struct termios  data;

    if (iocontrol(fildes, TIOCGLCKTRMIOS, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_termios(&sb, &data);
        report("TIOCGLCKTRMIOS", "Terminal attributes", text);
    }
}

#endif

#ifdef TIOCGPGRP

static void
tiocgpgrp(int fildes)
{
    pid_t           data;

    if (iocontrol(fildes, TIOCGPGRP, &data) >= 0)
    {
        report_int("TIOCGPGRP", "Process group", data, 0);
    }
}

#endif

#ifdef TIOCGPTN

static void
tiocgptn(int fildes)
{
    unsigned        data;

    if (iocontrol(fildes, TIOCGPTN, &data) >= 0)
    {
        report_int("TIOCGPTN", "Pseudo terminal number", data, 0);
    }
}

#endif

#if 0 /* def TIOCGRS485 */

static void
tiocgrs485(int fildes)
{
    struct serial_rs485_settings data;

    if (iocontrol(fildes, TIOCGRS485, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_serial_rs485_settings(&sb, &data);
        report_int("TIOCGRS485", "RS485 settings", data, 0);
    }
}

#endif

#ifdef TIOCGSERIAL
#ifdef HAVE_LINUX_SERIAL_H

static void
tiocgserial(int fildes)
{
    struct serial_struct data;

    if (iocontrol(fildes, TIOCGSERIAL, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_serial_struct(&sb, &data);
        report("TIOCGSERIAL", "Serial attributes", text);
    }
}

#endif
#endif

#ifdef TIOCGSID

static void
tiocgsid(int fildes)
{
    pid_t           data;

    if (iocontrol(fildes, TIOCGSID, &data) >= 0)
    {
        report_int("TIOCGSID", "Session ID", data, 0);
    }
}

#endif

#ifdef TIOCGSOFTCAR

static void
tiocgsoftcar(int fildes)
{
    int             data;

    /* same as the CLOCAL flag */
    if (iocontrol(fildes, TIOCGSOFTCAR, &data) >= 0)
    {
        report_bool("TIOCGSOFTCAR", "Software carrier", data);
    }
}

#endif

#ifdef TIOCGWINSZ

static void
tiocgwinsz(int fildes)
{
    struct winsize  data;

    if (iocontrol(fildes, TIOCGWINSZ, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_winsize(&sb, &data);
        report("TIOCGWINSZ", "Window size", text);
    }
}

#endif

#ifdef TIOCMGET

static void
tiocmget(int fildes)
{
    int             data;

    if (iocontrol(fildes, TIOCMGET, &data) >= 0)
    {
        char            text[1000];
        explain_string_buffer_t sb;

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_modem_flags(&sb, data);
        report("TIOCMGET", "Modem flags", text);
    }
}

#endif


void
scan_asm_ioctls(int fildes)
{
#ifdef TCGETA
    tcgeta(fildes);
#endif
#ifdef TCGETS
    tcgets(fildes);
#endif
#ifdef TCGETS2
#ifdef HAVE_LINUX_TERMIOS_H
    tcgets2(fildes);
#endif
#endif
#ifdef TCGETX
#ifdef HAVE_LINUX_TERMIOS_H
    tcgetx(fildes);
#endif
#endif
#ifdef TIOCGETD
    tiocgetd(fildes);
#endif
#ifdef TIOCGHAYESESP
#ifdef HAVE_LINUX_TERMIOS_H
    tiocghayesesp(fildes);
#endif
#endif
#ifdef TIOCGICOUNT
#ifdef HAVE_LINUX_SERIAL_H
    tiocgicount(fildes);
#endif
#endif
#ifdef TIOCGLCKTRMIOS
    tiocglcktrmios(fildes);
#endif
#ifdef TIOCGPGRP
    tiocgpgrp(fildes);
#endif
#ifdef TIOCGPTN
    tiocgptn(fildes);
#endif
#if 0 /* def TIOCGRS485 */
    tiocgrs485(fildes);
#endif
#ifdef TIOCGSERIAL
#ifdef HAVE_LINUX_SERIAL_H
    tiocgserial(fildes);
#endif
#endif
#ifdef TIOCGSID
    tiocgsid(fildes);
#endif
#ifdef TIOCGSOFTCAR
    tiocgsoftcar(fildes);
#endif
#ifdef TIOCGWINSZ
    tiocgwinsz(fildes);
#endif
#ifdef TIOCMGET
    tiocmget(fildes);
#endif
}
