/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.cache.config;

import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.util.Util;

import java.lang.reflect.Method;

public class EvictionRegionConfig extends ConfigurationComponent
{
   /**
    * The serialVersionUID
    */
   private static final long serialVersionUID = -5482474634995601400L;

   public static final String NAME = "name";
   public static final String REGION = "region";
   public static final String REGION_POLICY_CLASS = "policyClass";
   public static final String EVENT_QUEUE_SIZE = "eventQueueSize";

   private Fqn regionFqn;
   @Dynamic
   private Integer eventQueueSize;
   private EvictionPolicyConfig evictionPolicyConfig;

   public EvictionRegionConfig()
   {
   }

   public EvictionRegionConfig(Fqn regionFqn, EvictionPolicyConfig evictionPolicyConfig)
   {
      this.regionFqn = regionFqn;
      this.evictionPolicyConfig = evictionPolicyConfig;
   }

   public EvictionPolicyConfig getEvictionPolicyConfig()
   {
      return evictionPolicyConfig;
   }

   public void setEvictionPolicyConfig(EvictionPolicyConfig config)
   {
      testImmutability("evictionPolicyConfig");
      if (this.evictionPolicyConfig instanceof ConfigurationComponent)
      {
         removeChildConfig((ConfigurationComponent) this.evictionPolicyConfig);
      }
      if (config instanceof ConfigurationComponent)
      {
         addChildConfig((ConfigurationComponent) config);
      }

      // don't validate here - instead validate when we start things up.  See RegionManager.start()

      this.evictionPolicyConfig = config;
   }

   public Fqn getRegionFqn()
   {
      return regionFqn;
   }

   public void setRegionFqn(Fqn regionFqn)
   {
      testImmutability("regionFqn");
      this.regionFqn = regionFqn;
   }

   public String getRegionName()
   {
      return regionFqn == null ? null : regionFqn.toString();
   }

   public void setRegionName(String name)
   {
      setRegionFqn(name == null ? null : Fqn.fromString(name));
   }

   public int getEventQueueSize()
   {
      return eventQueueSize == null ? EvictionConfig.EVENT_QUEUE_SIZE_DEFAULT : eventQueueSize;
   }

   public void setEventQueueSize(int queueSize)
   {
      testImmutability("eventQueueSize");
      if (queueSize <= 0)
      {
         LogFactory.getLog(EvictionRegionConfig.class).warn("Ignoring invalid queue capacity " +
               queueSize + " -- using " +
               EvictionConfig.EVENT_QUEUE_SIZE_DEFAULT);
         queueSize = EvictionConfig.EVENT_QUEUE_SIZE_DEFAULT;
      }
      this.eventQueueSize = queueSize;
   }

   public void setDefaultEventQueueSize(int queueSize)
   {
      if (eventQueueSize == null)
         setEventQueueSize(queueSize);
   }


   @Override
   public boolean equals(Object obj)
   {
      if (this == obj)
         return true;

      if (obj instanceof EvictionRegionConfig)
      {
         EvictionRegionConfig other = (EvictionRegionConfig) obj;
         return Util.safeEquals(this.regionFqn, other.regionFqn);
      }
      return false;
   }

   @Override
   public int hashCode()
   {
      int result = 17;
      result = 31 * result + (regionFqn == null ? 0 : regionFqn.hashCode());

      return result;
   }

   @Override
   public EvictionRegionConfig clone() throws CloneNotSupportedException
   {
      EvictionRegionConfig clone = (EvictionRegionConfig) super.clone();
      if (evictionPolicyConfig != null)
      {
         if (!(evictionPolicyConfig instanceof Cloneable))
         {
            throw new CloneNotSupportedException(evictionPolicyConfig + " is not Cloneable");
         }

         if (evictionPolicyConfig instanceof ConfigurationComponent)
         {
            clone.setEvictionPolicyConfig((EvictionPolicyConfig) ((ConfigurationComponent) evictionPolicyConfig).clone());
         }
         else
         {
            try
            {
               Method cloneMethod = evictionPolicyConfig.getClass().getDeclaredMethod("clone");
               EvictionPolicyConfig epc = (EvictionPolicyConfig) cloneMethod.invoke(evictionPolicyConfig);
               clone.setEvictionPolicyConfig(epc);
            }
            catch (Exception e)
            {
               CloneNotSupportedException cnse = new CloneNotSupportedException("Cannot invoke clone() on " + evictionPolicyConfig);
               cnse.initCause(e);
               throw cnse;
            }
         }
      }

      return clone;
   }


}