/*
 * Copyright (C) 2012  Christian Mollekopf <mollekopf@kolabsys.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include "kolabobject.h"
#include "v2helpers.h"
#include "kolabdefinitions.h"
#include "errorhandler.h"
#include "libkolab-version.h"

#include <kolabbase.h>
#include <kolabformatV2/journal.h>
#include <kolabformatV2/task.h>
#include <kolabformatV2/event.h>
#include <kolabformatV2/contact.h>
#include <kolabformatV2/distributionlist.h>
#include <kolabformatV2/note.h>
#include <mime/mimeutils.h>
#include <conversion/kcalconversion.h>
#include <conversion/kabcconversion.h>
#include <conversion/kolabconversion.h>
#include <akonadi/notes/noteutils.h>
#include <kolab/kolabformat.h>


namespace Kolab {

KCalCore::Event::Ptr readV2EventXML(const QByteArray& xmlData, QStringList& attachments)
{
    return fromXML<KCalCore::Event::Ptr, KolabV2::Event>(xmlData, attachments);
}


//@cond PRIVATE
class KolabObjectReader::Private
{
public:
    Private()
    : mObjectType( InvalidObject ),
    mVersion( KolabV3 )
    {
        mAddressee = KABC::Addressee();
    }
    
    KCalCore::Incidence::Ptr mIncidence;
    KABC::Addressee mAddressee;
    KABC::ContactGroup mContactGroup;
    KMime::Message::Ptr mNote;
    ObjectType mObjectType;
    Version mVersion;
};
//@endcond

KolabObjectReader::KolabObjectReader()
: d( new KolabObjectReader::Private )
{
}

KolabObjectReader::KolabObjectReader(const KMime::Message::Ptr& msg)
: d( new KolabObjectReader::Private )
{
    parseMimeMessage(msg);
}

KolabObjectReader::~KolabObjectReader()
{
    delete d;
}

Kolab::ObjectType getObjectType(const QString &type)
{
    if (type == eventKolabType()) {
        return EventObject;
    } else if (type == todoKolabType()) {
        return TodoObject;
    } else if (type == journalKolabType()) {
        return JournalObject;
    } else if (type == contactKolabType()) {
        return ContactObject;
    }  else if (type == distlistKolabType() || type == distlistKolabTypeCompat()) {
        return DistlistObject;
    }  else if (type == noteKolabType()) {
        return NoteObject;
    }
    return Kolab::InvalidObject;
}

QByteArray getMimeType(Kolab::ObjectType type)
{
    switch (type) {
        case EventObject:
        case TodoObject:
        case JournalObject:
            return xCalMimeType().toLocal8Bit();
        case ContactObject:
        case DistlistObject:
            return xCardMimeType().toLocal8Bit();
        case NoteObject:
            return noteMimeType().toLocal8Bit();
        default:
            Critical() << "unknown type "<< type;
    }
    return QByteArray();
}

ObjectType KolabObjectReader::parseMimeMessage(const KMime::Message::Ptr &msg)
{
    ErrorHandler::clearErrors();
    d->mObjectType = InvalidObject;
    KMime::Headers::Base *xKolabHeader = msg->getHeaderByType(X_KOLAB_TYPE_HEADER);
    if (!xKolabHeader) {
        CRITICAL("could not find xKolabHeader");
        return InvalidObject;
    }
    const QString &kolabType = xKolabHeader->asUnicodeString(); //TODO we probably shouldn't use unicodeString
    
    KMime::Headers::Base *xKolabVersion = msg->getHeaderByType(X_KOLAB_VERSION_HEADER); //TODO make sure v3 is written out
    if (!xKolabVersion) {
        d->mVersion = KolabV2;
    } else {
        if (xKolabVersion->asUnicodeString() == KOLAB_VERSION_V3) {
            d->mVersion = KolabV3;
        }
    }
    d->mObjectType = getObjectType(kolabType);
    if (d->mVersion == KolabV2) {
        KMime::Content *xmlContent = Mime::findContentByType( msg, kolabType.toLocal8Bit() );
        if ( !xmlContent ) {
            CRITICAL("no part found");
            d->mObjectType = InvalidObject;
            return InvalidObject;
        }
        const QByteArray &xmlData = xmlContent->decodedContent();
        Q_ASSERT(!xmlData.isEmpty());
        QStringList attachments;

        switch (d->mObjectType) {
            case EventObject:
                d->mIncidence = fromXML<KCalCore::Event::Ptr, KolabV2::Event>(xmlData, attachments);
                break;
            case TodoObject:
                d->mIncidence = fromXML<KCalCore::Todo::Ptr, KolabV2::Task>(xmlData, attachments);
                break;
            case JournalObject:
                d->mIncidence = fromXML<KCalCore::Journal::Ptr, KolabV2::Journal>(xmlData, attachments);
                break;
            case ContactObject:
                d->mAddressee = addresseeFromKolab(xmlData, msg);
                break;
            case DistlistObject:
                d->mContactGroup = contactGroupFromKolab(xmlData);
                break;
            case NoteObject:
                d->mNote = noteFromKolab(xmlData, msg);
                break;
            default:
                CRITICAL("no kolab object found "+kolabType);
                break;
        }
        if (!d->mIncidence.isNull()) {
//             kDebug() << "v2 attachments " << attachments.size() << d->mIncidence->attachments().size();
            d->mIncidence->clearAttachments();
            Mime::getAttachments(d->mIncidence, attachments, msg);
            Q_ASSERT(d->mIncidence->attachments().size() == attachments.size());
        }
    } else { //V3
        KMime::Content *xmlContent = Mime::findContentByType( msg, getMimeType(d->mObjectType) );
        if ( !xmlContent ) {
            CRITICAL("no part found");
            d->mObjectType = InvalidObject;
            return InvalidObject;
        }
        switch (d->mObjectType) {
            case EventObject: {
                const Kolab::Event & event = Kolab::readEvent(std::string(xmlContent->decodedContent().data(), xmlContent->decodedContent().size()), false);
                d->mIncidence = Kolab::Conversion::toKCalCore(event);
            }
                break;
            case TodoObject: {
                const Kolab::Todo & event = Kolab::readTodo(std::string(xmlContent->decodedContent().data(), xmlContent->decodedContent().size()), false);
                d->mIncidence = Kolab::Conversion::toKCalCore(event);
            }
                break;
            case JournalObject: {
                const Kolab::Journal & event = Kolab::readJournal(std::string(xmlContent->decodedContent().data(), xmlContent->decodedContent().size()), false);
                d->mIncidence = Kolab::Conversion::toKCalCore(event);
            }
                break;
            case ContactObject: {
                const Kolab::Contact &contact = Kolab::readContact(std::string(xmlContent->decodedContent().data(), xmlContent->decodedContent().size()), false);
                d->mAddressee = Kolab::Conversion::toKABC(contact); //TODO extract attachments
            }
                break;
            case DistlistObject: {
                const Kolab::DistList &distlist = Kolab::readDistlist(std::string(xmlContent->decodedContent().data(), xmlContent->decodedContent().size()), false);
                d->mContactGroup = Kolab::Conversion::toKABC(distlist);
            }
                break;
            case NoteObject: {
                const Kolab::Note &note = Kolab::readNote(std::string(xmlContent->decodedContent().data(), xmlContent->decodedContent().size()), false);
                d->mNote = Kolab::Conversion::toNote(note);
            }
                break;
            default:
                CRITICAL("no kolab object found "+kolabType);
                break;
        }

        if (!d->mIncidence.isNull()) {
//             kDebug() << "getting attachments";
            Mime::getAttachmentsById(d->mIncidence, msg);
        }
        ErrorHandler::handleLibkolabxmlErrors();
    }
    return d->mObjectType;
}

Version KolabObjectReader::getVersion() const
{
    return d->mVersion;
}

ObjectType KolabObjectReader::getType() const
{
    return d->mObjectType;
}

KCalCore::Event::Ptr KolabObjectReader::getEvent() const
{
    return d->mIncidence.dynamicCast<KCalCore::Event>();
}

KCalCore::Todo::Ptr KolabObjectReader::getTodo() const
{
    return d->mIncidence.dynamicCast<KCalCore::Todo>();
}

KCalCore::Journal::Ptr KolabObjectReader::getJournal() const
{
    return d->mIncidence.dynamicCast<KCalCore::Journal>();
}

KCalCore::Incidence::Ptr KolabObjectReader::getIncidence() const
{
    return d->mIncidence;
}

KABC::Addressee KolabObjectReader::getContact() const
{
    return d->mAddressee;
}

KABC::ContactGroup KolabObjectReader::getDistlist() const
{
    return d->mContactGroup;
}

KMime::Message::Ptr KolabObjectReader::getNote() const
{
    return d->mNote;
}

//Normalize incidences before serializing them
KCalCore::Incidence::Ptr normalizeIncidence(KCalCore::Incidence::Ptr original)
{
    KCalCore::Incidence::Ptr i = KCalCore::Incidence::Ptr(original->clone()); //We copy to avoid destructive writing
    Q_FOREACH (KCalCore::Attachment::Ptr attachment, i->attachments()) {
        attachment->setUri(QString::fromLatin1("cid:")+QString::fromLatin1(KMime::uniqueString() + '@' + "kolab.resource.akonadi")); //Serialize the attachment as attachment with uri, referencing the created mime-part
    }
    return i;
}
/*
KABC::Addressee normalizeContact(const KABC::Addressee &a)
{
    KABC::Addressee addresee = a;
    Q_FOREACH (KCalCore::Attachment::Ptr attachment, addresee.photo()) {
        attachment->setUri(QString::fromLatin1("cid:")+QString::fromLatin1(KMime::uniqueString() + '@' + "kolab.resource.akonadi")); //Serialize the attachment as attachment with uri, referencing the created mime-part
    }
    return i;
}*/

QString getProductId(const QString &pId)
{
    if (pId.isEmpty()) {
        return LIBKOLAB_LIB_VERSION_STRING;
    }
    return pId+" "+LIBKOLAB_LIB_VERSION_STRING;
}

KMime::Message::Ptr KolabObjectWriter::writeEvent(const KCalCore::Event::Ptr &i, Version v, const QString &productId, const QString &tz)
{
    ErrorHandler::clearErrors();
    Q_ASSERT(!i.isNull());
    if (v == KolabV3) {
        KCalCore::Event::Ptr ic = normalizeIncidence(i).dynamicCast<KCalCore::Event>();
        const Kolab::Event &incidence = Kolab::Conversion::fromKCalCore(*ic);
        const std::string &v3String = Kolab::writeEvent(incidence, getProductId(productId).toStdString());
        ErrorHandler::handleLibkolabxmlErrors();
        return Mime::createMessage(ic, xCalMimeType(), eventKolabType(), QString::fromStdString(v3String).toLocal8Bit(), true, getProductId(productId));
    }
    const QString &xml = KolabV2::Event::eventToXML(i, tz);
    return Mime::createMessage(i, eventKolabType(), eventKolabType(), xml.toLocal8Bit(), false, getProductId(productId));
}

KMime::Message::Ptr KolabObjectWriter::writeTodo(const KCalCore::Todo::Ptr &i, Version v, const QString &productId, const QString &tz)
{
    ErrorHandler::clearErrors();
    Q_ASSERT(!i.isNull());
    if (v == KolabV3) {
        KCalCore::Todo::Ptr ic = normalizeIncidence(i).dynamicCast<KCalCore::Todo>();
        const Kolab::Todo &incidence = Kolab::Conversion::fromKCalCore(*ic);
        const std::string &v3String = Kolab::writeTodo(incidence, getProductId(productId).toStdString());
        ErrorHandler::handleLibkolabxmlErrors();
        return Mime::createMessage(ic, xCalMimeType(), todoKolabType(), QString::fromStdString(v3String).toLocal8Bit(), true, getProductId(productId));
    }
    const QString &xml = KolabV2::Task::taskToXML(i, tz);
    return Mime::createMessage(i, todoKolabType(), todoKolabType(), xml.toLocal8Bit(), false, getProductId(productId));
}

KMime::Message::Ptr KolabObjectWriter::writeJournal(const KCalCore::Journal::Ptr &i, Version v, const QString &productId, const QString &tz)
{
    ErrorHandler::clearErrors();
    Q_ASSERT(!i.isNull());
    if (v == KolabV3) {
        KCalCore::Journal::Ptr ic = normalizeIncidence(i).dynamicCast<KCalCore::Journal>();
        const Kolab::Journal &incidence = Kolab::Conversion::fromKCalCore(*ic);
        const std::string &v3String = Kolab::writeJournal(incidence, getProductId(productId).toStdString());
        ErrorHandler::handleLibkolabxmlErrors();
        return  Mime::createMessage(ic, xCalMimeType(), journalKolabType(), QString::fromStdString(v3String).toLocal8Bit(), true, getProductId(productId));
    }
    const QString &xml = KolabV2::Journal::journalToXML(i, tz);
    return Mime::createMessage(i, journalKolabType(), journalKolabType(), xml.toLocal8Bit(), false, getProductId(productId));
}

KMime::Message::Ptr KolabObjectWriter::writeIncidence(const KCalCore::Incidence::Ptr &i, Version v, const QString& productId, const QString& tz)
{
    switch (i->type()) {
        case KCalCore::IncidenceBase::TypeEvent:
            return writeEvent(i.dynamicCast<KCalCore::Event>(),v,productId,tz);
        case KCalCore::IncidenceBase::TypeTodo:
            return writeTodo(i.dynamicCast<KCalCore::Todo>(),v,productId,tz);
        case KCalCore::IncidenceBase::TypeJournal:
            return writeJournal(i.dynamicCast<KCalCore::Journal>(),v,productId,tz);
        default:
            Critical() << "unknown incidence type";
    }
    return KMime::Message::Ptr();
}


KMime::Message::Ptr KolabObjectWriter::writeContact(const KABC::Addressee &addressee, Version v, const QString &productId)
{
    ErrorHandler::clearErrors();
    if (v == KolabV3) {
        const Kolab::Contact &contact = Kolab::Conversion::fromKABC(addressee);
        const std::string &v3String = Kolab::writeContact(contact, getProductId(productId).toStdString());
        ErrorHandler::handleLibkolabxmlErrors();
        return  Mime::createMessage(addressee, xCardMimeType(), contactKolabType(), QString::fromStdString(v3String).toLocal8Bit(), true, getProductId(productId));
    }
    KolabV2::Contact contact(&addressee);
    return contactToKolabFormat(contact, getProductId(productId));
}

KMime::Message::Ptr KolabObjectWriter::writeDistlist(const KABC::ContactGroup &distlist, Version v, const QString &productId)
{
    ErrorHandler::clearErrors();
    if (v == KolabV3) {
        const Kolab::DistList &dist = Kolab::Conversion::fromKABC(distlist);
        const std::string &v3String = Kolab::writeDistlist(dist, getProductId(productId).toStdString());
        ErrorHandler::handleLibkolabxmlErrors();
        return  Mime::createMessage(QString(), xCardMimeType(), contactKolabType(), QString::fromStdString(v3String).toLocal8Bit(), true, getProductId(productId));
    }
    KolabV2::DistributionList d(&distlist);
    return distListToKolabFormat(d, getProductId(productId));
}

KMime::Message::Ptr KolabObjectWriter::writeNote(const KMime::Message::Ptr &note, Version v, const QString &productId)
{
    ErrorHandler::clearErrors();
    Q_ASSERT(note.get());
    if (v == KolabV3) {
        const Kolab::Note &n = Kolab::Conversion::fromNote(note);
        const std::string &v3String = Kolab::writeNote(n, getProductId(productId).toStdString());
        ErrorHandler::handleLibkolabxmlErrors();
        return  Mime::createMessage(QString::fromStdString(n.summary()) ,noteMimeType(), noteKolabType(), QString::fromStdString(v3String).toLocal8Bit(), true, getProductId(productId));
    }
    return noteToKolab(note, getProductId(productId));
}






}; //Namespace

