/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.options.macros;

import java.awt.Dialog;
import java.util.ResourceBundle;
import javax.swing.*;
import java.awt.event.*;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.text.JTextComponent;
import org.netbeans.api.editor.mimelookup.MimeLookup;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.MultiKeyBinding;
import org.netbeans.core.options.keymap.api.ShortcutAction;
import org.netbeans.core.options.keymap.api.ShortcutsFinder;
import org.netbeans.editor.BaseAction;
import org.netbeans.editor.BaseKit;
import org.netbeans.editor.DialogSupport;
import org.netbeans.editor.Settings;
import org.netbeans.editor.SettingsNames;
import org.netbeans.editor.Utilities;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.editor.options.BaseOptions;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.modules.editor.settings.storage.api.KeyBindingSettingsFactory;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;


/** The support for creating macros.
 *
 * @author  Petr Nejedly
 * @version 1.0
 */
public final class MacroDialogSupport implements ActionListener {

    private static final Logger LOG = Logger.getLogger(MacroDialogSupport.class.getName());

    private Dialog macroDialog;
    private final MacroSavePanel panel;
    private final JButton okButton;
    private final JButton cancelButton;

    private final String mimeType;
    private final Class kitClass;
    
    /** Creates new MacroDialogSupport */
    public MacroDialogSupport( String mimeType, Class kitClass ) {
        this.mimeType = mimeType;
        this.kitClass = kitClass;
        panel = new MacroSavePanel(mimeType);
        ResourceBundle bundle = NbBundle.getBundle(MacroDialogSupport.class);
        okButton = new JButton(bundle.getString("MDS_ok")); // NOI18N
        cancelButton = new JButton(bundle.getString("MDS_cancel")); // NOI18N
        okButton.getAccessibleContext().setAccessibleDescription(bundle.getString("ACSD_MDS_ok")); // NOI18N
        cancelButton.getAccessibleContext().setAccessibleDescription(bundle.getString("ACSD_MDS_cancel")); // NOI18N
    }

    public void setBody( String body ) {
        panel.setMacroBody( body );
    }
    
    public void showMacroDialog() {
        macroDialog = DialogSupport.createDialog(
                NbBundle.getBundle(MacroDialogSupport.class).getString("MDS_title"), // NOI18N
                panel, true, new JButton[] { okButton, cancelButton }, false, 0, 1, this );

        macroDialog.pack();
        panel.popupNotify();
        macroDialog.requestFocus();
        macroDialog.show();
    }
    
    private void saveMacro(boolean overwriting) {
        // Save the new macro
        Set<String> mimeTypes = EditorSettings.getDefault().getMimeTypes();
        for (String mt : mimeTypes) {
            BaseOptions baseOptions = MimeLookup.getLookup(MimePath.parse(mt)).lookup(BaseOptions.class);
            if (baseOptions != null) {
                Map<String, String> macroMap = baseOptions.getMacroMap();
                Map<String, String> newMap = new HashMap<String, String>(macroMap);
                newMap.put(panel.getMacroName(), panel.getMacroBody());
                baseOptions.setMacroMap(newMap);
            }
        }

        // Save the macro's keybinding
        String macroActionName = BaseKit.macroActionPrefix + panel.getMacroName();

        if (overwriting) {
            // overwriting existing macro. Remove all previously attached keybindings.
            String profile = EditorSettings.getDefault().getCurrentKeyMapProfile();
            KeyBindingSettingsFactory kbsf = EditorSettings.getDefault().getKeyBindingSettings(new String[]{mimeType});
            List<MultiKeyBinding> keybindings = kbsf.getKeyBindings(profile);
            
            List<MultiKeyBinding> removed = new ArrayList<MultiKeyBinding>();
            for (int i = 0; i < keybindings.size(); i++) {
                MultiKeyBinding multiKey = keybindings.get(i);
                if (multiKey.getActionName() != null && multiKey.getActionName().equals(macroActionName)) {
                    removed.add(multiKey);
                }
            }
            for (int i = 0; i < removed.size(); i++) {
                keybindings.remove(removed.get(i));
            }
            
            kbsf.setKeyBindings(profile, keybindings);
        }

        Set<String> shortcuts = panel.getKeySequences();
        if (shortcuts.size() > 0) {
            ShortcutsFinder f = Lookup.getDefault().lookup(ShortcutsFinder.class);
            f.refreshActions();
            ShortcutAction shortcutAction = f.findActionForId(macroActionName);
            f.setShortcuts(shortcutAction, shortcuts);
            f.apply();
        }
        
        // reload the model (if one was already created)
        MacrosModel macrosModel = MacrosModel.getModel(null);
        if (macrosModel != null) {
            macrosModel.reset();
        }
    }
    
    protected int showConfirmDialog(String macroName){
        return JOptionPane.showConfirmDialog(panel,                     
            MessageFormat.format(NbBundle.getBundle(MacroDialogSupport.class).getString("MDS_Overwrite"), //NOI18N
                new Object[] {panel.getMacroName()}), 
            NbBundle.getBundle(MacroDialogSupport.class).getString("MDS_Warning"), // NOI18N
            JOptionPane.YES_NO_CANCEL_OPTION,
            JOptionPane.WARNING_MESSAGE);
    }
    
    public void actionPerformed(java.awt.event.ActionEvent evt ) {
        Object source = evt.getSource();
        if( source == okButton ) {
            if (panel.getMacroName() == null || panel.getMacroName().length() == 0 || 
                panel.getMacroName().trim().length() == 0
            ) {
                DialogDisplayer.getDefault ().notify (
                    new NotifyDescriptor.Message (
                        NbBundle.getBundle(MacroDialogSupport.class).getString("MDS_Empty_Name"), //NOI18N
                        NotifyDescriptor.ERROR_MESSAGE
                    )
                );
                
                panel.nameField.requestFocusInWindow();
                return;
            }
            Map macroMap = (Map)Settings.getValue( kitClass, SettingsNames.MACRO_MAP);
            
            if (!macroMap.containsKey(panel.getMacroName())){
                saveMacro(false);
            }else{
                int retVal = showConfirmDialog(panel.getMacroName());
                if (retVal == JOptionPane.CANCEL_OPTION || retVal == JOptionPane.CLOSED_OPTION) return;
                if (retVal == JOptionPane.OK_OPTION) saveMacro(true);
            }
        }
        macroDialog.setVisible( false );
        macroDialog.dispose();        
    }

    public static class StartMacroRecordingAction extends BaseAction {

        static final long serialVersionUID =1L;

        public StartMacroRecordingAction() {
            super( BaseKit.startMacroRecordingAction, NO_RECORDING );
            putValue(BaseAction.ICON_RESOURCE_PROPERTY,
                "org/netbeans/modules/options/macros/start_macro_recording.png"); // NOI18N
        }
        
        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                if( !startRecording(target) ) target.getToolkit().beep();
            }
        }
        
        private boolean startRecording(JTextComponent c) {
            try {
                Method m = BaseAction.class.getDeclaredMethod("startRecording", JTextComponent.class); //NOI18N
                m.setAccessible(true);
                return (Boolean) m.invoke(this, c);
            } catch (Exception e) {
                LOG.log(Level.WARNING, "Can't call BaseAction.startRecording", e); //NOI18N
                return false;
            }
        }
    } // End of StartMacroRecordingAction class

    public static final class StopMacroRecordingAction extends BaseAction {

        static final long serialVersionUID =1L;

        public StopMacroRecordingAction() {
            super( BaseKit.stopMacroRecordingAction, NO_RECORDING );
            putValue(BaseAction.ICON_RESOURCE_PROPERTY,
                "org/netbeans/modules/options/macros/stop_macro_recording.png"); // NOI18N
        }
        
        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                String macro = stopRecording(target);
                if( macro == null ) { // not recording
                    target.getToolkit().beep();
                } else {
                    // popup a macro dialog
                    BaseKit kit = Utilities.getKit(target);
                    String mimeType = NbEditorUtilities.getMimeType(target);
                    MacroDialogSupport support = new MacroDialogSupport(mimeType, kit.getClass());
                    support.setBody( macro );
                    support.showMacroDialog();
                }
            }
        }
        
        private String stopRecording(JTextComponent c) {
            try {
                Method m = BaseAction.class.getDeclaredMethod("stopRecording", JTextComponent.class); //NOI18N
                m.setAccessible(true);
                return (String) m.invoke(this, c);
            } catch (Exception e) {
                LOG.log(Level.WARNING, "Can't call BaseAction.stopRecording", e); //NOI18N
                return null;
            }
        }
    } // End of StopMacroRecordingAction class
    
}
