/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.editor.settings.storage;

import java.awt.Color;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.text.AttributeSet;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.EditorStyleConstants;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.Repository;
import org.openide.xml.XMLUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;


/**
 * This class contains support static methods for loading / saving and 
 * translating coloring (fontsColors.xml) files. It calls XMLStorage utilities.
 *
 * @author Jan Jancura
 */
public final class ColoringStorage {

    // -J-Dorg.netbeans.modules.editor.settings.storage.ColoringStorage.level=FINE
    private static final Logger LOG = Logger.getLogger(ColoringStorage.class.getName());

    private static final String HIGHLIGHTING_FILE_NAME = "editorColoring.xml"; // NOI18N
    
    private static final String E_ROOT = "fontscolors"; //NOI18N
    private static final String E_FONTCOLOR = "fontcolor"; //NOI18N
    private static final String E_FONT = "font"; //NOI18N
    private static final String A_NAME = "name"; //NOI18N
    private static final String A_FOREGROUND = "foreColor"; //NOI18N
    private static final String A_BACKGROUND = "bgColor"; //NOI18N
    private static final String A_STRIKETHROUGH = "strikeThrough"; //NOI18N
    private static final String A_WAVEUNDERLINE = "waveUnderlined"; //NOI18N
    private static final String A_UNDERLINE = "underline"; //NOI18N
    private static final String A_DEFAULT = "default"; //NOI18N
    private static final String A_SIZE = "size"; //NOI18N
    private static final String A_STYLE = "style"; //NOI18N
    private static final String V_BOLD_ITALIC = "bold+italic"; //NOI18N
    private static final String V_BOLD = "bold"; //NOI18N
    private static final String V_ITALIC = "italic"; //NOI18N
    private static final String V_PLAIN = "plain"; //NOI18N
    
    private static final String PUBLIC_ID = "-//NetBeans//DTD Editor Fonts and Colors settings 1.1//EN"; //NOI18N
    private static final String SYSTEM_ID = "http://www.netbeans.org/dtds/EditorFontsColors-1_1.dtd"; //NOI18N

    private static final String FA_TYPE = "nbeditor-settings-ColoringType"; //NOI18N
    private static final String FAV_TOKEN = "token"; //NOI18N
    private static final String FAV_HIGHLIGHT = "highlight"; //NOI18N
    
    private static final Object ATTR_MODULE_SUPPLIED = new Object();
    
    private ColoringStorage() {
    }
    
    // load ....................................................................
    
    public static Map<String, AttributeSet> loadColorings(
        MimePath mimePath,
        String profile,
        boolean colorings, // true for colorings, false for highlightings
        boolean defaults   // read default values only
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        assert profile != null : "The parameter profile must not be null"; //NOI18N
        
        FileObject baseFolder = Repository.getDefault().getDefaultFileSystem().findResource("Editors"); //NOI18N
        Map<String, List<Object []>> files = new HashMap<String, List<Object []>>();
        SettingsType.FONTSCOLORS.getLocator().scan(baseFolder, mimePath.getPath(), profile, true, true, !defaults, files);
        
        assert files.size() <= 1 : "Too many results in the scan"; //NOI18N

        List<Object []> profileInfos = files.get(profile);
        if (profileInfos == null) {
            return null;
        }
        
        List<Object []> filesForLocalization; 
        if (!profile.equals(EditorSettingsImpl.DEFAULT_PROFILE)) {
            // If non-default profile load the default profile supplied by modules
            // to find the localizing bundles.
            Map<String, List<Object []>> defaultProfileModulesFiles = new HashMap<String, List<Object []>>();
            SettingsType.FONTSCOLORS.getLocator().scan(baseFolder, mimePath.getPath(), EditorSettingsImpl.DEFAULT_PROFILE, true, true, false, defaultProfileModulesFiles);
            filesForLocalization = defaultProfileModulesFiles.get(EditorSettingsImpl.DEFAULT_PROFILE);
            
            // if there is no default profile (eg. in tests)
            if (filesForLocalization == null) {
                filesForLocalization = Collections.<Object []>emptyList();
            }
        } else {
            filesForLocalization = profileInfos;
        }
        
        Map<String, SimpleAttributeSet> fontsColorsMap = new HashMap<String, SimpleAttributeSet>();
        for(Object [] info : profileInfos) {
            FileObject profileHome = (FileObject) info[0];
            FileObject settingFile = (FileObject) info[1];
            boolean modulesFile = ((Boolean) info[2]).booleanValue();

            // Skip files with wrong type of colorings
            boolean isTokenColoringFile = isTokenColoringFile(settingFile);
            if (isTokenColoringFile != colorings) {
                continue;
            }
            
            // Load colorings from the settingFile
            @SuppressWarnings("unchecked")
            List<SimpleAttributeSet> sets = (List<SimpleAttributeSet>) XMLStorage.load(settingFile, new ColoringsReader());
            
            // Process loaded colorings
            for(SimpleAttributeSet as : sets) {
                String name = (String) as.getAttribute(StyleConstants.NameAttribute);
                String translatedName = null;
                SimpleAttributeSet previous = fontsColorsMap.get(name);

                if (previous == null && !modulesFile && colorings) {
                    // User files normally don't define extra colorings unless
                    // for example loading a settings file from an older version
                    // of Netbeans (or in a completely new profile!!). In this case
                    // try simple heuristic for translating the name and if it does
                    // not work leave the name alone.
                    int idx = name.indexOf('-'); //NOI18N
                    if (idx != -1) {
                        translatedName = name.substring(idx + 1);
                        previous = fontsColorsMap.get(translatedName);
                        if (previous != null) {
                            // heuristics worked, fix the name and load the coloring
                            as.addAttribute(StyleConstants.NameAttribute, translatedName);
                            name = translatedName;
                        }
                    }
                }
                
                if (previous == null) {
                    // Find display name
                    String displayName = findDisplayName(name, settingFile, filesForLocalization);

                    if (displayName == null && !modulesFile) {
                        if (translatedName != null) {
                            displayName = findDisplayName(translatedName, settingFile, filesForLocalization);
                        }
                        if (displayName == null) {
                            // This coloring came from a user (no modules equivalent)
                            // and has no suitable display name. Probably an obsolete
                            // coloring from previous version, we will ignore it.
                            if (LOG.isLoggable(Level.FINE)) {
                                LOG.fine("Ignoring an extra coloring '" + name + "' that was not defined by modules."); //NOI18N
                            }
                            continue;
                        } else {
                            // fix the name
                            as.addAttribute(StyleConstants.NameAttribute, translatedName);
                            name = translatedName;
                        }
                    }
                    
                    if (displayName == null) {
                        displayName = name;
                    }

                    as.addAttribute(EditorStyleConstants.DisplayName, displayName);
                    as.addAttribute(ATTR_MODULE_SUPPLIED, modulesFile);
                    
                    fontsColorsMap.put(name, as);
                } else {
                    // the scanner alwyas returns modules files first, followed by user files
                    // when a coloring was defined in a user file it must not be merged
                    // with its default version supplied by modules
                    boolean moduleSupplied = (Boolean) previous.getAttribute(ATTR_MODULE_SUPPLIED);
                    if (moduleSupplied == modulesFile) {
                        mergeAttributeSets(previous, as);
                    } else {
                        // Copy over the display name and the link to the default coloring
                        as.addAttribute(EditorStyleConstants.DisplayName, previous.getAttribute(EditorStyleConstants.DisplayName));
                        Object df = previous.getAttribute(EditorStyleConstants.Default);
                        if (df != null) {
                            as.addAttribute(EditorStyleConstants.Default, df);
                        }
                        as.addAttribute(ATTR_MODULE_SUPPLIED, modulesFile);
                        
                        fontsColorsMap.put(name, as);
                    }
                }
            }
        }
            
        return Utils.immutize(fontsColorsMap, ATTR_MODULE_SUPPLIED);
    }

    private static String findDisplayName(String name, FileObject settingFile, List<Object []> filesForLocalization) {
        // Try the settingFile first
        String displayName = Utils.getLocalizedName(settingFile, name, null, true);

        // Then try all module files from the default profile
        if (displayName == null) {
            for(Object [] locFileInfo : filesForLocalization) {
                FileObject locFile = (FileObject) locFileInfo[1];
                displayName = Utils.getLocalizedName(locFile, name, null, true);
                if (displayName != null) {
                    break;
                }
            }
        }
        
        return displayName;
    }
    
    private static void mergeAttributeSets(SimpleAttributeSet original, AttributeSet toMerge) {
        for(Enumeration names = toMerge.getAttributeNames(); names.hasMoreElements(); ) {
            Object key = names.nextElement();
            Object value = toMerge.getAttribute(key);
            original.addAttribute(key, value);
        }
    }

    private static class ColoringsReader extends XMLStorage.Handler {
        
        private final List<AttributeSet> colorings = new ArrayList<AttributeSet>();
        
        public ColoringsReader() {
        }

        @Override
        public Object getResult () {
            return colorings;
        }
        
        @Override
        public void startElement (
            String uri, 
            String localName,
            String name, 
            Attributes attributes
        ) throws SAXException {
            try {
                if (name.equals(E_ROOT)) {
                    // We don't read anythhing from the root element
                    
                } else if (name.equals(E_FONTCOLOR)) {
                    SimpleAttributeSet a = new SimpleAttributeSet();
                    String value;

                    a.addAttribute(StyleConstants.NameAttribute, attributes.getValue(A_NAME));

                    value = attributes.getValue(A_BACKGROUND);
                    if (value != null) {
                        a.addAttribute(StyleConstants.Background, Utils.stringToColor(value));
                    }
                    
                    value = attributes.getValue(A_FOREGROUND);
                    if (value != null) {
                        a.addAttribute(StyleConstants.Foreground, Utils.stringToColor(value));
                    }

                    value = attributes.getValue(A_UNDERLINE);
                    if (value != null) {
                        a.addAttribute(StyleConstants.Underline, Utils.stringToColor(value));
                    }

                    value = attributes.getValue(A_STRIKETHROUGH);
                    if (value != null) {
                        a.addAttribute(StyleConstants.StrikeThrough, Utils.stringToColor(value));
                    }

                    value = attributes.getValue(A_WAVEUNDERLINE);
                    if (value != null) {
                        a.addAttribute(EditorStyleConstants.WaveUnderlineColor, Utils.stringToColor(value));
                    }
                    
                    value = attributes.getValue(A_DEFAULT);
                    if (value != null) {
                        a.addAttribute(EditorStyleConstants.Default, value);
                    }
                    
                    colorings.add (a);
                    
                } else if (name.equals(E_FONT)) {
                    SimpleAttributeSet a = (SimpleAttributeSet) colorings.get(colorings.size() - 1);
                    String value;
                    
                    value = attributes.getValue(A_NAME);
                    if (value != null) {
                        a.addAttribute(StyleConstants.FontFamily, value);
                    }

                    value = attributes.getValue(A_SIZE);
                    if (value != null) {
                        try {
                            a.addAttribute(StyleConstants.FontSize, Integer.decode(value));
                        } catch (NumberFormatException ex) {
                            LOG.log(Level.WARNING, value + " is not a valid Integer; parsing attribute " + A_SIZE + //NOI18N
                                getProcessedFile().getPath(), ex);
                        }
                    }
                    
                    value = attributes.getValue(A_STYLE);
                    if (value != null) {
                        a.addAttribute(StyleConstants.Bold,
                            Boolean.valueOf(value.indexOf(V_BOLD) >= 0)
                        );
                        a.addAttribute(
                            StyleConstants.Italic,
                            Boolean.valueOf(value.indexOf(V_ITALIC) >= 0)
                        );
                    }
                }
            } catch (Exception ex) {
                LOG.log(Level.WARNING, "Can't parse colorings file " + getProcessedFile().getPath(), ex); //NOI18N
            }
        }
    } // End of ColoringsReader class

    // delete ..........................................................
    
    public static void deleteColorings(
        MimePath mimePath,
        String profile,
        final boolean colorings, // true for colorings, false for highlightings
        boolean defaults   // delete default values
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        assert profile != null : "The parameter profile must not be null"; //NOI18N
        
        FileSystem sfs = Repository.getDefault().getDefaultFileSystem();
        FileObject baseFolder = sfs.findResource("Editors"); //NOI18N
        Map<String, List<Object []>> files = new HashMap<String, List<Object []>>();
        SettingsType.FONTSCOLORS.getLocator().scan(baseFolder, mimePath.getPath(), profile, true, defaults, !defaults, files);
        
        assert files.size() <= 1 : "Too many results in the scan"; //NOI18N

        final List<Object []> profileInfos = files.get(profile);
        if (profileInfos != null) {
            try {
                sfs.runAtomicAction(new FileSystem.AtomicAction() {
                    public void run() {
                        for(Object [] info : profileInfos) {
                            FileObject settingFile = (FileObject) info[1];

                            // Skip files with wrong type of colorings
                            boolean isTokenColoringFile = isTokenColoringFile(settingFile);
                            if (isTokenColoringFile != colorings) {
                                continue;
                            }

                            try {
                                settingFile.delete();
                            } catch (IOException ioe) {
                                LOG.log(Level.WARNING, "Can't delete editor settings file " + settingFile.getPath(), ioe); //NOI18N
                            }
                        }
                    }
                });
            } catch (IOException ioe) {
                LOG.log(Level.WARNING, "Can't delete editor colorings for " + mimePath.getPath() + ", " + profile, ioe); //NOI18N
            }
        }
    }
    
    // save ..........................................................
    
    public static void saveColorings(
        MimePath mimePath,
        String profile,
        final boolean colorings, // true for colorings, false for highlightings
        boolean defaults,  // save default values
        final Collection<AttributeSet> fontColors
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        assert profile != null : "The parameter profile must not be null"; //NOI18N
        
        final FileSystem sfs = Repository.getDefault().getDefaultFileSystem();
        final String settingFileName = SettingsType.FONTSCOLORS.getLocator().getWritableFileName(
                mimePath.getPath(), 
                profile, 
                colorings ? "-tokenColorings" : "-highlights", //NOI18N
                defaults);

        try {
            sfs.runAtomicAction(new FileSystem.AtomicAction() {
                public void run() throws IOException {
                    FileObject baseFolder = sfs.findResource("Editors"); //NOI18N
                    FileObject f = FileUtil.createData(baseFolder, settingFileName);
                    f.setAttribute(FA_TYPE, colorings ? FAV_TOKEN : FAV_HIGHLIGHT);
                    saveColorings(f, fontColors);
                }
            });
        } catch (IOException ioe) {
            LOG.log(Level.WARNING, "Can't save editor colorings for " + mimePath.getPath() + ", " + profile, ioe); //NOI18N
        }
    }
    
    private static void saveColorings(FileObject fo, Collection<AttributeSet> colorings) {
        Document doc = XMLUtil.createDocument(E_ROOT, null, PUBLIC_ID, SYSTEM_ID);
        Node root = doc.getElementsByTagName(E_ROOT).item(0);
        
        for(AttributeSet category : colorings) {
            Element fontColor = doc.createElement(E_FONTCOLOR);
            root.appendChild(fontColor);
            fontColor.setAttribute(A_NAME, (String) category.getAttribute(StyleConstants.NameAttribute));
            
            if (category.isDefined(StyleConstants.Foreground)) {
                fontColor.setAttribute(
                    A_FOREGROUND, 
                    Utils.colorToString((Color) category.getAttribute(StyleConstants.Foreground))
                );
            }
            if (category.isDefined(StyleConstants.Background)) {
                fontColor.setAttribute(
                    A_BACKGROUND,
                    Utils.colorToString((Color) category.getAttribute(StyleConstants.Background))
                );
            }
            if (category.isDefined(StyleConstants.StrikeThrough)) {
                fontColor.setAttribute(
                    A_STRIKETHROUGH,
                    Utils.colorToString((Color) category.getAttribute(StyleConstants.StrikeThrough))
                );
            }
            if (category.isDefined(EditorStyleConstants.WaveUnderlineColor)) {
                fontColor.setAttribute(
                    A_WAVEUNDERLINE,
                    Utils.colorToString((Color) category.getAttribute(EditorStyleConstants.WaveUnderlineColor))
                );
            }
            if (category.isDefined(StyleConstants.Underline)) {
                fontColor.setAttribute(
                    A_UNDERLINE,
                    Utils.colorToString((Color) category.getAttribute(StyleConstants.Underline))
                );
            }
            if (category.isDefined(EditorStyleConstants.Default)) {
                fontColor.setAttribute(
                    A_DEFAULT,
                    (String) category.getAttribute(EditorStyleConstants.Default)
                );
            }
            
            if ( category.isDefined(StyleConstants.FontFamily) ||
                 category.isDefined(StyleConstants.FontSize) ||
                 category.isDefined(StyleConstants.Bold) ||
                 category.isDefined(StyleConstants.Italic)
            ) {
                Element font = doc.createElement(E_FONT);
                fontColor.appendChild(font);
                
                if (category.isDefined(StyleConstants.FontFamily)) {
                    font.setAttribute(
                        A_NAME,
                        (String) category.getAttribute(StyleConstants.FontFamily)
                    );
                }
                if (category.isDefined(StyleConstants.FontSize)) {
                    font.setAttribute(
                        A_SIZE,
                        ((Integer) category.getAttribute(StyleConstants.FontSize)).toString()
                    );
                }
                if (category.isDefined(StyleConstants.Bold) ||
                    category.isDefined(StyleConstants.Italic)
                ) {
                    Boolean bold = Boolean.FALSE, italic = Boolean.FALSE;
                    
                    if (category.isDefined(StyleConstants.Bold)) {
                        bold = (Boolean) category.getAttribute(StyleConstants.Bold);
                    }
                    if (category.isDefined(StyleConstants.Italic)) {
                        italic = (Boolean) category.getAttribute(StyleConstants.Italic);
                    }
                    
                    font.setAttribute(A_STYLE, bold.booleanValue() ?
                        (italic.booleanValue() ? V_BOLD_ITALIC : V_BOLD) :
                        (italic.booleanValue() ? V_ITALIC : V_PLAIN)
                    );
                }
            }
        }

        XMLStorage.save(fo, doc);
    }

    private static boolean isTokenColoringFile(FileObject f) {
        Object typeValue = f.getAttribute(FA_TYPE);
        if (typeValue instanceof String) {
            return typeValue.equals(FAV_TOKEN);
        } else {
            return !f.getNameExt().equals(HIGHLIGHTING_FILE_NAME);
        }
    }
}
