/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.ejb.entity;

import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.*;

import org.apache.commons.logging.Log;

import xjavadoc.*;
import xdoclet.DocletContext;

import xdoclet.DocletTask;
import xdoclet.XDocletException;
import xdoclet.modules.ejb.EjbTagsHandler;
import xdoclet.modules.ejb.XDocletModulesEjbMessages;
import xdoclet.modules.ejb.entity.ValueObjectSubTask;
import xdoclet.modules.ejb.intf.InterfaceTagsHandler;
import xdoclet.tagshandler.MethodTagsHandler;
import xdoclet.util.LogUtil;
import xdoclet.util.Translator;

import xdoclet.util.TypeConversionUtil;

/**
 * Tags used by the Value Object templates.
 *
 * @author               Vincent Harcq (vincent.harcq@hubmethods.com)
 * @created              13. juni 2002
 * @xdoclet.taghandler   namespace="EjbValueObj"
 * @version              $Revision: 1.28 $
 */
public class ValueObjectTagsHandler
     extends EjbTagsHandler
{
    private XTag    currentTag;

    private String  currentValueObjectClass;

    private String  currentValueObjectAttribute;

    private String  currentValueObjectMatch;

    // For aggregation
    private String  currentAggregateType;

    private String  currentAggregateName;
    private String  currentAggregateNamePlural;

    private String  currentRelationBeanClass;

    /**
     * Checks if a method is a value object relation (aggregate or compose) matching a certain valueObject
     *
     * @param method
     * @param valueObject
     * @return
     */
    public static boolean isValueObjectRelation(XMethod method, String valueObject)
    {
        Log log = LogUtil.getLog(ValueObjectTagsHandler.class, "isValueObjectRelation");
        boolean ret = method.getDoc().hasTag("ejb:value-object");

        if (log.isDebugEnabled())
            log.debug(method.getName() + " has a ejb:value-object Tag " + ret);
        if (ret) {
            Collection tags = method.getDoc().getTags("ejb:value-object");

            if (tags.size() == 0 && !"*".equals(valueObject)) {
                ret = false;
            }
            else {
                ret = false;

                boolean excluded = true;

                for (Iterator i = tags.iterator(); i.hasNext(); ) {
                    XTag tag = (XTag) i.next();
                    String exclude = tag.getAttributeValue("exclude");
                    String aggreg = tag.getAttributeValue("aggregate");
                    String comp = tag.getAttributeValue("compose");

                    if ("true".equals(exclude) || (aggreg == null && comp == null)) {
                        excluded = true;
                        ret = false;
                        break;
                    }

                    ret = matches(tag, valueObject);
                    if (ret)
                        break;
                }
                if ("*".equals(valueObject) && !excluded) {
                    ret = true;
                }
            }
        }
        return ret;
    }

    /**
     * Gets the GenerationNeeded attribute of the ValueObjectTagsHandler class
     *
     * @param clazz  Describe what the parameter does
     * @return       The GenerationNeeded value
     */
    public static boolean isGenerationNeeded(XClass clazz)
    {
        return true;
        //TODO
    }

    /**
     * Gets the CurrentValueObjectClass attribute of the ValueObjectTagsHandler class
     *
     * @param clazz                 Describe what the parameter does
     * @param tag                   Describe what the parameter does
     * @return                      The CurrentValueObjectClass value
     * @exception XDocletException
     */
    public static String getCurrentValueObjectClass(XClass clazz, XTag tag)
         throws XDocletException
    {
        String name = getCurrentValueObjectName(tag);

        String _currentValueObjectClass;

        _currentValueObjectClass = MessageFormat.format(getSubTask().getValueObjectClassPattern(),
            new Object[]{(name != null) ? name : getShortEjbNameFor(clazz)});

        String packageName = clazz.getContainingPackage().getName();

        packageName = choosePackage(packageName, null, DocletTask.getSubTaskName(ValueObjectSubTask.class));

        _currentValueObjectClass = packageName + '.' + _currentValueObjectClass;

        return _currentValueObjectClass;
    }

    /**
     * Gets the CurrentValueObjectName attribute of the ValueObjectTagsHandler class
     *
     * @param tag                   Describe what the parameter does
     * @return                      The CurrentValueObjectName value
     * @exception XDocletException
     */
    public static String getCurrentValueObjectName(XTag tag) throws XDocletException
    {
        String name = tag.getAttributeValue("name");

        if (name != null) {
            return name;
        }

        // name is null, must look whether it's defined in current class level
        XClass clazz = getCurrentClass();

        while (clazz != null) {
            for (Iterator i = clazz.getDoc().getTags(tag.getName()).iterator(); i.hasNext(); ) {
                if (tag.equals(i.next())) {
                    // ok, we are defined here return defined ejb name
                    name = EjbTagsHandler.getShortEjbNameFor(clazz);
                    if (name == null) {
                        throw new XDocletException("unable to determine value object name in class " + clazz.getName());
                    }
                    return name;
                }
            }
            // not found on current level. try with superclass
            clazz = clazz.getSuperclass();
        }
        throw new XDocletException("class defining value object is not EJB");
    }

    /**
     * Gets the CurrentValueObjectAttribute attribute of the ValueObjectTagsHandler class
     *
     * @param tag                   Describe what the parameter does
     * @return                      The CurrentValueObjectAttribute value
     * @exception XDocletException
     */
    public static String getCurrentValueObjectAttribute(XTag tag) throws XDocletException
    {
        String name = getCurrentValueObjectName(tag);

        String _currentValueObjectAttribute;

        if (name == null) {
            _currentValueObjectAttribute = "Value";
        }
        else {
            _currentValueObjectAttribute = MessageFormat.format(getSubTask().getValueObjectClassPattern(), new Object[]{name});
        }

        return _currentValueObjectAttribute;
    }

    /**
     * Gets the CurrentValueObjectMatch attribute of the ValueObjectTagsHandler class
     *
     * @param tag  Describe what the parameter does
     * @return     The CurrentValueObjectMatch value
     */
    public static String getCurrentValueObjectMatch(XTag tag)
    {
        String match = tag.getAttributeValue("match");

        if (match == null) {
            match = "*";
        }
        return match;
    }

    public static String getCurrentValueObjectImplements(XTag tag)
    {
        String toImplement = tag.getAttributeValue("implements");

        return toImplement != null ? "," + toImplement : "";
    }

    public static String getCurrentValueObjectExtends(XTag tag)
    {
        String toImplement = tag.getAttributeValue("extends");

        return toImplement != null ? toImplement : "java.lang.Object";
    }

    /**
     * Gets a reference to the &lt;valueobject&gt; subtask.
     *
     * @return   subtask
     */
    private static ValueObjectSubTask getSubTask()
    {
        ValueObjectSubTask subtask = ((ValueObjectSubTask) DocletContext.getInstance().getSubTaskBy(DocletTask.getSubTaskName(ValueObjectSubTask.class)));

        return subtask;
    }

    /**
     * Returns whether the passed tag matches for the passed value object
     *
     * @param tag          the tag
     * @param valueObject  the value-object's match
     * @return             <code>true</code> if the passed tag matches
     */
    private static boolean matches(XTag tag, String valueObject)
    {
        Log log = LogUtil.getLog(ValueObjectTagsHandler.class, "matches");
        String value = tag.getAttributeValue("match");

        if (log.isDebugEnabled()) {
            log.debug("Match=" + value + "==" + valueObject);
        }

        if (value == null) {
            return "*".equals(valueObject);
        }
        return value.equals(valueObject) || value.equals("*") || "*".equals(valueObject);
    }

    /**
     * Whether or not the <code>abstract</code> parameter is set to true on the supplied class' ejb.value-object tag for
     * the given VO name.
     *
     * @param valueObjectName
     * @param currentClass
     * @return
     * @exception XDocletException
     */
    public boolean isAbstractValueObject(String valueObjectName,
        XClass currentClass)
         throws XDocletException
    {
        boolean isAbstract = false;

        Collection valueObjectTags =
            currentClass.getDoc().getTags("ejb:value-object");

        for (Iterator i = valueObjectTags.iterator();
            i.hasNext() && !isAbstract; ) {
            XTag tag = (XTag) i.next();

            isAbstract = isAbstractValueObject(valueObjectName, tag);
        }

        return isAbstract;
    }

    /**
     * @param clazz                 Description of Parameter
     * @return                      the full qualified data-object class name
     * @exception XDocletException
     */
    public String getValueMostSuperObjectClass(XClass clazz) throws XDocletException
    {
        String currentDataClass = currentValueObjectClass;
        // Begin at the first super class

        XClass cur_clazz = clazz.getSuperclass();

        do {
            // Find if we have an abstract data class definition to generate
            Collection methods = cur_clazz.getMethods();
            boolean found = false;

            for (Iterator j = methods.iterator(); j.hasNext(); ) {
                XMethod method = (XMethod) j.next();

                if (method.getName().equals("get" + currentValueObjectAttribute)) {
                    found = true;
                    currentDataClass = getCurrentValueObjectClass(cur_clazz, currentTag);
                }

                if (found) {
                    break;
                }
            }
            cur_clazz = cur_clazz.getSuperclass();
        } while (cur_clazz != null);

        return currentDataClass;
    }

    /**
     * Evaluates the body if the <code>valueobject</code> subtask's <code>generatePKConstructor</code> parameter is
     * <code>true</code>.
     *
     * @param template              The body of the block tag
     * @param attributes            The attributes of the template tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifGeneratePKConstructor(String template, Properties attributes)
         throws XDocletException
    {
        if (getSubTask().getGeneratePKConstructor()) {
            generate(template);
        }
    }

    /**
     * Evaluates the body if the <code>abstract</code> parameter is set to true on the ejb.value-object tag for the
     * current VO.
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifIsAbstractValueObject(String template)
         throws XDocletException
    {
        if (isAbstractValueObject(valueObjectName(), getCurrentClass())) {
            generate(template);
        }
    }

    /**
     * Evaluates the body if the <code>abstract</code> parameter is set to false (or is missing) on the ejb.value-object
     * tag for the current VO.
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifNotIsAbstractValueObject(String template)
         throws XDocletException
    {
        if (!isAbstractValueObject(valueObjectName(), getCurrentClass())) {
            generate(template);
        }
    }

    /**
     * Return the current value object's class name.
     *
     * @return                      class name
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String valueObjectClass() throws XDocletException
    {
        return getSubTask().getCurrentValueObjectClass();
    }

    /**
     * Return the current value object's name.
     *
     * @return                      VO name
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String valueObjectName() throws XDocletException
    {
        return getSubTask().getCurrentValueObjectName();
    }

    /**
     * Return the current value object's match parameter.
     *
     * @return                      match value
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String valueObjectMatch() throws XDocletException
    {
        return getSubTask().getCurrentValueObjectMatch();
    }

    /**
     * Returns the name of the class the specified value object extends. If no <code>extends</code> parameter exists on
     * the <code>ejb.value-object</code> tag, <code>java.lang.Object</code> is returned.
     *
     * @param attributes            The attributes of the template tag
     * @return                      The name of generated PK class.
     * @exception XDocletException
     * @doc.tag                     type="content"
     * @doc.param                   name="valueobject" optional="false" description="The name of the value object to
     *      check."
     */
    public String extendsFrom(Properties attributes) throws XDocletException
    {
        return getSubTask().getCurrentValueObjectExtends();
    }

    /**
     * Loops over all the ejb.value-object tags in the class, and generates the body for each one.
     *
     * @param pTemplate             The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void forAllValueObjects(String pTemplate) throws XDocletException
    {
        Log log = LogUtil.getLog(ValueObjectTagsHandler.class, "forAllValueObjects");
        Collection dos = getCurrentClass().getDoc().getTags("ejb:value-object", true);

        if (log.isDebugEnabled()) {
            log.debug("Number of tags ejb:value-object in " + getCurrentClass() + " = " + dos.size());
        }
        for (Iterator i = dos.iterator(); i.hasNext(); ) {
            currentTag = (XTag) i.next();

            if (!isAbstractValueObject(getCurrentValueObjectName(currentTag),
                currentTag)) {

                currentValueObjectClass = getCurrentValueObjectClass(getCurrentClass(), currentTag);
                currentValueObjectAttribute = getCurrentValueObjectAttribute(currentTag);
                currentValueObjectMatch = getCurrentValueObjectMatch(currentTag);
                if (log.isDebugEnabled()) {
                    log.debug("Generate for " + currentValueObjectClass + " attr=" + currentValueObjectAttribute + " match=" + currentValueObjectMatch);
                }

                generate(pTemplate);
            }
        }
    }

    /**
     * Return the current value object's class name.
     *
     * @return    class name
     * @doc.tag   type="content"
     */
    public String currentValueObjectClass()
    {
        return currentValueObjectClass;
    }

    /**
     * Return the current value object's attribute name.
     *
     * @return    attribute
     * @doc.tag   type="content"
     */
    public String currentValueObjectAttribute()
    {
        return currentValueObjectAttribute;
    }

    /**
     * Return the current value object's match parameter.
     *
     * @return    match value
     * @doc.tag   type="content"
     */
    public String currentValueObjectMatch()
    {
        return currentValueObjectMatch;
    }

    /**
     * Returns the class name of the current aggregate attribute's type.
     *
     * @param attributes  The attributes of the template tag
     * @return            class
     * @doc.tag           type="content"
     * @doc.param         name="short" description="Use the short (not fully-qualified) class name."
     */
    public String currentAggregateType(Properties attributes)
    {
        String sh = attributes.getProperty("short");
        String ret = "";

        if (sh == null) {
            ret = currentAggregateType;
        }
        else {
            StringTokenizer st = new StringTokenizer(currentAggregateType, ".");

            while (st.hasMoreTokens()) {
                ret = st.nextToken();
            }
        }
        return ret;
    }

    /**
     * return interfaces to be implemented
     *
     * @return    interfaces
     * @doc.tag   type="content"
     */
    public String valueObjectImplements()
    {
        return getSubTask().getCurrentValueObjectImplements();
    }

    /**
     * Returns the current aggregate's name
     *
     * @param attributes
     * @return                   aggregate name
     * @throws XDocletException  if an error occures
     * @doc.param                name="plural" optional="true" values="true, false" default="false" descriptions="return
     *      the plural of the aggregate's name if set to true"
     * @doc.param                name="decapitalize" optional="true" values="true, false" default="false"
     *      descriptions="return the decapitalize aggregate's name if set to true"
     * @doc.tag                  type="content"
     */
    public String currentAggregateName(Properties attributes)
    {
        String plural = attributes.getProperty("plural");
        String decapitalize = attributes.getProperty("decapitalize");
        String name = "true".equals(plural) ? currentAggregateNamePlural : currentAggregateName;

        if ("true".equals(decapitalize)) {
            if (name != null && name.length() > 0) {
                name = Character.toLowerCase(name.charAt(0)) + name.substring(1);
            }
        }
        return name;
    }

    /**
     * Return the bean class name for the current relation.
     *
     * @return    class name
     * @doc.tag   type="content"
     */
    public String currentRelationBeanClass()
    {
        return currentRelationBeanClass;
    }

    /**
     * Type of the constructor for aggregates or compositions.
     *
     * @return                      Type of the constructor for aggregates or compositions.
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String concreteCollectionType() throws XDocletException
    {
        String concreteType = getCurrentMethod().getDoc().getTagAttributeValue("ejb.value-object", "concrete-type");
        Class concreteClass = null;

        // test the concrete class
        if (concreteType != null) {
            try {
                concreteClass = Class.forName(concreteType);
            }
            catch (ClassNotFoundException e) {
                throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class,
                    XDocletModulesEjbMessages.VALUE_OBJECT_CONCRETE_TYPE_NOT_FOUND,
                    new String[]{concreteType}));
            }
            if (concreteClass.isInterface()) {
                throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class,
                    XDocletModulesEjbMessages.VALUE_OBJECT_CONCRETE_TYPE_IS_INTF,
                    new String[]{concreteType}));
            }
            if (Modifier.isAbstract(concreteClass.getModifiers())) {
                throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class,
                    XDocletModulesEjbMessages.VALUE_OBJECT_CONCRETE_TYPE_IS_ABSTRACT,
                    new String[]{concreteType}));
            }
        }

        String currentReturnType = getCurrentMethod().getReturnType().getType().getQualifiedName();

        if (currentReturnType.equals("java.util.Collection")) {
            if (concreteClass == null) {
                return "java.util.ArrayList";
            }
            else {
                // verify that the concrete class is a Collection
                if (!Collection.class.isAssignableFrom(concreteClass)) {
                    throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class,
                        XDocletModulesEjbMessages.VALUE_OBJECT_CONCRETE_TYPE_INVALID,
                        new String[]{concreteType, "java.util.Collection"}));
                }
                return concreteType;
            }

        }
        else if (currentReturnType.equals("java.util.Set")) {
            if (concreteClass == null) {
                return "java.util.HashSet";
            }
            else {
                // verify that the concrete class is a Set
                if (!Set.class.isAssignableFrom(concreteClass)) {
                    throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class,
                        XDocletModulesEjbMessages.VALUE_OBJECT_CONCRETE_TYPE_INVALID,
                        new String[]{concreteType, "java.util.Set"}));
                }
                return concreteType;
            }
        }
        else {
            throw new XDocletException("Invalid return type (" +
                currentReturnType +
                " on aggregate or composition.");
        }
    }

    /**
     * Returns the collection type for the current field.
     *
     * @return                   the type
     * @throws XDocletException  if an error occures
     * @doc.tag                  type="content"
     */
    public String collectionType() throws XDocletException
    {
        return getCurrentMethod().getReturnType().getType().getQualifiedName();
    }

    /**
     * Executes the body only if the current field is a collection.
     *
     * @param template              the template
     * @exception XDocletException  if an error occures
     * @doc.tag                     type="block"
     */
    public void ifIsCollection(String template) throws XDocletException
    {
        if (isCollection()) {
            generate(template);
        }
    }

    /**
     * Executes the body only if the current field is not a collection.
     *
     * @param template              the template
     * @exception XDocletException  if an error occures
     * @doc.tag                     type="block"
     */
    public void ifIsNotCollection(String template) throws XDocletException
    {
        if (!isCollection()) {
            generate(template);
        }
    }

    /**
     * Returns the data-object class name highest in the hierarchy of derived beans. Because of possible inheritance
     * between entity bean, the type of the generated getData method must be the one of the most super class of the
     * current entity bean. The current Data class must extend the corresponding super Data class.
     *
     * @return                      The data-object class name highest in the hierarchy of derived beans.
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String valueMostSuperObjectClass() throws XDocletException
    {
        return getValueMostSuperObjectClass(getCurrentClass());
    }

    /**
     * Describe what the method does
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void forAllSuperSetValue(String template) throws XDocletException
    {
        XClass oldClass = getCurrentClass();

        XClass superclass = null;

        do {
            Collection dos = getCurrentClass().getDoc().getTags("ejb:value-object", false);

            for (Iterator i = dos.iterator(); i.hasNext(); ) {
                currentTag = (XTag) i.next();
                currentValueObjectClass = getCurrentValueObjectClass(getCurrentClass(), currentTag);
                currentValueObjectAttribute = getCurrentValueObjectAttribute(currentTag);
                currentValueObjectMatch = getCurrentValueObjectMatch(currentTag);

                forAllSetters(template, "set" + currentValueObjectAttribute);
            }

            superclass = getCurrentClass().getSuperclass();

            if (superclass != null) {
                pushCurrentClass(superclass);
            }
        } while (superclass != null);

        setCurrentClass(oldClass);

    }

    /**
     * Loop over all the aggregate fields in the given value object, and generate the body for each one.
     *
     * @param template              The body of the block tag
     * @param attributes            The attributes of the template tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     * @doc.param                   name="superclasses" values="true,false" description="Whether to include fields in
     *      superclasses."
     * @doc.param                   name="valueobject" optional="false" description="The value object name."
     */
    public void forAllAggregates(String template, Properties attributes) throws XDocletException
    {
        String superclasses_str = attributes.getProperty("superclasses");
        boolean superclasses = TypeConversionUtil.stringToBoolean(superclasses_str, true);

        String valueObject = attributes.getProperty("valueobject");

        forAllRelations(template, superclasses, valueObject, "aggregate");
    }

    /**
     * Loop over all the composed fields in the given value object, and generate the body for each one.
     *
     * @param template              The body of the block tag
     * @param attributes            The attributes of the template tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     * @doc.param                   name="superclasses" values="true,false" description="Whether to include fields in
     *      superclasses."
     * @doc.param                   name="valueobject" optional="false" description="The value object name."
     */
    public void forAllComposes(String template, Properties attributes) throws XDocletException
    {
        String superclasses_str = attributes.getProperty("superclasses");
        boolean superclasses = TypeConversionUtil.stringToBoolean(superclasses_str, true);

        String valueObject = attributes.getProperty("valueobject");

        forAllRelations(template, superclasses, valueObject, "compose");
    }

    /**
     * Loop over all the relation fields in the given value object, and generate the body for each one.
     *
     * @param template              The body of the block tag
     * @param attributes            The attributes of the template tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     * @doc.param                   name="superclasses" values="true,false" description="Whether to include fields in
     *      superclasses."
     * @doc.param                   name="valueobject" optional="false" description="The value object name."
     */
    public void forAllRelations(String template, Properties attributes) throws XDocletException
    {
        String superclasses_str = attributes.getProperty("superclasses");
        boolean superclasses = TypeConversionUtil.stringToBoolean(superclasses_str, true);

        String valueObject = attributes.getProperty("valueobject");

        forAllRelations(template, superclasses, valueObject, "aggregate");
        forAllRelations(template, superclasses, valueObject, "compose");
    }

    /**
     * Evaluate the body block if Value Object subtask being used.
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifUsingValueObject(String template) throws XDocletException
    {
        if (DocletContext.getInstance().isSubTaskDefined(DocletTask.getSubTaskName(ValueObjectSubTask.class))) {
            generate(template);
        }
    }

    protected void forAllSetters(String template, String methodName) throws XDocletException
    {
        Log log = LogUtil.getLog(ValueObjectTagsHandler.class, "forAllSetters");

        if (log.isDebugEnabled()) {
            log.debug(methodName);
        }

        // Find if we have an abstract data class definition to generate
        Collection methods = getCurrentClass().getMethods();
        XMethod methodFound = null;

        for (Iterator j = methods.iterator(); j.hasNext(); ) {
            XMethod method = (XMethod) j.next();

            if (method.getName().equals(methodName)) {
                methodFound = method;
            }

            if (methodFound != null) {
                if (log.isDebugEnabled()) {
                    log.debug(methodFound);
                }
                break;
            }
        }

        if (methodFound != null) {
            generate(template);
        }

    }

    /**
     * Returns whether the current field is a collection
     *
     * @return   <code>true</code> it the current field is a collection
     */
    private boolean isCollection()
    {
        String currentReturnType = getCurrentMethod().getReturnType().getType().getQualifiedName();

        return "java.util.Collection".equals(currentReturnType) || "java.util.Set".equals(currentReturnType);
    }

    private boolean isAbstractValueObject(String valueObjectName, XTag tag)
    {
        boolean isAbstract = false;
        String attr = tag.getAttributeValue("abstract");

        if (valueObjectName.equals(tag.getAttributeValue("name")) &&
            attr != null) {
            isAbstract = TypeConversionUtil.stringToBoolean(attr, false);
        }
        return isAbstract;
    }

    /**
     * @param template              Describe what the parameter does
     * @param superclasses          Describe what the parameter does
     * @param valueObject           Describe what the parameter does
     * @param type                  Describe what the parameter does
     * @exception XDocletException
     * @todo                        (Aslak) use a HashSet instead of HashMap for foundFields
     */
    private void forAllRelations(String template, boolean superclasses, String valueObject, String type) throws XDocletException
    {
        Log log = LogUtil.getLog(ValueObjectTagsHandler.class, "forAllRelations");
        Map foundFields = new HashMap();
        XClass currentClass = getCurrentClass();
        XMethod oldMethod = getCurrentMethod();

        if (log.isDebugEnabled())
            log.debug("*** forAllRelations on class=" + currentClass.getName() + " valueobject=" + valueObject);
        do {
            pushCurrentClass(currentClass);

            if (log.isDebugEnabled()) {
                log.debug("****** CurrentClass=" + getCurrentClass());
            }

            Collection methods = getCurrentClass().getMethods();

            for (Iterator j = methods.iterator(); j.hasNext(); ) {
                XMethod method = (XMethod) j.next();

                setCurrentMethod(method);

                if (MethodTagsHandler.isGetter(getCurrentMethod().getName()) &&
                    !foundFields.containsKey(getCurrentMethod().getName()) &&
                    isValueObjectRelation(getCurrentMethod(), valueObject)) {

                    boolean ret = getCurrentMethod().getDoc().hasTag("ejb:value-object");

                    if (log.isDebugEnabled())
                        log.debug("****** Method " + getCurrentMethod().getName() + " has VO tag : " + ret);
                    if (ret) {
                        Collection tags = getCurrentMethod().getDoc().getTags("ejb:value-object");

                        for (Iterator i = tags.iterator(); i.hasNext(); ) {
                            XTag tag = (XTag) i.next();

                            if (!matches(tag, valueObject)) {
                                continue;
                            }

                            String aggreg = tag.getAttributeValue(type);
                            String aggregName = tag.getAttributeValue(type + "-name");
                            String aggregNamePlural = tag.getAttributeValue(type + "s-name");

                            if (aggregNamePlural == null) {
                                aggregNamePlural = aggregName + "s";
                            }

                            if (log.isDebugEnabled()) {
                                log.debug("********* " + method.getName() + " ejb:value-object Tag - Type = " + type + " - Value = " + aggreg + " - Name = " + aggregName);
                            }

                            if (aggreg != null) {
                                String currentReturnType = getCurrentMethod().getReturnType().getType().getQualifiedName();

                                if (log.isDebugEnabled()) {
                                    log.debug("********* METHOD=" + getCurrentMethod().getName() + " " + currentReturnType);
                                }

                                // Store that we found this field so we don't add it twice
                                foundFields.put(getCurrentMethod().getName(), getCurrentMethod().getName());
                                if (currentReturnType.equals("java.util.Collection") ||
                                    currentReturnType.equals("java.util.Set")) {

                                    if (log.isDebugEnabled()) {
                                        log.debug("********* Type Collection or Set");
                                    }
                                    currentAggregateType = aggreg;
                                    currentAggregateName = aggregName;
                                    currentAggregateNamePlural = aggregNamePlural;

                                    String relationInterface = tag.getAttributeValue("members");

                                    if (log.isDebugEnabled()) {
                                        log.debug(relationInterface);
                                    }
                                    if (relationInterface != null && !relationInterface.equals("")) {
                                        currentRelationBeanClass = InterfaceTagsHandler.getBeanClassNameFromInterfaceNameFor(relationInterface);
                                    }
                                }
                                else {
                                    if (log.isDebugEnabled()) {
                                        log.debug("********* Type " + getCurrentMethod().getReturnType().getType().toString());
                                    }
                                    currentAggregateType = aggreg;
                                    currentAggregateName = aggregName;
                                    currentAggregateNamePlural = aggregNamePlural;
                                    currentRelationBeanClass = InterfaceTagsHandler.getBeanClassNameFromInterfaceNameFor(getCurrentMethod().getReturnType().getType().getQualifiedName());
                                }
                                generate(template);
                                currentAggregateType = null;
                                currentAggregateName = null;
                                currentAggregateNamePlural = null;
                                currentRelationBeanClass = null;
                            }
                        }
                    }
                }
            }

            // Add super class info
            XClass superclass = getCurrentClass().getSuperclass();

            if (superclass == null) {
                popCurrentClass();
                break;
            }

            if (superclass.getQualifiedName().equals("java.lang.Object")) {
                popCurrentClass();
                break;
            }

            popCurrentClass();

            // superclasses is true for *CMP/BMP/Session
            if (superclasses == true) {
                currentClass = currentClass.getSuperclass();
            }
            else {
                if (shouldTraverseSuperclassForDependentClass(currentClass.getSuperclass(), getDependentClassTagName())) {
                    currentClass = currentClass.getSuperclass();
                }
                else {
                    break;
                }
            }
        } while (true);

        setCurrentMethod(oldMethod);

        log.debug("Finished.");
    }
}
