# Copyright 2008 by Bartek Wilczynski.  All rights reserved.
# Adapted from test_Mymodule.py by Jeff Chang
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.

import os
import unittest

from Bio import Motif


class MotifTestsBasic(unittest.TestCase):
    def setUp(self):
        self.PFMin = open("Motif/SRF.pfm")
        self.SITESin = open("Motif/Arnt.sites")
        self.TFout = "Motif/tf.out"
        self.FAout = "Motif/fa.out"
        self.PFMout = "Motif/fa.out"
        from Bio.Seq import Seq
        self.m=Motif.Motif()
        self.m.add_instance(Seq("ATATA",self.m.alphabet))
        
    def tearDown(self):
        self.PFMin.close()
        self.SITESin.close()
        if os.path.exists(self.TFout):
            os.remove(self.TFout)
        if os.path.exists(self.FAout):
            os.remove(self.FAout)

    def test_alignace_parsing(self):
        """Test if Motif can parse AlignAce output files.
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import AlignAce
        handle = open("Motif/alignace.out")
        record = AlignAce.read(handle)
        handle.close()
        self.assertEqual(record.ver, "AlignACE 4.0 05/13/04\n")
        self.assertEqual(record.cmd_line, "./AlignACE -i test.fa \n")
        self.assertEqual(len(record.param_dict), 7)
        self.assertEqual(record.param_dict['expect'], "10")
        self.assertEqual(record.param_dict['gcback'], "0.38")
        self.assertEqual(record.param_dict['minpass'], "200")
        self.assertEqual(record.param_dict['seed'], "1227623309")
        self.assertEqual(record.param_dict['numcols'], "10")
        self.assertEqual(record.param_dict['undersample'], "1")
        self.assertEqual(record.param_dict['oversample'], "1")
        self.assertEqual(len(record.seq_dict), 10)
        self.assertEqual(record.seq_dict[0], "SEQ1; M: CTCAATCGTAGA at 52\n")
        self.assertEqual(record.seq_dict[1], "SEQ2; M: CTCAATCGTAGA at 172\n")
        self.assertEqual(record.seq_dict[2], "SEQ3; M: CTCAATCGTAGA at 112\n")
        self.assertEqual(record.seq_dict[3], "SEQ4; M: CTCAATCGTAGA at 173\n")
        self.assertEqual(record.seq_dict[4], "SEQ5; M: CTCAATCGTAGA at 185\n")
        self.assertEqual(record.seq_dict[5], "SEQ6; M: CTCAATCGTAGA at 105\n")
        self.assertEqual(record.seq_dict[6], "SEQ7; M: CTCAATCGTAGA at 177\n")
        self.assertEqual(record.seq_dict[7], "SEQ8; M: CTCAATCGTAGA at 172\n")
        self.assertEqual(record.seq_dict[8], "SEQ9; M: CTCAATCGTAGA at 93\n")
        self.assertEqual(record.seq_dict[9], "SEQ10; M: CTCAATCGTAGA at 3\n")
        self.assertEqual(len(record.motifs), 16)
        self.assertEqual(record.motifs[0].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[0].instances), 11)
        self.assertEqual(record.motifs[0].instances[0].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[1].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[2].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[3].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[4].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[5].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[6].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[7].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[8].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].instances[9].tostring(), "TCAAAGATAGAG")
        self.assertEqual(record.motifs[0].instances[10].tostring(), "TCTACGATTGAG")
        self.assertEqual(record.motifs[0].mask, [1,1,0,1,1,1,1,1,0,1,1,1])
        self.assertAlmostEqual(record.motifs[0].score, 57.9079)
        self.assertEqual(record.motifs[1].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[1].instances), 22)
        self.assertEqual(record.motifs[1].instances[0].tostring(), "GCGAAGGAAGCAGCGCGTGTG")
        self.assertEqual(record.motifs[1].instances[1].tostring(), "GGCACCGCCTCTACGATTGAG")
        self.assertEqual(record.motifs[1].instances[2].tostring(), "CAGAGCTTAGCATTGAACGCG")
        self.assertEqual(record.motifs[1].instances[3].tostring(), "CTAATGAAAGCAATGAGAGTG")
        self.assertEqual(record.motifs[1].instances[4].tostring(), "CTTGTGCCCTCTAAGCGTCCG")
        self.assertEqual(record.motifs[1].instances[5].tostring(), "GAGCACGACGCTTTGTACCTG")
        self.assertEqual(record.motifs[1].instances[6].tostring(), "CGGCACTTAGCAGCGTATCGT")
        self.assertEqual(record.motifs[1].instances[7].tostring(), "CTGGTTTCATCTACGATTGAG")
        self.assertEqual(record.motifs[1].instances[8].tostring(), "GGGCCAATAGCGGCGCCGGAG")
        self.assertEqual(record.motifs[1].instances[9].tostring(), "GTGGAGTTATCTTAGTGCGCG")
        self.assertEqual(record.motifs[1].instances[10].tostring(), "GAGAGGTTATCTACGATTGAG")
        self.assertEqual(record.motifs[1].instances[11].tostring(), "CTGCTCCCCGCATACAGCGCG")
        self.assertEqual(record.motifs[1].instances[12].tostring(), "CAGAACCGAGGTCCGGTACGG")
        self.assertEqual(record.motifs[1].instances[13].tostring(), "GTGCCCCAAGCTTACCCAGGG")
        self.assertEqual(record.motifs[1].instances[14].tostring(), "CGCCTCTGATCTACGATTGAG")
        self.assertEqual(record.motifs[1].instances[15].tostring(), "GTGCTCATAGGGACGTCGCGG")
        self.assertEqual(record.motifs[1].instances[16].tostring(), "CTGCCCCCCGCATAGTAGGGG")
        self.assertEqual(record.motifs[1].instances[17].tostring(), "GTAAAGAAATCGATGTGCCAG")
        self.assertEqual(record.motifs[1].instances[18].tostring(), "CACCTGCAATTGCTGGCAGCG")
        self.assertEqual(record.motifs[1].instances[19].tostring(), "GGCGGGCCATCCCTGTATGAA")
        self.assertEqual(record.motifs[1].instances[20].tostring(), "CTCCAGGTCGCATGGAGAGAG")
        self.assertEqual(record.motifs[1].instances[21].tostring(), "CCTCGGATCGCTTGGGAAGAG")
        self.assertEqual(record.motifs[1].mask, [1,0,1,1,0,1,0,0,1,1,1,0,0,0,1,0,0,0,1,0,1])
        self.assertAlmostEqual(record.motifs[1].score, 19.6235)

        self.assertEqual(record.motifs[2].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[2].instances), 18)
        self.assertEqual(record.motifs[2].instances[0].tostring(), "GTGCGCGAAGGAAGCAGCGCG")
        self.assertEqual(record.motifs[2].instances[1].tostring(), "CAGAGCTTAGCATTGAACGCG")
        self.assertEqual(record.motifs[2].instances[2].tostring(), "GTGCCCGATGACCACCCGTCG")
        self.assertEqual(record.motifs[2].instances[3].tostring(), "GCCCTCTAAGCGTCCGCGGAT")
        self.assertEqual(record.motifs[2].instances[4].tostring(), "GAGCACGACGCTTTGTACCTG")
        self.assertEqual(record.motifs[2].instances[5].tostring(), "CGGCACTTAGCAGCGTATCGT")
        self.assertEqual(record.motifs[2].instances[6].tostring(), "GGGCCAATAGCGGCGCCGGAG")
        self.assertEqual(record.motifs[2].instances[7].tostring(), "GCGCACTAAGATAACTCCACG")
        self.assertEqual(record.motifs[2].instances[8].tostring(), "CGGCCCGTTGTCCAGCAGACG")
        self.assertEqual(record.motifs[2].instances[9].tostring(), "CTGCTCCCCGCATACAGCGCG")
        self.assertEqual(record.motifs[2].instances[10].tostring(), "GTGCCCCAAGCTTACCCAGGG")
        self.assertEqual(record.motifs[2].instances[11].tostring(), "GTGCTCATAGGGACGTCGCGG")
        self.assertEqual(record.motifs[2].instances[12].tostring(), "CTGCCCCCCGCATAGTAGGGG")
        self.assertEqual(record.motifs[2].instances[13].tostring(), "CGCCGCCATGCGACGCAGAGG")
        self.assertEqual(record.motifs[2].instances[14].tostring(), "AACCTCTAAGCATACTCTACG")
        self.assertEqual(record.motifs[2].instances[15].tostring(), "GACCTGGAGGCTTAGACTTGG")
        self.assertEqual(record.motifs[2].instances[16].tostring(), "GCGCTCTTCCCAAGCGATCCG")
        self.assertEqual(record.motifs[2].instances[17].tostring(), "GGGCCGTCAGCTCTCAAGTCT")
        self.assertEqual(record.motifs[2].mask, [1,0,1,1,0,1,0,0,0,1,1,0,0,0,1,0,1,0,0,1,1])
        self.assertAlmostEqual(record.motifs[2].score, 19.1804)

        self.assertEqual(record.motifs[3].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[3].instances), 16)
        self.assertEqual(record.motifs[3].instances[0].tostring(), "GCCCCAAGCTTACCCAGGGAC")
        self.assertEqual(record.motifs[3].instances[1].tostring(), "GCCGTCTGCTGGACAACGGGC")
        self.assertEqual(record.motifs[3].instances[2].tostring(), "GCCGACGGGTGGTCATCGGGC")
        self.assertEqual(record.motifs[3].instances[3].tostring(), "GCCAATAGCGGCGCCGGAGTC")
        self.assertEqual(record.motifs[3].instances[4].tostring(), "GCCCCCCGCATAGTAGGGGGA")
        self.assertEqual(record.motifs[3].instances[5].tostring(), "GCCCGTACCGGACCTCGGTTC")
        self.assertEqual(record.motifs[3].instances[6].tostring(), "GCCTCATGTACCGGAAGGGAC")
        self.assertEqual(record.motifs[3].instances[7].tostring(), "GACACGCGCCTGGGAGGGTTC")
        self.assertEqual(record.motifs[3].instances[8].tostring(), "GCCTTTGGCCTTGGATGAGAA")
        self.assertEqual(record.motifs[3].instances[9].tostring(), "GGCCCTCGGATCGCTTGGGAA")
        self.assertEqual(record.motifs[3].instances[10].tostring(), "GCATGTTGGGAATCCGCGGAC")
        self.assertEqual(record.motifs[3].instances[11].tostring(), "GACACGCGCTGTATGCGGGGA")
        self.assertEqual(record.motifs[3].instances[12].tostring(), "GCCAGGTACAAAGCGTCGTGC")
        self.assertEqual(record.motifs[3].instances[13].tostring(), "GCGATCAGCTTGTGGGCGTGC")
        self.assertEqual(record.motifs[3].instances[14].tostring(), "GACAAATCGGATACTGGGGCA")
        self.assertEqual(record.motifs[3].instances[15].tostring(), "GCACTTAGCAGCGTATCGTTA")
        self.assertEqual(record.motifs[3].mask, [1,1,1,0,0,0,0,1,1,0,0,0,0,1,0,0,1,1,1,0,1])
        self.assertAlmostEqual(record.motifs[3].score, 18.0097)
        self.assertEqual(record.motifs[4].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[4].instances), 15)
        self.assertEqual(record.motifs[4].instances[0].tostring(), "CGGCACAGAGCTT")
        self.assertEqual(record.motifs[4].instances[1].tostring(), "ATCCGCGGACGCT")
        self.assertEqual(record.motifs[4].instances[2].tostring(), "CGCCTGGGAGGGT")
        self.assertEqual(record.motifs[4].instances[3].tostring(), "CGGAAGGGACGTT")
        self.assertEqual(record.motifs[4].instances[4].tostring(), "ACACACAGACGGT")
        self.assertEqual(record.motifs[4].instances[5].tostring(), "TGCCAGAGAGGTT")
        self.assertEqual(record.motifs[4].instances[6].tostring(), "AGACTGAGACGTT")
        self.assertEqual(record.motifs[4].instances[7].tostring(), "AATCGTAGAGGAT")
        self.assertEqual(record.motifs[4].instances[8].tostring(), "CGTCTCGTAGGGT")
        self.assertEqual(record.motifs[4].instances[9].tostring(), "CGTCGCGGAGGAT")
        self.assertEqual(record.motifs[4].instances[10].tostring(), "CTTCTTAGACGCT")
        self.assertEqual(record.motifs[4].instances[11].tostring(), "CGACGCAGAGGAT")
        self.assertEqual(record.motifs[4].instances[12].tostring(), "ATGCTTAGAGGTT")
        self.assertEqual(record.motifs[4].instances[13].tostring(), "AGACTTGGGCGAT")
        self.assertEqual(record.motifs[4].instances[14].tostring(), "CGACCTGGAGGCT")
        self.assertEqual(record.motifs[4].mask, [1,1,0,1,0,1,1,1,1,1,1,0,1])
        self.assertAlmostEqual(record.motifs[4].score, 16.8287)
        self.assertEqual(record.motifs[5].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[5].instances), 18)
        self.assertEqual(record.motifs[5].instances[0].tostring(), "GTGCGCGAAGGAAGCAGCGCGTG")
        self.assertEqual(record.motifs[5].instances[1].tostring(), "TTGAGCCGAGTAAAGGGCTGGTG")
        self.assertEqual(record.motifs[5].instances[2].tostring(), "CAATGCTAAGCTCTGTGCCGACG")
        self.assertEqual(record.motifs[5].instances[3].tostring(), "CAACTCTCTATGTAGTGCCCGAG")
        self.assertEqual(record.motifs[5].instances[4].tostring(), "CGACGCTTTGTACCTGGCTTGCG")
        self.assertEqual(record.motifs[5].instances[5].tostring(), "CGAGTCAATGACACGCGCCTGGG")
        self.assertEqual(record.motifs[5].instances[6].tostring(), "CGATACGCTGCTAAGTGCCGTCC")
        self.assertEqual(record.motifs[5].instances[7].tostring(), "CCGGGCCAATAGCGGCGCCGGAG")
        self.assertEqual(record.motifs[5].instances[8].tostring(), "CCACGCTTCGACACGTGGTATAG")
        self.assertEqual(record.motifs[5].instances[9].tostring(), "CCGAGCCTCATGTACCGGAAGGG")
        self.assertEqual(record.motifs[5].instances[10].tostring(), "CTGCTCCCCGCATACAGCGCGTG")
        self.assertEqual(record.motifs[5].instances[11].tostring(), "CCGAGGTCCGGTACGGGCAAGCC")
        self.assertEqual(record.motifs[5].instances[12].tostring(), "GTGCTCATAGGGACGTCGCGGAG")
        self.assertEqual(record.motifs[5].instances[13].tostring(), "CCCTACTATGCGGGGGGCAGGTC")
        self.assertEqual(record.motifs[5].instances[14].tostring(), "GCCAGCAATTGCAGGTGGTCGTG")
        self.assertEqual(record.motifs[5].instances[15].tostring(), "CTCTGCGTCGCATGGCGGCGTGG")
        self.assertEqual(record.motifs[5].instances[16].tostring(), "GGAGGCTTAGACTTGGGCGATAC")
        self.assertEqual(record.motifs[5].instances[17].tostring(), "GCATGGAGAGAGATCCGGAGGAG")
        self.assertEqual(record.motifs[5].mask, [1,0,1,0,1,1,0,0,0,1,0,0,0,0,1,0,1,1,0,0,1,0,1])
        self.assertAlmostEqual(record.motifs[5].score, 15.0441)
        self.assertEqual(record.motifs[6].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[6].instances), 20)
        self.assertEqual(record.motifs[6].instances[0].tostring(), "GCGCGTGTGTGTAAC")
        self.assertEqual(record.motifs[6].instances[1].tostring(), "GCACAGAGCTTAGCA")
        self.assertEqual(record.motifs[6].instances[2].tostring(), "GGTGGTCATCGGGCA")
        self.assertEqual(record.motifs[6].instances[3].tostring(), "GCGCGTGTCATTGAC")
        self.assertEqual(record.motifs[6].instances[4].tostring(), "GGACGGCACTTAGCA")
        self.assertEqual(record.motifs[6].instances[5].tostring(), "GCGCGTCCCGGGCCA")
        self.assertEqual(record.motifs[6].instances[6].tostring(), "GCTCGGCCCGTTGTC")
        self.assertEqual(record.motifs[6].instances[7].tostring(), "GCGCGTGTCCTTTAA")
        self.assertEqual(record.motifs[6].instances[8].tostring(), "GCTGATCGCTGCTCC")
        self.assertEqual(record.motifs[6].instances[9].tostring(), "GCCCGTACCGGACCT")
        self.assertEqual(record.motifs[6].instances[10].tostring(), "GGACGTCGCGGAGGA")
        self.assertEqual(record.motifs[6].instances[11].tostring(), "GCGGGGGGCAGGTCA")
        self.assertEqual(record.motifs[6].instances[12].tostring(), "GGACGTACTGGCACA")
        self.assertEqual(record.motifs[6].instances[13].tostring(), "GCAGGTGGTCGTGCA")
        self.assertEqual(record.motifs[6].instances[14].tostring(), "GCGCATACCTTAACA")
        self.assertEqual(record.motifs[6].instances[15].tostring(), "GCACGGGACTTCAAC")
        self.assertEqual(record.motifs[6].instances[16].tostring(), "GCACGTAGCTGGTAA")
        self.assertEqual(record.motifs[6].instances[17].tostring(), "GCTCGTCTATGGTCA")
        self.assertEqual(record.motifs[6].instances[18].tostring(), "GCGCATGCTGGATCC")
        self.assertEqual(record.motifs[6].instances[19].tostring(), "GGCCGTCAGCTCTCA")
        self.assertEqual(record.motifs[6].mask, [1,1,0,1,1,1,1,0,1,0,1,0,0,1,1])
        self.assertAlmostEqual(record.motifs[6].score, 13.3145)
        self.assertEqual(record.motifs[7].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[7].instances), 20)
        self.assertEqual(record.motifs[7].instances[0].tostring(), "GAACCGAGGTCCGGTACGGGC")
        self.assertEqual(record.motifs[7].instances[1].tostring(), "GCCCCCCGCATAGTAGGGGGA")
        self.assertEqual(record.motifs[7].instances[2].tostring(), "GTCCCTGGGTAAGCTTGGGGC")
        self.assertEqual(record.motifs[7].instances[3].tostring(), "ACTCCACGCTTCGACACGTGG")
        self.assertEqual(record.motifs[7].instances[4].tostring(), "ATCCTCTGCGTCGCATGGCGG")
        self.assertEqual(record.motifs[7].instances[5].tostring(), "GTTCAATGCTAAGCTCTGTGC")
        self.assertEqual(record.motifs[7].instances[6].tostring(), "GCTCATAGGGACGTCGCGGAG")
        self.assertEqual(record.motifs[7].instances[7].tostring(), "GTCCCGGGCCAATAGCGGCGC")
        self.assertEqual(record.motifs[7].instances[8].tostring(), "GCACTTAGCAGCGTATCGTTA")
        self.assertEqual(record.motifs[7].instances[9].tostring(), "GGCCCTCGGATCGCTTGGGAA")
        self.assertEqual(record.motifs[7].instances[10].tostring(), "CTGCTGGACAACGGGCCGAGC")
        self.assertEqual(record.motifs[7].instances[11].tostring(), "GGGCACTACATAGAGAGTTGC")
        self.assertEqual(record.motifs[7].instances[12].tostring(), "AGCCTCCAGGTCGCATGGAGA")
        self.assertEqual(record.motifs[7].instances[13].tostring(), "AATCGTAGATCAGAGGCGAGA")
        self.assertEqual(record.motifs[7].instances[14].tostring(), "GAACTCCACTAAGACTTGAGA")
        self.assertEqual(record.motifs[7].instances[15].tostring(), "GAGCAGCGATCAGCTTGTGGG")
        self.assertEqual(record.motifs[7].instances[16].tostring(), "GCCAGGTACAAAGCGTCGTGC")
        self.assertEqual(record.motifs[7].instances[17].tostring(), "AGTCAATGACACGCGCCTGGG")
        self.assertEqual(record.motifs[7].instances[18].tostring(), "GGTCATGGAATCTTATGTAGC")
        self.assertEqual(record.motifs[7].instances[19].tostring(), "GTAGATAACAGAGGTCGGGGG")
        self.assertEqual(record.motifs[7].mask, [1,0,0,1,0,0,0,1,1,0,0,1,1,0,0,0,1,1,0,1,1])
        self.assertAlmostEqual(record.motifs[7].score, 11.6098)
        self.assertEqual(record.motifs[8].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[8].instances), 14)
        self.assertEqual(record.motifs[8].instances[0].tostring(), "CCGAGTAAAGGGCTG")
        self.assertEqual(record.motifs[8].instances[1].tostring(), "GTGGTCATCGGGCAC")
        self.assertEqual(record.motifs[8].instances[2].tostring(), "GATAACAGAGGTCGG")
        self.assertEqual(record.motifs[8].instances[3].tostring(), "CGGCGCCGGAGTCTG")
        self.assertEqual(record.motifs[8].instances[4].tostring(), "GCGCGTCCCGGGCCA")
        self.assertEqual(record.motifs[8].instances[5].tostring(), "CTGGACAACGGGCCG")
        self.assertEqual(record.motifs[8].instances[6].tostring(), "CGGATACTGGGGCAG")
        self.assertEqual(record.motifs[8].instances[7].tostring(), "GGGAGCAGCGATCAG")
        self.assertEqual(record.motifs[8].instances[8].tostring(), "CAGAACCGAGGTCCG")
        self.assertEqual(record.motifs[8].instances[9].tostring(), "GGGTCCCTGGGTAAG")
        self.assertEqual(record.motifs[8].instances[10].tostring(), "GTGCTCATAGGGACG")
        self.assertEqual(record.motifs[8].instances[11].tostring(), "GAGATCCGGAGGAGG")
        self.assertEqual(record.motifs[8].instances[12].tostring(), "GCGATCCGAGGGCCG")
        self.assertEqual(record.motifs[8].instances[13].tostring(), "GAGTTCACATGGCTG")
        self.assertEqual(record.motifs[8].mask, [1,0,1,0,0,1,1,0,1,1,1,1,1,0,1])
        self.assertAlmostEqual(record.motifs[8].score, 11.2943)
        self.assertEqual(record.motifs[9].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[9].instances), 18)
        self.assertEqual(record.motifs[9].instances[0].tostring(), "TAGAGGCGGTG")
        self.assertEqual(record.motifs[9].instances[1].tostring(), "GCTAAGCTCTG")
        self.assertEqual(record.motifs[9].instances[2].tostring(), "TGGAAGCAGTG")
        self.assertEqual(record.motifs[9].instances[3].tostring(), "GCGAGGCTGTG")
        self.assertEqual(record.motifs[9].instances[4].tostring(), "ACGACGCTTTG")
        self.assertEqual(record.motifs[9].instances[5].tostring(), "GGGACGCGCAC")
        self.assertEqual(record.motifs[9].instances[6].tostring(), "TCGAAGCGTGG")
        self.assertEqual(record.motifs[9].instances[7].tostring(), "TGTATGCGGGG")
        self.assertEqual(record.motifs[9].instances[8].tostring(), "GGTAAGCTTGG")
        self.assertEqual(record.motifs[9].instances[9].tostring(), "TGTACGCTGGG")
        self.assertEqual(record.motifs[9].instances[10].tostring(), "ACTATGCGGGG")
        self.assertEqual(record.motifs[9].instances[11].tostring(), "GGTATGCGCTG")
        self.assertEqual(record.motifs[9].instances[12].tostring(), "GGTACCCGGAG")
        self.assertEqual(record.motifs[9].instances[13].tostring(), "GCGACGCAGAG")
        self.assertEqual(record.motifs[9].instances[14].tostring(), "TGGCGGCGTGG")
        self.assertEqual(record.motifs[9].instances[15].tostring(), "TCTAGGCGGGC")
        self.assertEqual(record.motifs[9].instances[16].tostring(), "AGTATGCTTAG")
        self.assertEqual(record.motifs[9].instances[17].tostring(), "TGGAGGCTTAG")
        self.assertEqual(record.motifs[9].mask, [1,1,1,1,0,1,1,1,1,1,1])
        self.assertAlmostEqual(record.motifs[9].score, 9.7924)
        self.assertEqual(record.motifs[10].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[10].instances), 13)
        self.assertEqual(record.motifs[10].instances[0].tostring(), "GCACAGAGCTTAGCATTGAAC")
        self.assertEqual(record.motifs[10].instances[1].tostring(), "GTCCGCGGATTCCCAACATGC")
        self.assertEqual(record.motifs[10].instances[2].tostring(), "ATACACAGCCTCGCAAGCCAG")
        self.assertEqual(record.motifs[10].instances[3].tostring(), "GGCCCGGGACGCGCACTAAGA")
        self.assertEqual(record.motifs[10].instances[4].tostring(), "GCCCGTTGTCCAGCAGACGGC")
        self.assertEqual(record.motifs[10].instances[5].tostring(), "GAGCAGCGATCAGCTTGTGGG")
        self.assertEqual(record.motifs[10].instances[6].tostring(), "GAACCGAGGTCCGGTACGGGC")
        self.assertEqual(record.motifs[10].instances[7].tostring(), "GTCCCTGGGTAAGCTTGGGGC")
        self.assertEqual(record.motifs[10].instances[8].tostring(), "GACCTGCCCCCCGCATAGTAG")
        self.assertEqual(record.motifs[10].instances[9].tostring(), "AACCAGCGCATACCTTAACAG")
        self.assertEqual(record.motifs[10].instances[10].tostring(), "ATCCTCTGCGTCGCATGGCGG")
        self.assertEqual(record.motifs[10].instances[11].tostring(), "GACCATAGACGAGCATCAAAG")
        self.assertEqual(record.motifs[10].instances[12].tostring(), "GGCCCTCGGATCGCTTGGGAA")
        self.assertEqual(record.motifs[10].mask, [1,0,1,1,0,0,0,1,0,0,0,1,1,1,1,0,0,0,0,1,1])
        self.assertAlmostEqual(record.motifs[10].score, 9.01393)
        self.assertEqual(record.motifs[11].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[11].instances), 16)
        self.assertEqual(record.motifs[11].instances[0].tostring(), "GCCGTCCGTC")
        self.assertEqual(record.motifs[11].instances[1].tostring(), "GGCGTGCGCG")
        self.assertEqual(record.motifs[11].instances[2].tostring(), "GGCGCGTGTC")
        self.assertEqual(record.motifs[11].instances[3].tostring(), "AGCGCGTGTG")
        self.assertEqual(record.motifs[11].instances[4].tostring(), "GCGGTGCGTG")
        self.assertEqual(record.motifs[11].instances[5].tostring(), "AGCGCGTGTC")
        self.assertEqual(record.motifs[11].instances[6].tostring(), "AGCGTCCGCG")
        self.assertEqual(record.motifs[11].instances[7].tostring(), "ACCGTCTGTG")
        self.assertEqual(record.motifs[11].instances[8].tostring(), "GCCATGCGAC")
        self.assertEqual(record.motifs[11].instances[9].tostring(), "ACCACCCGTC")
        self.assertEqual(record.motifs[11].instances[10].tostring(), "GGCGCCGGAG")
        self.assertEqual(record.motifs[11].instances[11].tostring(), "ACCACGTGTC")
        self.assertEqual(record.motifs[11].instances[12].tostring(), "GGCTTGCGAG")
        self.assertEqual(record.motifs[11].instances[13].tostring(), "GCGATCCGAG")
        self.assertEqual(record.motifs[11].instances[14].tostring(), "AGTGCGCGTC")
        self.assertEqual(record.motifs[11].instances[15].tostring(), "AGTGCCCGAG")
        self.assertEqual(record.motifs[11].mask, [1,1,1,1,1,1,1,1,1,1])
        self.assertAlmostEqual(record.motifs[11].score, 7.51121)
        self.assertEqual(record.motifs[12].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[12].instances), 16)
        self.assertEqual(record.motifs[12].instances[0].tostring(), "GCCGACGGGTGGTCATCGGG")
        self.assertEqual(record.motifs[12].instances[1].tostring(), "GCACGACGCTTTGTACCTGG")
        self.assertEqual(record.motifs[12].instances[2].tostring(), "CCTGGGAGGGTTCAATAACG")
        self.assertEqual(record.motifs[12].instances[3].tostring(), "GCGCGTCCCGGGCCAATAGC")
        self.assertEqual(record.motifs[12].instances[4].tostring(), "GCCGTCTGCTGGACAACGGG")
        self.assertEqual(record.motifs[12].instances[5].tostring(), "GTCCCTTCCGGTACATGAGG")
        self.assertEqual(record.motifs[12].instances[6].tostring(), "GCTGCTCCCCGCATACAGCG")
        self.assertEqual(record.motifs[12].instances[7].tostring(), "GCCCCAAGCTTACCCAGGGA")
        self.assertEqual(record.motifs[12].instances[8].tostring(), "ACCGGCTGACGCTAATACGG")
        self.assertEqual(record.motifs[12].instances[9].tostring(), "GCGGGGGGCAGGTCATTACA")
        self.assertEqual(record.motifs[12].instances[10].tostring(), "GCTGGCAGCGTCTAAGAAGG")
        self.assertEqual(record.motifs[12].instances[11].tostring(), "GCAGGTGGTCGTGCAATACG")
        self.assertEqual(record.motifs[12].instances[12].tostring(), "GCTGGTTGAAGTCCCGTGCG")
        self.assertEqual(record.motifs[12].instances[13].tostring(), "GCACGTAGCTGGTAAATAGG")
        self.assertEqual(record.motifs[12].instances[14].tostring(), "GCGGCGTGGATTTCATACAG")
        self.assertEqual(record.motifs[12].instances[15].tostring(), "CCTGGAGGCTTAGACTTGGG")
        self.assertEqual(record.motifs[12].mask, [1,1,0,1,1,0,0,1,1,0,1,0,0,0,1,0,0,0,1,1])
        self.assertAlmostEqual(record.motifs[12].score, 5.63667)
        self.assertEqual(record.motifs[13].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[13].instances), 15)
        self.assertEqual(record.motifs[13].instances[0].tostring(), "GCCGACGGGTGGTCATCGGG")
        self.assertEqual(record.motifs[13].instances[1].tostring(), "ATCCGCGGACGCTTAGAGGG")
        self.assertEqual(record.motifs[13].instances[2].tostring(), "ACGCTTTGTACCTGGCTTGC")
        self.assertEqual(record.motifs[13].instances[3].tostring(), "ACGGACGGCACTTAGCAGCG")
        self.assertEqual(record.motifs[13].instances[4].tostring(), "GCCGTCTGCTGGACAACGGG")
        self.assertEqual(record.motifs[13].instances[5].tostring(), "ACACACAGACGGTTGAAAGG")
        self.assertEqual(record.motifs[13].instances[6].tostring(), "GCCGATAGTGCTTAAGTTCG")
        self.assertEqual(record.motifs[13].instances[7].tostring(), "CTTGCCCGTACCGGACCTCG")
        self.assertEqual(record.motifs[13].instances[8].tostring(), "ACCGGCTGACGCTAATACGG")
        self.assertEqual(record.motifs[13].instances[9].tostring(), "GCCCCCCGCATAGTAGGGGG")
        self.assertEqual(record.motifs[13].instances[10].tostring(), "GCTGGCAGCGTCTAAGAAGG")
        self.assertEqual(record.motifs[13].instances[11].tostring(), "GCAGGTGGTCGTGCAATACG")
        self.assertEqual(record.motifs[13].instances[12].tostring(), "ACGCACGGGACTTCAACCAG")
        self.assertEqual(record.motifs[13].instances[13].tostring(), "GCACGTAGCTGGTAAATAGG")
        self.assertEqual(record.motifs[13].instances[14].tostring(), "ATCCTCTGCGTCGCATGGCG")
        self.assertEqual(record.motifs[13].mask, [1,1,0,1,0,1,0,1,0,0,1,0,1,0,1,0,0,0,1,1])
        self.assertAlmostEqual(record.motifs[13].score, 3.89842)
        self.assertEqual(record.motifs[14].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[14].instances), 14)
        self.assertEqual(record.motifs[14].instances[0].tostring(), "GAGGCTGTGTAT")
        self.assertEqual(record.motifs[14].instances[1].tostring(), "GAGGTCGGGGGT")
        self.assertEqual(record.motifs[14].instances[2].tostring(), "GACGGACGGCAC")
        self.assertEqual(record.motifs[14].instances[3].tostring(), "TTGGCCCGGGAC")
        self.assertEqual(record.motifs[14].instances[4].tostring(), "GAGGCTCGGCCC")
        self.assertEqual(record.motifs[14].instances[5].tostring(), "CACGCGCTGTAT")
        self.assertEqual(record.motifs[14].instances[6].tostring(), "TAGGCCAGGTAT")
        self.assertEqual(record.motifs[14].instances[7].tostring(), "GAGGTCCGGTAC")
        self.assertEqual(record.motifs[14].instances[8].tostring(), "TACGCTGGGGAT")
        self.assertEqual(record.motifs[14].instances[9].tostring(), "GTCGCGGAGGAT")
        self.assertEqual(record.motifs[14].instances[10].tostring(), "TACGCACGGGAC")
        self.assertEqual(record.motifs[14].instances[11].tostring(), "TACTCCGGGTAC")
        self.assertEqual(record.motifs[14].instances[12].tostring(), "GACGCAGAGGAT")
        self.assertEqual(record.motifs[14].instances[13].tostring(), "TAGGCGGGCCAT")
        self.assertEqual(record.motifs[14].mask, [1,1,1,1,1,0,1,1,1,0,1,1])
        self.assertAlmostEqual(record.motifs[14].score, 3.33444)
        self.assertEqual(record.motifs[15].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs[15].instances), 21)
        self.assertEqual(record.motifs[15].instances[0].tostring(), "CGGCTCAATCGTAGAGGC")
        self.assertEqual(record.motifs[15].instances[1].tostring(), "CGACGGGTGGTCATCGGG")
        self.assertEqual(record.motifs[15].instances[2].tostring(), "CGCTTAGAGGGCACAAGC")
        self.assertEqual(record.motifs[15].instances[3].tostring(), "TGACACGCGCCTGGGAGG")
        self.assertEqual(record.motifs[15].instances[4].tostring(), "CGATACGCTGCTAAGTGC")
        self.assertEqual(record.motifs[15].instances[5].tostring(), "CGTCCCGGGCCAATAGCG")
        self.assertEqual(record.motifs[15].instances[6].tostring(), "CCACGCTTCGACACGTGG")
        self.assertEqual(record.motifs[15].instances[7].tostring(), "CGTCTGCTGGACAACGGG")
        self.assertEqual(record.motifs[15].instances[8].tostring(), "ACACAGACGGTTGAAAGG")
        self.assertEqual(record.motifs[15].instances[9].tostring(), "TGCTCCCCGCATACAGCG")
        self.assertEqual(record.motifs[15].instances[10].tostring(), "TGAGGCTTGCCCGTACCG")
        self.assertEqual(record.motifs[15].instances[11].tostring(), "TGCCCCAAGCTTACCCAG")
        self.assertEqual(record.motifs[15].instances[12].tostring(), "CGGCTGACGCTAATACGG")
        self.assertEqual(record.motifs[15].instances[13].tostring(), "CGCGACGTCCCTATGAGC")
        self.assertEqual(record.motifs[15].instances[14].tostring(), "TGCCCCCCGCATAGTAGG")
        self.assertEqual(record.motifs[15].instances[15].tostring(), "CGTTGCCTTCTTAGACGC")
        self.assertEqual(record.motifs[15].instances[16].tostring(), "TGACTCAATCGTAGACCC")
        self.assertEqual(record.motifs[15].instances[17].tostring(), "AGTCCCGTGCGTATGTGG")
        self.assertEqual(record.motifs[15].instances[18].tostring(), "AGGCTCGCACGTAGCTGG")
        self.assertEqual(record.motifs[15].instances[19].tostring(), "CCACGCCGCCATGCGACG")
        self.assertEqual(record.motifs[15].instances[20].tostring(), "AGCCTCCAGGTCGCATGG")
        self.assertEqual(record.motifs[15].mask, [1,1,0,1,0,1,0,0,1,1,0,1,1,0,0,0,1,1])
        self.assertAlmostEqual(record.motifs[15].score, 1.0395)
 
    def test_pfm_parsing(self):
        """Test to be sure that Motif can parse pfm  files.
        """
        motif= Motif.read(self.PFMin,"jaspar-pfm")
        assert motif.length==12

    def test_sites_parsing(self):
        """Test to be sure that Motif can parse sites files.
        """
        motif= Motif.read(self.SITESin,"jaspar-sites")
        assert motif.length==6

    def test_FAoutput(self):
        """Ensure that we can write proper FASTA output files.
        """
        output_handle = open(self.FAout, "w")
        output_handle.write(self.m.format("fasta"))
        output_handle.close()

    def test_TFoutput(self):
        """Ensure that we can write proper TransFac output files.
        """
        output_handle = open(self.TFout, "w")
        output_handle.write(self.m.format("transfac"))
        output_handle.close()

    def test_pfm_output(self):
        """Ensure that we can write proper pfm output files.
        """
        output_handle = open(self.PFMout, "w")
        output_handle.write(self.m.format("jaspar-pfm"))
        output_handle.close()
        
        
class TestMEME(unittest.TestCase):
        
    def test_meme_parser_1(self):
        """Test if Motif can parse MEME output files (first test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MEME
        handle = open("Motif/meme.out")
        record = MEME.read(handle)
        self.assertEqual(record.version, '3.5.7')
        self.assertEqual(record.datafile, 'test.fa')
        self.assertEqual(record.alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.sequence_names), 10)
        self.assertEqual(record.sequence_names[0], 'SEQ1;')
        self.assertEqual(record.sequence_names[1], 'SEQ2;')
        self.assertEqual(record.sequence_names[2], 'SEQ3;')
        self.assertEqual(record.sequence_names[3], 'SEQ4;')
        self.assertEqual(record.sequence_names[4], 'SEQ5;')
        self.assertEqual(record.sequence_names[5], 'SEQ6;')
        self.assertEqual(record.sequence_names[6], 'SEQ7;')
        self.assertEqual(record.sequence_names[7], 'SEQ8;')
        self.assertEqual(record.sequence_names[8], 'SEQ9;')
        self.assertEqual(record.sequence_names[9], 'SEQ10;')
        self.assertEqual(record.command, 'meme test.fa -dna -w 10 -dir /home/bartek/MetaMotif/meme')
        self.assertEqual(len(record.motifs), 1)
        motif = record.motifs[0]
        self.assertEqual(motif.num_occurrences, 10)
        self.assertAlmostEqual(motif.evalue, 1.1e-22)
        self.assertEqual(motif.alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.name, "Motif 1")
        self.assertEqual(len(motif.instances), 10)
        self.assertAlmostEqual(motif.instances[0].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[1].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[2].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[3].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[4].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[5].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[6].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[7].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[8].pvalue, 8.71e-07)
        self.assertAlmostEqual(motif.instances[9].pvalue, 8.71e-07)
        self.assertEqual(motif.instances[0].sequence_name, 'SEQ10;')
        self.assertEqual(motif.instances[1].sequence_name, 'SEQ9;')
        self.assertEqual(motif.instances[2].sequence_name, 'SEQ8;')
        self.assertEqual(motif.instances[3].sequence_name, 'SEQ7;')
        self.assertEqual(motif.instances[4].sequence_name, 'SEQ6;')
        self.assertEqual(motif.instances[5].sequence_name, 'SEQ5;')
        self.assertEqual(motif.instances[6].sequence_name, 'SEQ4;')
        self.assertEqual(motif.instances[7].sequence_name, 'SEQ3;')
        self.assertEqual(motif.instances[8].sequence_name, 'SEQ2;')
        self.assertEqual(motif.instances[9].sequence_name, 'SEQ1;')
        self.assertEqual(motif.instances[0].start, 3)
        self.assertEqual(motif.instances[1].start, 93)
        self.assertEqual(motif.instances[2].start, 172)
        self.assertEqual(motif.instances[3].start, 177)
        self.assertEqual(motif.instances[4].start, 105)
        self.assertEqual(motif.instances[5].start, 185)
        self.assertEqual(motif.instances[6].start, 173)
        self.assertEqual(motif.instances[7].start, 112)
        self.assertEqual(motif.instances[8].start, 172)
        self.assertEqual(motif.instances[9].start, 52)
        self.assertEqual(motif.instances[0].strand, '+')
        self.assertEqual(motif.instances[1].strand, '+')
        self.assertEqual(motif.instances[2].strand, '+')
        self.assertEqual(motif.instances[3].strand, '+')
        self.assertEqual(motif.instances[4].strand, '+')
        self.assertEqual(motif.instances[5].strand, '+')
        self.assertEqual(motif.instances[6].strand, '+')
        self.assertEqual(motif.instances[7].strand, '+')
        self.assertEqual(motif.instances[8].strand, '+')
        self.assertEqual(motif.instances[9].strand, '+')
        self.assertEqual(motif.instances[0].length, 10)
        self.assertEqual(motif.instances[1].length, 10)
        self.assertEqual(motif.instances[2].length, 10)
        self.assertEqual(motif.instances[3].length, 10)
        self.assertEqual(motif.instances[4].length, 10)
        self.assertEqual(motif.instances[5].length, 10)
        self.assertEqual(motif.instances[6].length, 10)
        self.assertEqual(motif.instances[7].length, 10)
        self.assertEqual(motif.instances[8].length, 10)
        self.assertEqual(motif.instances[9].length, 10)
        self.assertEqual(motif.instances[0].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[1].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[2].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[3].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[4].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[5].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[6].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[7].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[8].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[9].motif_name, 'Motif 1')
        self.assertEqual(motif.instances[0].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[1].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[2].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[3].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[4].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[5].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[6].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[7].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[8].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[9].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.instances[0].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[1].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[2].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[3].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[4].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[5].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[6].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[7].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[8].tostring(), "CTCAATCGTA")
        self.assertEqual(motif.instances[9].tostring(), "CTCAATCGTA")
        handle.close()

    def test_meme_parser_2(self):
        """Test if Motif can parse MEME output files (second test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MEME
        handle = open("Motif/meme.dna.oops.txt")
        record = MEME.read(handle)
        self.assertEqual(record.version, '3.0')
        self.assertEqual(record.datafile, 'INO_up800.s')
        self.assertEqual(record.alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.sequence_names), 7)
        self.assertEqual(record.sequence_names[0], 'CHO1')
        self.assertEqual(record.sequence_names[1], 'CHO2')
        self.assertEqual(record.sequence_names[2], 'FAS1')
        self.assertEqual(record.sequence_names[3], 'FAS2')
        self.assertEqual(record.sequence_names[4], 'ACC1')
        self.assertEqual(record.sequence_names[5], 'INO1')
        self.assertEqual(record.sequence_names[6], 'OPI3')
        self.assertEqual(record.command, 'meme -mod oops -dna -revcomp -nmotifs 2 -bfile yeast.nc.6.freq INO_up800.s')
        self.assertEqual(len(record.motifs), 2)
        motif = record.motifs[0]
        self.assertEqual(motif.num_occurrences, 7)
        self.assertAlmostEqual(motif.evalue, 0.2)
        self.assertEqual(motif.alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.name, "Motif 1")
        self.assertEqual(len(motif.instances), 7)
        self.assertAlmostEqual(motif.instances[0].pvalue, 1.85e-08)
        self.assertAlmostEqual(motif.instances[1].pvalue, 1.85e-08)
        self.assertAlmostEqual(motif.instances[2].pvalue, 1.52e-07)
        self.assertAlmostEqual(motif.instances[3].pvalue, 2.52e-07)
        self.assertAlmostEqual(motif.instances[4].pvalue, 4.23e-07)
        self.assertAlmostEqual(motif.instances[5].pvalue, 9.43e-07)
        self.assertAlmostEqual(motif.instances[6].pvalue, 3.32e-06)
        self.assertEqual(motif.instances[0].sequence_name, 'INO1')
        self.assertEqual(motif.instances[1].sequence_name, 'FAS1')
        self.assertEqual(motif.instances[2].sequence_name, 'ACC1')
        self.assertEqual(motif.instances[3].sequence_name, 'CHO2')
        self.assertEqual(motif.instances[4].sequence_name, 'CHO1')
        self.assertEqual(motif.instances[5].sequence_name, 'FAS2')
        self.assertEqual(motif.instances[6].sequence_name, 'OPI3')
        self.assertEqual(motif.instances[0].strand, '-')
        self.assertEqual(motif.instances[1].strand, '+')
        self.assertEqual(motif.instances[2].strand, '+')
        self.assertEqual(motif.instances[3].strand, '+')
        self.assertEqual(motif.instances[4].strand, '+')
        self.assertEqual(motif.instances[5].strand, '+')
        self.assertEqual(motif.instances[6].strand, '+')
        self.assertEqual(motif.instances[0].length, 12)
        self.assertEqual(motif.instances[1].length, 12)
        self.assertEqual(motif.instances[2].length, 12)
        self.assertEqual(motif.instances[3].length, 12)
        self.assertEqual(motif.instances[4].length, 12)
        self.assertEqual(motif.instances[5].length, 12)
        self.assertEqual(motif.instances[6].length, 12)
        self.assertEqual(motif.instances[0].start, 620)
        self.assertEqual(motif.instances[1].start,  95)
        self.assertEqual(motif.instances[2].start,  83)
        self.assertEqual(motif.instances[3].start, 354)
        self.assertEqual(motif.instances[4].start, 611)
        self.assertEqual(motif.instances[5].start, 567)
        self.assertEqual(motif.instances[6].start, 340)
        self.assertEqual(motif.instances[0].tostring(), "TTCACATGCCGC")
        self.assertEqual(motif.instances[1].tostring(), "TTCACATGCCGC")
        self.assertEqual(motif.instances[2].tostring(), "TTCACATGGCCC")
        self.assertEqual(motif.instances[3].tostring(), "TTCTCATGCCGC")
        self.assertEqual(motif.instances[4].tostring(), "TTCACACGGCAC")
        self.assertEqual(motif.instances[5].tostring(), "TTCACATGCTAC")
        self.assertEqual(motif.instances[6].tostring(), "TTCAGATCGCTC")
        motif = record.motifs[1]
        self.assertEqual(motif.num_occurrences, 7)
        self.assertAlmostEqual(motif.evalue, 110)
        self.assertEqual(motif.alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(motif.name, "Motif 2")
        self.assertEqual(len(motif.instances), 7)
        self.assertAlmostEqual(motif.instances[0].pvalue, 3.24e-07)
        self.assertAlmostEqual(motif.instances[1].pvalue, 3.24e-07)
        self.assertAlmostEqual(motif.instances[2].pvalue, 3.24e-07)
        self.assertAlmostEqual(motif.instances[3].pvalue, 5.29e-06)
        self.assertAlmostEqual(motif.instances[4].pvalue, 6.25e-06)
        self.assertAlmostEqual(motif.instances[5].pvalue, 8.48e-06)
        self.assertAlmostEqual(motif.instances[6].pvalue, 8.48e-06)
        self.assertEqual(motif.instances[0].sequence_name, 'OPI3')
        self.assertEqual(motif.instances[1].sequence_name, 'ACC1')
        self.assertEqual(motif.instances[2].sequence_name, 'CHO1')
        self.assertEqual(motif.instances[3].sequence_name, 'INO1')
        self.assertEqual(motif.instances[4].sequence_name, 'FAS1')
        self.assertEqual(motif.instances[5].sequence_name, 'FAS2')
        self.assertEqual(motif.instances[6].sequence_name, 'CHO2')
        self.assertEqual(motif.instances[0].strand, '-')
        self.assertEqual(motif.instances[1].strand, '+')
        self.assertEqual(motif.instances[2].strand, '-')
        self.assertEqual(motif.instances[3].strand, '-')
        self.assertEqual(motif.instances[4].strand, '+')
        self.assertEqual(motif.instances[5].strand, '-')
        self.assertEqual(motif.instances[6].strand, '-')
        self.assertEqual(motif.instances[0].length, 10)
        self.assertEqual(motif.instances[1].length, 10)
        self.assertEqual(motif.instances[2].length, 10)
        self.assertEqual(motif.instances[3].length, 10)
        self.assertEqual(motif.instances[4].length, 10)
        self.assertEqual(motif.instances[5].length, 10)
        self.assertEqual(motif.instances[6].length, 10)
        self.assertEqual(motif.instances[0].start, 186)
        self.assertEqual(motif.instances[1].start, 232)
        self.assertEqual(motif.instances[2].start, 559)
        self.assertEqual(motif.instances[3].start, 283)
        self.assertEqual(motif.instances[4].start,  44)
        self.assertEqual(motif.instances[5].start, 185)
        self.assertEqual(motif.instances[6].start, 413)
        self.assertEqual(motif.instances[0].tostring(), "TCTGGCACAG")
        self.assertEqual(motif.instances[1].tostring(), "TCTGGCACAG")
        self.assertEqual(motif.instances[2].tostring(), "TCTGGCACAG")
        self.assertEqual(motif.instances[3].tostring(), "GCGGGCGCAG")
        self.assertEqual(motif.instances[4].tostring(), "GCAGGCACGG")
        self.assertEqual(motif.instances[5].tostring(), "TCTGGCACTC")
        self.assertEqual(motif.instances[6].tostring(), "TCTGGCATCG")
        handle.close()

    def test_meme_parser_3(self):
        """Test if Motif can parse MEME output files (third test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MEME
        handle = open("Motif/meme.protein.oops.txt")
        record = MEME.read(handle)
        self.assertEqual(record.version, '3.0')
        self.assertEqual(record.datafile, 'adh.s')
        self.assertEqual(record.alphabet, IUPAC.protein)
        self.assertEqual(len(record.sequence_names), 33)
        self.assertEqual(record.sequence_names[0], "2BHD_STREX")
        self.assertEqual(record.sequence_names[1], "3BHD_COMTE")
        self.assertEqual(record.sequence_names[2], "ADH_DROME")
        self.assertEqual(record.sequence_names[3], "AP27_MOUSE")
        self.assertEqual(record.sequence_names[4], "BA72_EUBSP")
        self.assertEqual(record.sequence_names[5], "BDH_HUMAN")
        self.assertEqual(record.sequence_names[6], "BPHB_PSEPS")
        self.assertEqual(record.sequence_names[7], "BUDC_KLETE")
        self.assertEqual(record.sequence_names[8], "DHES_HUMAN")
        self.assertEqual(record.sequence_names[9], "DHGB_BACME")
        self.assertEqual(record.sequence_names[10], "DHII_HUMAN")
        self.assertEqual(record.sequence_names[11], "DHMA_FLAS1")
        self.assertEqual(record.sequence_names[12], "ENTA_ECOLI")
        self.assertEqual(record.sequence_names[13], "FIXR_BRAJA")
        self.assertEqual(record.sequence_names[14], "GUTD_ECOLI")
        self.assertEqual(record.sequence_names[15], "HDE_CANTR")
        self.assertEqual(record.sequence_names[16], "HDHA_ECOLI")
        self.assertEqual(record.sequence_names[17], "LIGD_PSEPA")
        self.assertEqual(record.sequence_names[18], "NODG_RHIME")
        self.assertEqual(record.sequence_names[19], "RIDH_KLEAE")
        self.assertEqual(record.sequence_names[20], "YINL_LISMO")
        self.assertEqual(record.sequence_names[21], "YRTP_BACSU")
        self.assertEqual(record.sequence_names[22], "CSGA_MYXXA")
        self.assertEqual(record.sequence_names[23], "DHB2_HUMAN")
        self.assertEqual(record.sequence_names[24], "DHB3_HUMAN")
        self.assertEqual(record.sequence_names[25], "DHCA_HUMAN")
        self.assertEqual(record.sequence_names[26], "FABI_ECOLI")
        self.assertEqual(record.sequence_names[27], "FVT1_HUMAN")
        self.assertEqual(record.sequence_names[28], "HMTR_LEIMA")
        self.assertEqual(record.sequence_names[29], "MAS1_AGRRA")
        self.assertEqual(record.sequence_names[30], "PCR_PEA")
        self.assertEqual(record.sequence_names[31], "RFBB_NEIGO")
        self.assertEqual(record.sequence_names[32], "YURA_MYXXA")
        self.assertEqual(record.command, 'meme adh.s -mod oops -protein -nmotifs 2')
        self.assertEqual(len(record.motifs), 2)
        motif = record.motifs[0]
        self.assertEqual(motif.num_occurrences, 33)
        self.assertAlmostEqual(motif.evalue, 3.6e-165)
        self.assertEqual(motif.alphabet, IUPAC.protein)
        self.assertEqual(motif.name, "Motif 1")
        self.assertEqual(len(motif.instances), 33)
        self.assertAlmostEqual(motif.instances[0].pvalue, 1.64e-22)
        self.assertAlmostEqual(motif.instances[1].pvalue, 6.32e-22)
        self.assertAlmostEqual(motif.instances[2].pvalue, 1.13e-21)
        self.assertAlmostEqual(motif.instances[3].pvalue, 4.04e-21)
        self.assertAlmostEqual(motif.instances[4].pvalue, 6.12e-21)
        self.assertAlmostEqual(motif.instances[5].pvalue, 7.52e-20)
        self.assertAlmostEqual(motif.instances[6].pvalue, 3.35e-19)
        self.assertAlmostEqual(motif.instances[7].pvalue, 4.82e-19)
        self.assertAlmostEqual(motif.instances[8].pvalue, 4.82e-19)
        self.assertAlmostEqual(motif.instances[9].pvalue, 1.11e-18)
        self.assertAlmostEqual(motif.instances[10].pvalue, 1.25e-18)
        self.assertAlmostEqual(motif.instances[11].pvalue, 2.23e-18)
        self.assertAlmostEqual(motif.instances[12].pvalue, 5.53e-18)
        self.assertAlmostEqual(motif.instances[13].pvalue, 9.65e-18)
        self.assertAlmostEqual(motif.instances[14].pvalue, 2.86e-17)
        self.assertAlmostEqual(motif.instances[15].pvalue, 8.20e-17)
        self.assertAlmostEqual(motif.instances[16].pvalue, 9.09e-17)
        self.assertAlmostEqual(motif.instances[17].pvalue, 1.37e-16)
        self.assertAlmostEqual(motif.instances[18].pvalue, 2.52e-16)
        self.assertAlmostEqual(motif.instances[19].pvalue, 1.21e-15)
        self.assertAlmostEqual(motif.instances[20].pvalue, 1.61e-15)
        self.assertAlmostEqual(motif.instances[21].pvalue, 1.77e-15)
        self.assertAlmostEqual(motif.instances[22].pvalue, 7.81e-15)
        self.assertAlmostEqual(motif.instances[23].pvalue, 8.55e-15)
        self.assertAlmostEqual(motif.instances[24].pvalue, 1.47e-14)
        self.assertAlmostEqual(motif.instances[25].pvalue, 3.24e-14)
        self.assertAlmostEqual(motif.instances[26].pvalue, 1.80e-12)
        self.assertAlmostEqual(motif.instances[27].pvalue, 2.10e-12)
        self.assertAlmostEqual(motif.instances[28].pvalue, 4.15e-12)
        self.assertAlmostEqual(motif.instances[29].pvalue, 5.20e-12)
        self.assertAlmostEqual(motif.instances[30].pvalue, 4.80e-10)
        self.assertAlmostEqual(motif.instances[31].pvalue, 2.77e-08)
        self.assertAlmostEqual(motif.instances[32].pvalue, 5.72e-08)
        self.assertEqual(motif.instances[0].sequence_name, 'YRTP_BACSU')
        self.assertEqual(motif.instances[1].sequence_name, 'AP27_MOUSE')
        self.assertEqual(motif.instances[2].sequence_name, 'NODG_RHIME')
        self.assertEqual(motif.instances[3].sequence_name, 'BUDC_KLETE')
        self.assertEqual(motif.instances[4].sequence_name, 'FIXR_BRAJA')
        self.assertEqual(motif.instances[5].sequence_name, 'DHGB_BACME')
        self.assertEqual(motif.instances[6].sequence_name, 'HMTR_LEIMA')
        self.assertEqual(motif.instances[7].sequence_name, 'YURA_MYXXA')
        self.assertEqual(motif.instances[8].sequence_name, 'GUTD_ECOLI')
        self.assertEqual(motif.instances[9].sequence_name, '2BHD_STREX')
        self.assertEqual(motif.instances[10].sequence_name, 'HDHA_ECOLI')
        self.assertEqual(motif.instances[11].sequence_name, 'DHB2_HUMAN')
        self.assertEqual(motif.instances[12].sequence_name, 'DHMA_FLAS1')
        self.assertEqual(motif.instances[13].sequence_name, 'HDE_CANTR')
        self.assertEqual(motif.instances[14].sequence_name, 'FVT1_HUMAN')
        self.assertEqual(motif.instances[15].sequence_name, 'BDH_HUMAN')
        self.assertEqual(motif.instances[16].sequence_name, 'RIDH_KLEAE')
        self.assertEqual(motif.instances[17].sequence_name, 'DHES_HUMAN')
        self.assertEqual(motif.instances[18].sequence_name, 'BA72_EUBSP')
        self.assertEqual(motif.instances[19].sequence_name, 'LIGD_PSEPA')
        self.assertEqual(motif.instances[20].sequence_name, 'DHII_HUMAN')
        self.assertEqual(motif.instances[21].sequence_name, 'ENTA_ECOLI')
        self.assertEqual(motif.instances[22].sequence_name, '3BHD_COMTE')
        self.assertEqual(motif.instances[23].sequence_name, 'DHB3_HUMAN')
        self.assertEqual(motif.instances[24].sequence_name, 'RFBB_NEIGO')
        self.assertEqual(motif.instances[25].sequence_name, 'YINL_LISMO')
        self.assertEqual(motif.instances[26].sequence_name, 'BPHB_PSEPS')
        self.assertEqual(motif.instances[27].sequence_name, 'CSGA_MYXXA')
        self.assertEqual(motif.instances[28].sequence_name, 'FABI_ECOLI')
        self.assertEqual(motif.instances[29].sequence_name, 'ADH_DROME')
        self.assertEqual(motif.instances[30].sequence_name, 'DHCA_HUMAN')
        self.assertEqual(motif.instances[31].sequence_name, 'PCR_PEA')
        self.assertEqual(motif.instances[32].sequence_name, 'MAS1_AGRRA')
        self.assertEqual(motif.instances[0].strand, '+')
        self.assertEqual(motif.instances[1].strand, '+')
        self.assertEqual(motif.instances[2].strand, '+')
        self.assertEqual(motif.instances[3].strand, '+')
        self.assertEqual(motif.instances[4].strand, '+')
        self.assertEqual(motif.instances[5].strand, '+')
        self.assertEqual(motif.instances[6].strand, '+')
        self.assertEqual(motif.instances[7].strand, '+')
        self.assertEqual(motif.instances[8].strand, '+')
        self.assertEqual(motif.instances[9].strand, '+')
        self.assertEqual(motif.instances[10].strand, '+')
        self.assertEqual(motif.instances[11].strand, '+')
        self.assertEqual(motif.instances[12].strand, '+')
        self.assertEqual(motif.instances[13].strand, '+')
        self.assertEqual(motif.instances[14].strand, '+')
        self.assertEqual(motif.instances[15].strand, '+')
        self.assertEqual(motif.instances[16].strand, '+')
        self.assertEqual(motif.instances[17].strand, '+')
        self.assertEqual(motif.instances[18].strand, '+')
        self.assertEqual(motif.instances[19].strand, '+')
        self.assertEqual(motif.instances[20].strand, '+')
        self.assertEqual(motif.instances[21].strand, '+')
        self.assertEqual(motif.instances[22].strand, '+')
        self.assertEqual(motif.instances[23].strand, '+')
        self.assertEqual(motif.instances[24].strand, '+')
        self.assertEqual(motif.instances[25].strand, '+')
        self.assertEqual(motif.instances[26].strand, '+')
        self.assertEqual(motif.instances[27].strand, '+')
        self.assertEqual(motif.instances[28].strand, '+')
        self.assertEqual(motif.instances[29].strand, '+')
        self.assertEqual(motif.instances[30].strand, '+')
        self.assertEqual(motif.instances[31].strand, '+')
        self.assertEqual(motif.instances[32].strand, '+')
        self.assertEqual(motif.instances[0].length, 29)
        self.assertEqual(motif.instances[1].length, 29)
        self.assertEqual(motif.instances[2].length, 29)
        self.assertEqual(motif.instances[3].length, 29)
        self.assertEqual(motif.instances[4].length, 29)
        self.assertEqual(motif.instances[5].length, 29)
        self.assertEqual(motif.instances[6].length, 29)
        self.assertEqual(motif.instances[7].length, 29)
        self.assertEqual(motif.instances[8].length, 29)
        self.assertEqual(motif.instances[9].length, 29)
        self.assertEqual(motif.instances[10].length, 29)
        self.assertEqual(motif.instances[11].length, 29)
        self.assertEqual(motif.instances[12].length, 29)
        self.assertEqual(motif.instances[13].length, 29)
        self.assertEqual(motif.instances[14].length, 29)
        self.assertEqual(motif.instances[15].length, 29)
        self.assertEqual(motif.instances[16].length, 29)
        self.assertEqual(motif.instances[17].length, 29)
        self.assertEqual(motif.instances[18].length, 29)
        self.assertEqual(motif.instances[19].length, 29)
        self.assertEqual(motif.instances[20].length, 29)
        self.assertEqual(motif.instances[21].length, 29)
        self.assertEqual(motif.instances[22].length, 29)
        self.assertEqual(motif.instances[23].length, 29)
        self.assertEqual(motif.instances[24].length, 29)
        self.assertEqual(motif.instances[25].length, 29)
        self.assertEqual(motif.instances[26].length, 29)
        self.assertEqual(motif.instances[27].length, 29)
        self.assertEqual(motif.instances[28].length, 29)
        self.assertEqual(motif.instances[29].length, 29)
        self.assertEqual(motif.instances[30].length, 29)
        self.assertEqual(motif.instances[31].length, 29)
        self.assertEqual(motif.instances[32].length, 29)
        self.assertEqual(motif.instances[0].start, 155)
        self.assertEqual(motif.instances[1].start, 149)
        self.assertEqual(motif.instances[2].start, 152)
        self.assertEqual(motif.instances[3].start, 152)
        self.assertEqual(motif.instances[4].start, 189)
        self.assertEqual(motif.instances[5].start, 160)
        self.assertEqual(motif.instances[6].start, 193)
        self.assertEqual(motif.instances[7].start, 160)
        self.assertEqual(motif.instances[8].start, 154)
        self.assertEqual(motif.instances[9].start, 152)
        self.assertEqual(motif.instances[10].start, 159)
        self.assertEqual(motif.instances[11].start, 232)
        self.assertEqual(motif.instances[12].start, 165)
        self.assertEqual(motif.instances[13].start, 467)
        self.assertEqual(motif.instances[14].start, 186)
        self.assertEqual(motif.instances[15].start, 208)
        self.assertEqual(motif.instances[16].start, 160)
        self.assertEqual(motif.instances[17].start, 155)
        self.assertEqual(motif.instances[18].start, 157)
        self.assertEqual(motif.instances[19].start, 157)
        self.assertEqual(motif.instances[20].start, 183)
        self.assertEqual(motif.instances[21].start, 144)
        self.assertEqual(motif.instances[22].start, 151)
        self.assertEqual(motif.instances[23].start, 198)
        self.assertEqual(motif.instances[24].start, 165)
        self.assertEqual(motif.instances[25].start, 154)
        self.assertEqual(motif.instances[26].start, 153)
        self.assertEqual(motif.instances[27].start,  88)
        self.assertEqual(motif.instances[28].start, 159)
        self.assertEqual(motif.instances[29].start, 152)
        self.assertEqual(motif.instances[30].start, 193)
        self.assertEqual(motif.instances[31].start,  26)
        self.assertEqual(motif.instances[32].start, 349)
        self.assertEqual(motif.instances[0].tostring(), "YSASKFAVLGLTESLMQEVRKHNIRVSAL")
        self.assertEqual(motif.instances[1].tostring(), "YSSTKGAMTMLTKAMAMELGPHKIRVNSV")
        self.assertEqual(motif.instances[2].tostring(), "YCASKAGMIGFSKSLAQEIATRNITVNCV")
        self.assertEqual(motif.instances[3].tostring(), "YSSSKFAVRGLTQTAARDLAPLGITVNGF")
        self.assertEqual(motif.instances[4].tostring(), "YATSKAALASLTRELAHDYAPHGIRVNAI")
        self.assertEqual(motif.instances[5].tostring(), "YAASKGGMKLMTETLALEYAPKGIRVNNI")
        self.assertEqual(motif.instances[6].tostring(), "YTMAKGALEGLTRSAALELAPLQIRVNGV")
        self.assertEqual(motif.instances[7].tostring(), "YSASKAFLSTFMESLRVDLRGTGVRVTCI")
        self.assertEqual(motif.instances[8].tostring(), "YSAAKFGGVGLTQSLALDLAEYGITVHSL")
        self.assertEqual(motif.instances[9].tostring(), "YGASKWGVRGLSKLAAVELGTDRIRVNSV")
        self.assertEqual(motif.instances[10].tostring(), "YASSKAAASHLVRNMAFDLGEKNIRVNGI")
        self.assertEqual(motif.instances[11].tostring(), "YGSSKAAVTMFSSVMRLELSKWGIKVASI")
        self.assertEqual(motif.instances[12].tostring(), "YVAAKGGVAMLTRAMAVDLARHGILVNMI")
        self.assertEqual(motif.instances[13].tostring(), "YSSSKAGILGLSKTMAIEGAKNNIKVNIV")
        self.assertEqual(motif.instances[14].tostring(), "YSASKFAIRGLAEALQMEVKPYNVYITVA")
        self.assertEqual(motif.instances[15].tostring(), "YCITKFGVEAFSDCLRYEMYPLGVKVSVV")
        self.assertEqual(motif.instances[16].tostring(), "YTASKFAVQAFVHTTRRQVAQYGVRVGAV")
        self.assertEqual(motif.instances[17].tostring(), "YCASKFALEGLCESLAVLLLPFGVHLSLI")
        self.assertEqual(motif.instances[18].tostring(), "YPASKASVIGLTHGLGREIIRKNIRVVGV")
        self.assertEqual(motif.instances[19].tostring(), "YSAAKAASINLMEGYRQGLEKYGIGVSVC")
        self.assertEqual(motif.instances[20].tostring(), "YSASKFALDGFFSSIRKEYSVSRVNVSIT")
        self.assertEqual(motif.instances[21].tostring(), "YGASKAALKSLALSVGLELAGSGVRCNVV")
        self.assertEqual(motif.instances[22].tostring(), "YSASKAAVSALTRAAALSCRKQGYAIRVN")
        self.assertEqual(motif.instances[23].tostring(), "YSASKAFVCAFSKALQEEYKAKEVIIQVL")
        self.assertEqual(motif.instances[24].tostring(), "YSASKAAADHLVRAWQRTYRLPSIVSNCS")
        self.assertEqual(motif.instances[25].tostring(), "YGATKWAVRDLMEVLRMESAQEGTNIRTA")
        self.assertEqual(motif.instances[26].tostring(), "YTAAKQAIVGLVRELAFELAPYVRVNGVG")
        self.assertEqual(motif.instances[27].tostring(), "YRMSKAALNMAVRSMSTDLRPEGFVTVLL")
        self.assertEqual(motif.instances[28].tostring(), "MGLAKASLEANVRYMANAMGPEGVRVNAI")
        self.assertEqual(motif.instances[29].tostring(), "YSGTKAAVVNFTSSLAKLAPITGVTAYTV")
        self.assertEqual(motif.instances[30].tostring(), "YGVTKIGVTVLSRIHARKLSEQRKGDKIL")
        self.assertEqual(motif.instances[31].tostring(), "KDSTLFGVSSLSDSLKGDFTSSALRCKEL")
        self.assertEqual(motif.instances[32].tostring(), "YINCVAPLRMTELCLPHLYETGSGRIVNI")
        motif = record.motifs[1]
        self.assertEqual(motif.num_occurrences, 33)
        self.assertAlmostEqual(motif.evalue, 2.3e-159)
        self.assertEqual(motif.alphabet, IUPAC.protein)
        self.assertEqual(motif.name, "Motif 2")
        self.assertEqual(len(motif.instances), 33)
        self.assertAlmostEqual(motif.instances[0].pvalue, 2.44e-23)
        self.assertAlmostEqual(motif.instances[1].pvalue, 5.50e-23)
        self.assertAlmostEqual(motif.instances[2].pvalue, 5.38e-22)
        self.assertAlmostEqual(motif.instances[3].pvalue, 5.65e-20)
        self.assertAlmostEqual(motif.instances[4].pvalue, 1.17e-19)
        self.assertAlmostEqual(motif.instances[5].pvalue, 1.17e-19)
        self.assertAlmostEqual(motif.instances[6].pvalue, 4.74e-19)
        self.assertAlmostEqual(motif.instances[7].pvalue, 9.31e-19)
        self.assertAlmostEqual(motif.instances[8].pvalue, 2.50e-18)
        self.assertAlmostEqual(motif.instances[9].pvalue, 3.45e-18)
        self.assertAlmostEqual(motif.instances[10].pvalue, 5.86e-18)
        self.assertAlmostEqual(motif.instances[11].pvalue, 9.86e-18)
        self.assertAlmostEqual(motif.instances[12].pvalue, 2.47e-17)
        self.assertAlmostEqual(motif.instances[13].pvalue, 3.01e-17)
        self.assertAlmostEqual(motif.instances[14].pvalue, 3.33e-17)
        self.assertAlmostEqual(motif.instances[15].pvalue, 4.06e-17)
        self.assertAlmostEqual(motif.instances[16].pvalue, 4.06e-17)
        self.assertAlmostEqual(motif.instances[17].pvalue, 8.05e-17)
        self.assertAlmostEqual(motif.instances[18].pvalue, 1.90e-16)
        self.assertAlmostEqual(motif.instances[19].pvalue, 2.77e-16)
        self.assertAlmostEqual(motif.instances[20].pvalue, 3.65e-16)
        self.assertAlmostEqual(motif.instances[21].pvalue, 8.31e-16)
        self.assertAlmostEqual(motif.instances[22].pvalue, 4.05e-15)
        self.assertAlmostEqual(motif.instances[23].pvalue, 5.24e-15)
        self.assertAlmostEqual(motif.instances[24].pvalue, 3.00e-14)
        self.assertAlmostEqual(motif.instances[25].pvalue, 8.47e-14)
        self.assertAlmostEqual(motif.instances[26].pvalue, 1.46e-13)
        self.assertAlmostEqual(motif.instances[27].pvalue, 1.46e-13)
        self.assertAlmostEqual(motif.instances[28].pvalue, 1.59e-12)
        self.assertAlmostEqual(motif.instances[29].pvalue, 6.97e-10)
        self.assertAlmostEqual(motif.instances[30].pvalue, 3.15e-09)
        self.assertAlmostEqual(motif.instances[31].pvalue, 2.77e-07)
        self.assertAlmostEqual(motif.instances[32].pvalue, 4.24e-07)
        self.assertEqual(motif.instances[0].sequence_name, 'HDE_CANTR')
        self.assertEqual(motif.instances[1].sequence_name, 'DHII_HUMAN')
        self.assertEqual(motif.instances[2].sequence_name, 'YINL_LISMO')
        self.assertEqual(motif.instances[3].sequence_name, 'HDHA_ECOLI')
        self.assertEqual(motif.instances[4].sequence_name, 'RIDH_KLEAE')
        self.assertEqual(motif.instances[5].sequence_name, 'BUDC_KLETE')
        self.assertEqual(motif.instances[6].sequence_name, 'ENTA_ECOLI')
        self.assertEqual(motif.instances[7].sequence_name, 'AP27_MOUSE')
        self.assertEqual(motif.instances[8].sequence_name, 'DHMA_FLAS1')
        self.assertEqual(motif.instances[9].sequence_name, 'YRTP_BACSU')
        self.assertEqual(motif.instances[10].sequence_name, 'DHGB_BACME')
        self.assertEqual(motif.instances[11].sequence_name, 'DHB3_HUMAN')
        self.assertEqual(motif.instances[12].sequence_name, 'PCR_PEA')
        self.assertEqual(motif.instances[13].sequence_name, 'BDH_HUMAN')
        self.assertEqual(motif.instances[14].sequence_name, 'BA72_EUBSP')
        self.assertEqual(motif.instances[15].sequence_name, 'FIXR_BRAJA')
        self.assertEqual(motif.instances[16].sequence_name, '3BHD_COMTE')
        self.assertEqual(motif.instances[17].sequence_name, '2BHD_STREX')
        self.assertEqual(motif.instances[18].sequence_name, 'HMTR_LEIMA')
        self.assertEqual(motif.instances[19].sequence_name, 'FVT1_HUMAN')
        self.assertEqual(motif.instances[20].sequence_name, 'DHB2_HUMAN')
        self.assertEqual(motif.instances[21].sequence_name, 'LIGD_PSEPA')
        self.assertEqual(motif.instances[22].sequence_name, 'NODG_RHIME')
        self.assertEqual(motif.instances[23].sequence_name, 'DHCA_HUMAN')
        self.assertEqual(motif.instances[24].sequence_name, 'MAS1_AGRRA')
        self.assertEqual(motif.instances[25].sequence_name, 'BPHB_PSEPS')
        self.assertEqual(motif.instances[26].sequence_name, 'GUTD_ECOLI')
        self.assertEqual(motif.instances[27].sequence_name, 'DHES_HUMAN')
        self.assertEqual(motif.instances[28].sequence_name, 'RFBB_NEIGO')
        self.assertEqual(motif.instances[29].sequence_name, 'ADH_DROME')
        self.assertEqual(motif.instances[30].sequence_name, 'FABI_ECOLI')
        self.assertEqual(motif.instances[31].sequence_name, 'YURA_MYXXA')
        self.assertEqual(motif.instances[32].sequence_name, 'CSGA_MYXXA')
        self.assertEqual(motif.instances[0].start, 323)
        self.assertEqual(motif.instances[1].start, 35)
        self.assertEqual(motif.instances[2].start, 6)
        self.assertEqual(motif.instances[3].start, 12)
        self.assertEqual(motif.instances[4].start, 15)
        self.assertEqual(motif.instances[5].start, 3)
        self.assertEqual(motif.instances[6].start, 6)
        self.assertEqual(motif.instances[7].start, 8)
        self.assertEqual(motif.instances[8].start, 15)
        self.assertEqual(motif.instances[9].start, 7)
        self.assertEqual(motif.instances[10].start, 8)
        self.assertEqual(motif.instances[11].start, 49)
        self.assertEqual(motif.instances[12].start, 87)
        self.assertEqual(motif.instances[13].start, 56)
        self.assertEqual(motif.instances[14].start, 7)
        self.assertEqual(motif.instances[15].start, 37)
        self.assertEqual(motif.instances[16].start, 7)
        self.assertEqual(motif.instances[17].start, 7)
        self.assertEqual(motif.instances[18].start, 7)
        self.assertEqual(motif.instances[19].start, 33)
        self.assertEqual(motif.instances[20].start, 83)
        self.assertEqual(motif.instances[21].start, 7)
        self.assertEqual(motif.instances[22].start, 7)
        self.assertEqual(motif.instances[23].start, 5)
        self.assertEqual(motif.instances[24].start, 246)
        self.assertEqual(motif.instances[25].start, 6)
        self.assertEqual(motif.instances[26].start, 3)
        self.assertEqual(motif.instances[27].start, 3)
        self.assertEqual(motif.instances[28].start, 7)
        self.assertEqual(motif.instances[29].start, 7)
        self.assertEqual(motif.instances[30].start, 7)
        self.assertEqual(motif.instances[31].start, 117)
        self.assertEqual(motif.instances[32].start, 52)
        self.assertEqual(motif.instances[0].tostring(), 'KVVLITGAGAGLGKEYAKWFAKYGAKVVV')
        self.assertEqual(motif.instances[1].tostring(), 'KKVIVTGASKGIGREMAYHLAKMGAHVVV')
        self.assertEqual(motif.instances[2].tostring(), 'KVIIITGASSGIGKATALLLAEKGAKLVL')
        self.assertEqual(motif.instances[3].tostring(), 'KCAIITGAGAGIGKEIAITFATAGASVVV')
        self.assertEqual(motif.instances[4].tostring(), 'KVAAITGAASGIGLECARTLLGAGAKVVL')
        self.assertEqual(motif.instances[5].tostring(), 'KVALVTGAGQGIGKAIALRLVKDGFAVAI')
        self.assertEqual(motif.instances[6].tostring(), 'KNVWVTGAGKGIGYATALAFVEAGAKVTG')
        self.assertEqual(motif.instances[7].tostring(), 'LRALVTGAGKGIGRDTVKALHASGAKVVA')
        self.assertEqual(motif.instances[8].tostring(), 'KAAIVTGAAGGIGRATVEAYLREGASVVA')
        self.assertEqual(motif.instances[9].tostring(), 'KTALITGGGRGIGRATALALAKEGVNIGL')
        self.assertEqual(motif.instances[10].tostring(), 'KVVVITGSSTGLGKSMAIRFATEKAKVVV')
        self.assertEqual(motif.instances[11].tostring(), 'QWAVITGAGDGIGKAYSFELAKRGLNVVL')
        self.assertEqual(motif.instances[12].tostring(), 'GNVVITGASSGLGLATAKALAESGKWHVI')
        self.assertEqual(motif.instances[13].tostring(), 'KAVLVTGCDSGFGFSLAKHLHSKGFLVFA')
        self.assertEqual(motif.instances[14].tostring(), 'KVTIITGGTRGIGFAAAKIFIDNGAKVSI')
        self.assertEqual(motif.instances[15].tostring(), 'KVMLLTGASRGIGHATAKLFSEAGWRIIS')
        self.assertEqual(motif.instances[16].tostring(), 'KVALVTGGASGVGLEVVKLLLGEGAKVAF')
        self.assertEqual(motif.instances[17].tostring(), 'KTVIITGGARGLGAEAARQAVAAGARVVL')
        self.assertEqual(motif.instances[18].tostring(), 'PVALVTGAAKRLGRSIAEGLHAEGYAVCL')
        self.assertEqual(motif.instances[19].tostring(), 'AHVVVTGGSSGIGKCIAIECYKQGAFITL')
        self.assertEqual(motif.instances[20].tostring(), 'KAVLVTGGDCGLGHALCKYLDELGFTVFA')
        self.assertEqual(motif.instances[21].tostring(), 'QVAFITGGASGAGFGQAKVFGQAGAKIVV')
        self.assertEqual(motif.instances[22].tostring(), 'RKALVTGASGAIGGAIARVLHAQGAIVGL')
        self.assertEqual(motif.instances[23].tostring(), 'HVALVTGGNKGIGLAIVRDLCRLFSGDVV')
        self.assertEqual(motif.instances[24].tostring(), 'PVILVSGSNRGVGKAIAEDLIAHGYRLSL')
        self.assertEqual(motif.instances[25].tostring(), 'EAVLITGGASGLGRALVDRFVAEAKVAVL')
        self.assertEqual(motif.instances[26].tostring(), 'QVAVVIGGGQTLGAFLCHGLAAEGYRVAV')
        self.assertEqual(motif.instances[27].tostring(), 'TVVLITGCSSGIGLHLAVRLASDPSQSFK')
        self.assertEqual(motif.instances[28].tostring(), 'KNILVTGGAGFIGSAVVRHIIQNTRDSVV')
        self.assertEqual(motif.instances[29].tostring(), 'KNVIFVAGLGGIGLDTSKELLKRDLKNLV')
        self.assertEqual(motif.instances[30].tostring(), 'KRILVTGVASKLSIAYGIAQAMHREGAEL')
        self.assertEqual(motif.instances[31].tostring(), 'IDTNVTGAAATLSAVLPQMVERKRGHLVG')
        self.assertEqual(motif.instances[32].tostring(), 'TSAMLPGLRQGALRRVAHVTSRMGSLAAN')
        handle.close()

    def test_meme_parser_4(self):
        """Test if Motif can parse MEME output files (fourth test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MEME
        handle = open("Motif/meme.protein.tcm.txt")
        record = MEME.read(handle)
        self.assertEqual(record.version, '3.0')
        self.assertEqual(record.datafile, 'farntrans5.s')
        self.assertEqual(record.alphabet, IUPAC.protein)
        self.assertEqual(len(record.sequence_names), 5)
        self.assertEqual(record.sequence_names[0], "RAM1_YEAST")
        self.assertEqual(record.sequence_names[1], "PFTB_RAT")
        self.assertEqual(record.sequence_names[2], "BET2_YEAST")
        self.assertEqual(record.sequence_names[3], "RATRABGERB")
        self.assertEqual(record.sequence_names[4], "CAL1_YEAST")
        self.assertEqual(record.command, 'meme farntrans5.s -mod tcm -protein -nmotifs 2')
        self.assertEqual(len(record.motifs), 2)
        motif = record.motifs[0]
        self.assertEqual(motif.num_occurrences, 24)
        self.assertAlmostEqual(motif.evalue, 2.2e-94)
        self.assertEqual(motif.alphabet, IUPAC.protein)
        self.assertEqual(motif.name, "Motif 1")
        self.assertEqual(len(motif.instances), 24)
        self.assertAlmostEqual(motif.instances[0].pvalue, 7.28e-22)
        self.assertAlmostEqual(motif.instances[1].pvalue, 6.18e-21)
        self.assertAlmostEqual(motif.instances[2].pvalue, 9.17e-20)
        self.assertAlmostEqual(motif.instances[3].pvalue, 1.15e-19)
        self.assertAlmostEqual(motif.instances[4].pvalue, 4.30e-19)
        self.assertAlmostEqual(motif.instances[5].pvalue, 7.36e-19)
        self.assertAlmostEqual(motif.instances[6].pvalue, 8.19e-19)
        self.assertAlmostEqual(motif.instances[7].pvalue, 2.10e-18)
        self.assertAlmostEqual(motif.instances[8].pvalue, 1.43e-17)
        self.assertAlmostEqual(motif.instances[9].pvalue, 3.41e-17)
        self.assertAlmostEqual(motif.instances[10].pvalue, 5.00e-17)
        self.assertAlmostEqual(motif.instances[11].pvalue, 6.64e-17)
        self.assertAlmostEqual(motif.instances[12].pvalue, 1.27e-16)
        self.assertAlmostEqual(motif.instances[13].pvalue, 3.17e-16)
        self.assertAlmostEqual(motif.instances[14].pvalue, 3.47e-16)
        self.assertAlmostEqual(motif.instances[15].pvalue, 4.30e-15)
        self.assertAlmostEqual(motif.instances[16].pvalue, 2.40e-14)
        self.assertAlmostEqual(motif.instances[17].pvalue, 2.81e-14)
        self.assertAlmostEqual(motif.instances[18].pvalue, 7.78e-14)
        self.assertAlmostEqual(motif.instances[19].pvalue, 1.14e-13)
        self.assertAlmostEqual(motif.instances[20].pvalue, 1.33e-13)
        self.assertAlmostEqual(motif.instances[21].pvalue, 3.52e-13)
        self.assertAlmostEqual(motif.instances[22].pvalue, 5.47e-13)
        self.assertAlmostEqual(motif.instances[23].pvalue, 3.11e-10)
        self.assertEqual(motif.instances[0].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[1].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[2].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[3].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[4].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[5].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[6].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[7].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[8].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[9].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[10].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[11].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[12].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[13].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[14].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[15].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[16].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[17].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[18].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[19].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[20].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[21].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[22].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[23].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[0].strand, '+')
        self.assertEqual(motif.instances[1].strand, '+')
        self.assertEqual(motif.instances[2].strand, '+')
        self.assertEqual(motif.instances[3].strand, '+')
        self.assertEqual(motif.instances[4].strand, '+')
        self.assertEqual(motif.instances[5].strand, '+')
        self.assertEqual(motif.instances[6].strand, '+')
        self.assertEqual(motif.instances[7].strand, '+')
        self.assertEqual(motif.instances[8].strand, '+')
        self.assertEqual(motif.instances[9].strand, '+')
        self.assertEqual(motif.instances[10].strand, '+')
        self.assertEqual(motif.instances[11].strand, '+')
        self.assertEqual(motif.instances[12].strand, '+')
        self.assertEqual(motif.instances[13].strand, '+')
        self.assertEqual(motif.instances[14].strand, '+')
        self.assertEqual(motif.instances[15].strand, '+')
        self.assertEqual(motif.instances[16].strand, '+')
        self.assertEqual(motif.instances[17].strand, '+')
        self.assertEqual(motif.instances[18].strand, '+')
        self.assertEqual(motif.instances[19].strand, '+')
        self.assertEqual(motif.instances[20].strand, '+')
        self.assertEqual(motif.instances[21].strand, '+')
        self.assertEqual(motif.instances[22].strand, '+')
        self.assertEqual(motif.instances[23].strand, '+')
        self.assertEqual(motif.instances[0].length, 30)
        self.assertEqual(motif.instances[1].length, 30)
        self.assertEqual(motif.instances[2].length, 30)
        self.assertEqual(motif.instances[3].length, 30)
        self.assertEqual(motif.instances[4].length, 30)
        self.assertEqual(motif.instances[5].length, 30)
        self.assertEqual(motif.instances[6].length, 30)
        self.assertEqual(motif.instances[7].length, 30)
        self.assertEqual(motif.instances[8].length, 30)
        self.assertEqual(motif.instances[9].length, 30)
        self.assertEqual(motif.instances[10].length, 30)
        self.assertEqual(motif.instances[11].length, 30)
        self.assertEqual(motif.instances[12].length, 30)
        self.assertEqual(motif.instances[13].length, 30)
        self.assertEqual(motif.instances[14].length, 30)
        self.assertEqual(motif.instances[15].length, 30)
        self.assertEqual(motif.instances[16].length, 30)
        self.assertEqual(motif.instances[17].length, 30)
        self.assertEqual(motif.instances[18].length, 30)
        self.assertEqual(motif.instances[19].length, 30)
        self.assertEqual(motif.instances[20].length, 30)
        self.assertEqual(motif.instances[21].length, 30)
        self.assertEqual(motif.instances[22].length, 30)
        self.assertEqual(motif.instances[23].length, 30)
        self.assertEqual(motif.instances[0].start, 223)
        self.assertEqual(motif.instances[1].start, 227)
        self.assertEqual(motif.instances[2].start, 275)
        self.assertEqual(motif.instances[3].start, 237)
        self.assertEqual(motif.instances[4].start, 138)
        self.assertEqual(motif.instances[5].start, 179)
        self.assertEqual(motif.instances[6].start, 131)
        self.assertEqual(motif.instances[7].start, 172)
        self.assertEqual(motif.instances[8].start, 276)
        self.assertEqual(motif.instances[9].start, 124)
        self.assertEqual(motif.instances[10].start, 247)
        self.assertEqual(motif.instances[11].start, 272)
        self.assertEqual(motif.instances[12].start, 145)
        self.assertEqual(motif.instances[13].start, 286)
        self.assertEqual(motif.instances[14].start, 296)
        self.assertEqual(motif.instances[15].start, 348)
        self.assertEqual(motif.instances[16].start, 83)
        self.assertEqual(motif.instances[17].start, 189)
        self.assertEqual(motif.instances[18].start, 73)
        self.assertEqual(motif.instances[19].start, 205)
        self.assertEqual(motif.instances[20].start, 198)
        self.assertEqual(motif.instances[21].start, 349)
        self.assertEqual(motif.instances[22].start, 327)
        self.assertEqual(motif.instances[23].start, 24)
        self.assertEqual(motif.instances[0].tostring(), "GGLNGRPSKLPDVCYSWWVLSSLAIIGRLD")
        self.assertEqual(motif.instances[1].tostring(), "GGLNGRPEKLPDVCYSWWVLASLKIIGRLH")
        self.assertEqual(motif.instances[2].tostring(), "GGFQGRENKFADTCYAFWCLNSLHLLTKDW")
        self.assertEqual(motif.instances[3].tostring(), "GGIGGVPGMEAHGGYTFCGLAALVILKKER")
        self.assertEqual(motif.instances[4].tostring(), "GGFGGGPGQYPHLAPTYAAVNALCIIGTEE")
        self.assertEqual(motif.instances[5].tostring(), "GGFGCRPGSESHAGQIYCCTGFLAITSQLH")
        self.assertEqual(motif.instances[6].tostring(), "GSFAGDIWGEIDTRFSFCAVATLALLGKLD")
        self.assertEqual(motif.instances[7].tostring(), "GGFGLCPNAESHAAQAFTCLGALAIANKLD")
        self.assertEqual(motif.instances[8].tostring(), "GGFADRPGDMVDPFHTLFGIAGLSLLGEEQ")
        self.assertEqual(motif.instances[9].tostring(), "GSFQGDRFGEVDTRFVYTALSALSILGELT")
        self.assertEqual(motif.instances[10].tostring(), "GFGSCPHVDEAHGGYTFCATASLAILRSMD")
        self.assertEqual(motif.instances[11].tostring(), "GGISDRPENEVDVFHTVFGVAGLSLMGYDN")
        self.assertEqual(motif.instances[12].tostring(), "GPFGGGPGQLSHLASTYAAINALSLCDNID")
        self.assertEqual(motif.instances[13].tostring(), "GGFQGRCNKLVDGCYSFWQAGLLPLLHRAL")
        self.assertEqual(motif.instances[14].tostring(), "RGFCGRSNKLVDGCYSFWVGGSAAILEAFG")
        self.assertEqual(motif.instances[15].tostring(), "GGLLDKPGKSRDFYHTCYCLSGLSIAQHFG")
        self.assertEqual(motif.instances[16].tostring(), "GGVSASIGHDPHLLYTLSAVQILTLYDSIH")
        self.assertEqual(motif.instances[17].tostring(), "GSFLMHVGGEVDVRSAYCAASVASLTNIIT")
        self.assertEqual(motif.instances[18].tostring(), "GAFAPFPRHDAHLLTTLSAVQILATYDALD")
        self.assertEqual(motif.instances[19].tostring(), "YNGAFGAHNEPHSGYTSCALSTLALLSSLE")
        self.assertEqual(motif.instances[20].tostring(), "GFKTCLEVGEVDTRGIYCALSIATLLNILT")
        self.assertEqual(motif.instances[21].tostring(), "PGLRDKPGAHSDFYHTNYCLLGLAVAESSY")
        self.assertEqual(motif.instances[22].tostring(), "GGFSKNDEEDADLYHSCLGSAALALIEGKF")
        self.assertEqual(motif.instances[23].tostring(), "HNFEYWLTEHLRLNGIYWGLTALCVLDSPE")
        motif = record.motifs[1]
        self.assertEqual(motif.num_occurrences, 21)
        self.assertAlmostEqual(motif.evalue, 3.1e-19)
        self.assertEqual(motif.alphabet, IUPAC.protein)
        self.assertEqual(motif.name, "Motif 2")
        self.assertEqual(len(motif.instances), 21)
        self.assertAlmostEqual(motif.instances[0].pvalue, 2.24e-13)
        self.assertAlmostEqual(motif.instances[1].pvalue, 1.30e-12)
        self.assertAlmostEqual(motif.instances[2].pvalue, 4.20e-12)
        self.assertAlmostEqual(motif.instances[3].pvalue, 9.60e-12)
        self.assertAlmostEqual(motif.instances[4].pvalue, 5.08e-11)
        self.assertAlmostEqual(motif.instances[5].pvalue, 5.01e-10)
        self.assertAlmostEqual(motif.instances[6].pvalue, 6.90e-10)
        self.assertAlmostEqual(motif.instances[7].pvalue, 1.57e-09)
        self.assertAlmostEqual(motif.instances[8].pvalue, 2.34e-09)
        self.assertAlmostEqual(motif.instances[9].pvalue, 4.59e-09)
        self.assertAlmostEqual(motif.instances[10].pvalue, 1.65e-08)
        self.assertAlmostEqual(motif.instances[11].pvalue, 1.65e-08)
        self.assertAlmostEqual(motif.instances[12].pvalue, 1.65e-08)
        self.assertAlmostEqual(motif.instances[13].pvalue, 2.54e-08)
        self.assertAlmostEqual(motif.instances[14].pvalue, 4.58e-08)
        self.assertAlmostEqual(motif.instances[15].pvalue, 5.86e-08)
        self.assertAlmostEqual(motif.instances[16].pvalue, 1.52e-07)
        self.assertAlmostEqual(motif.instances[17].pvalue, 1.91e-07)
        self.assertAlmostEqual(motif.instances[18].pvalue, 4.34e-07)
        self.assertAlmostEqual(motif.instances[19].pvalue, 5.01e-07)
        self.assertAlmostEqual(motif.instances[20].pvalue, 5.78e-07)
        self.assertEqual(motif.instances[0].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[1].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[2].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[3].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[4].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[5].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[6].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[7].sequence_name, "RATRABGERB")
        self.assertEqual(motif.instances[8].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[9].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[10].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[11].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[12].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[13].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[14].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[15].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[16].sequence_name, "PFTB_RAT")
        self.assertEqual(motif.instances[17].sequence_name, "CAL1_YEAST")
        self.assertEqual(motif.instances[18].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[19].sequence_name, "BET2_YEAST")
        self.assertEqual(motif.instances[20].sequence_name, "RAM1_YEAST")
        self.assertEqual(motif.instances[0].strand, '+')
        self.assertEqual(motif.instances[1].strand, '+')
        self.assertEqual(motif.instances[2].strand, '+')
        self.assertEqual(motif.instances[3].strand, '+')
        self.assertEqual(motif.instances[4].strand, '+')
        self.assertEqual(motif.instances[5].strand, '+')
        self.assertEqual(motif.instances[6].strand, '+')
        self.assertEqual(motif.instances[7].strand, '+')
        self.assertEqual(motif.instances[8].strand, '+')
        self.assertEqual(motif.instances[9].strand, '+')
        self.assertEqual(motif.instances[10].strand, '+')
        self.assertEqual(motif.instances[11].strand, '+')
        self.assertEqual(motif.instances[12].strand, '+')
        self.assertEqual(motif.instances[13].strand, '+')
        self.assertEqual(motif.instances[14].strand, '+')
        self.assertEqual(motif.instances[15].strand, '+')
        self.assertEqual(motif.instances[16].strand, '+')
        self.assertEqual(motif.instances[17].strand, '+')
        self.assertEqual(motif.instances[18].strand, '+')
        self.assertEqual(motif.instances[19].strand, '+')
        self.assertEqual(motif.instances[20].strand, '+')
        self.assertEqual(motif.instances[0].length, 14)
        self.assertEqual(motif.instances[1].length, 14)
        self.assertEqual(motif.instances[2].length, 14)
        self.assertEqual(motif.instances[3].length, 14)
        self.assertEqual(motif.instances[4].length, 14)
        self.assertEqual(motif.instances[5].length, 14)
        self.assertEqual(motif.instances[6].length, 14)
        self.assertEqual(motif.instances[7].length, 14)
        self.assertEqual(motif.instances[8].length, 14)
        self.assertEqual(motif.instances[9].length, 14)
        self.assertEqual(motif.instances[10].length, 14)
        self.assertEqual(motif.instances[11].length, 14)
        self.assertEqual(motif.instances[12].length, 14)
        self.assertEqual(motif.instances[13].length, 14)
        self.assertEqual(motif.instances[14].length, 14)
        self.assertEqual(motif.instances[15].length, 14)
        self.assertEqual(motif.instances[16].length, 14)
        self.assertEqual(motif.instances[17].length, 14)
        self.assertEqual(motif.instances[18].length, 14)
        self.assertEqual(motif.instances[19].length, 14)
        self.assertEqual(motif.instances[20].length, 14)
        self.assertEqual(motif.instances[0].start, 254)
        self.assertEqual(motif.instances[1].start, 258)
        self.assertEqual(motif.instances[2].start, 162)
        self.assertEqual(motif.instances[3].start,  66)
        self.assertEqual(motif.instances[4].start, 278)
        self.assertEqual(motif.instances[5].start, 190)
        self.assertEqual(motif.instances[6].start,  55)
        self.assertEqual(motif.instances[7].start, 114)
        self.assertEqual(motif.instances[8].start, 172)
        self.assertEqual(motif.instances[9].start, 330)
        self.assertEqual(motif.instances[10].start, 126)
        self.assertEqual(motif.instances[11].start, 268)
        self.assertEqual(motif.instances[12].start, 220)
        self.assertEqual(motif.instances[13].start, 229)
        self.assertEqual(motif.instances[14].start, 330)
        self.assertEqual(motif.instances[15].start, 239)
        self.assertEqual(motif.instances[16].start, 121)
        self.assertEqual(motif.instances[17].start, 362)
        self.assertEqual(motif.instances[18].start, 107)
        self.assertEqual(motif.instances[19].start, 155)
        self.assertEqual(motif.instances[20].start, 180)
        self.assertEqual(motif.instances[0].tostring(), "INYEKLTEFILKCQ")
        self.assertEqual(motif.instances[1].tostring(), "IDREKLRSFILACQ")
        self.assertEqual(motif.instances[2].tostring(), "INVEKAIEFVLSCM")
        self.assertEqual(motif.instances[3].tostring(), "MNKEEILVFIKSCQ")
        self.assertEqual(motif.instances[4].tostring(), "INVEKLLEWSSARQ")
        self.assertEqual(motif.instances[5].tostring(), "IDTEKLLGYIMSQQ")
        self.assertEqual(motif.instances[6].tostring(), "FVKEEVISFVLSCW")
        self.assertEqual(motif.instances[7].tostring(), "INVDKVVAYVQSLQ")
        self.assertEqual(motif.instances[8].tostring(), "INREKLLQYLYSLK")
        self.assertEqual(motif.instances[9].tostring(), "FNKHALRDYILYCC")
        self.assertEqual(motif.instances[10].tostring(), "LDKRSLARFVSKCQ")
        self.assertEqual(motif.instances[11].tostring(), "LNLKSLLQWVTSRQ")
        self.assertEqual(motif.instances[12].tostring(), "DLFEGTAEWIARCQ")
        self.assertEqual(motif.instances[13].tostring(), "ELTEGVLNYLKNCQ")
        self.assertEqual(motif.instances[14].tostring(), "FHQQALQEYILMCC")
        self.assertEqual(motif.instances[15].tostring(), "KFKEDTITWLLHRQ")
        self.assertEqual(motif.instances[16].tostring(), "IVATDVCQFLELCQ")
        self.assertEqual(motif.instances[17].tostring(), "IPQEIFNDFSKRCC")
        self.assertEqual(motif.instances[18].tostring(), "DRKVRLISFIRGNQ")
        self.assertEqual(motif.instances[19].tostring(), "EVVDPAVDFVLKCY")
        self.assertEqual(motif.instances[20].tostring(), "IDRKGIYQWLISLK")
        handle.close()

 
class TestMAST(unittest.TestCase):

    def test_mast_parser_1(self):
        """Test if Motif can parse MAST output files (first test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MAST
        handle = open("Motif/mast.dna.oops.txt")
        record = MAST.read(handle)
        self.assertEqual(record.version, "3.0")
        self.assertEqual(record.database, "INO_up800.s")
        self.assertEqual(record.alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(len(record.motifs), 2)
        self.assertEqual(record.motifs[0].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(record.motifs[0].length, 12)
        self.assertEqual(record.motifs[0].name, "1")
        self.assertEqual(record.motifs[1].alphabet, IUPAC.unambiguous_dna)
        self.assertEqual(record.motifs[1].length, 10)
        self.assertEqual(record.motifs[1].name, "2")
        self.assertEqual(len(record.sequences), 7)
        self.assertEqual(record.sequences[0], "ACC1")
        self.assertEqual(record.sequences[1], "CHO1")
        self.assertEqual(record.sequences[2], "INO1")
        self.assertEqual(record.sequences[3], "FAS1")
        self.assertEqual(record.sequences[4], "OPI3")
        self.assertEqual(record.sequences[5], "CHO2")
        self.assertEqual(record.sequences[6], "FAS2")
        self.assertEqual(record.diagrams["ACC1"], "82_[+1]_137_[+2]_559")
        self.assertEqual(record.diagrams["CHO1"], "152_[+2]_396_[-2]_42_[+1]_17_[+1]_149")
        self.assertEqual(record.diagrams["INO1"], "282_[-2]_327_[-1]_55_[+1]_102")
        self.assertEqual(record.diagrams["FAS1"], "43_[+2]_41_[+1]_694")
        self.assertEqual(record.diagrams["OPI3"], "185_[-2]_144_[+1]_449")
        self.assertEqual(record.diagrams["CHO2"], "353_[+1]_47_[-2]_378")
        self.assertEqual(record.diagrams["FAS2"], "184_[-2]_372_[+1]_222")
        handle.close()

    def test_mast_parser_2(self):
        """Test if Motif can parse MAST output files (second test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MAST
        handle = open("Motif/mast.protein.oops.txt")
        record = MAST.read(handle)
        self.assertEqual(record.version, "3.0")
        self.assertEqual(record.database, "adh.s")
        self.assertEqual(record.alphabet, IUPAC.protein)
        self.assertEqual(len(record.motifs), 2)
        self.assertEqual(record.motifs[0].alphabet, IUPAC.protein)
        self.assertEqual(record.motifs[0].length, 29)
        self.assertEqual(record.motifs[0].name, "1")
        self.assertEqual(record.motifs[1].alphabet, IUPAC.protein)
        self.assertEqual(record.motifs[1].length, 29)
        self.assertEqual(record.motifs[1].name, "2")
        self.assertEqual(len(record.sequences), 33)
        self.assertEqual(record.sequences[0], "BUDC_KLETE")
        self.assertEqual(record.sequences[1], "YRTP_BACSU")
        self.assertEqual(record.sequences[2], "AP27_MOUSE")
        self.assertEqual(record.sequences[3], "HDE_CANTR")
        self.assertEqual(record.sequences[4], "HDHA_ECOLI")
        self.assertEqual(record.sequences[5], "DHII_HUMAN")
        self.assertEqual(record.sequences[6], "FIXR_BRAJA")
        self.assertEqual(record.sequences[7], "DHGB_BACME")
        self.assertEqual(record.sequences[8], "NODG_RHIME")
        self.assertEqual(record.sequences[9], "RIDH_KLEAE")
        self.assertEqual(record.sequences[10], "YINL_LISMO")
        self.assertEqual(record.sequences[11], "DHMA_FLAS1")
        self.assertEqual(record.sequences[12], "HMTR_LEIMA")
        self.assertEqual(record.sequences[13], "2BHD_STREX")
        self.assertEqual(record.sequences[14], "ENTA_ECOLI")
        self.assertEqual(record.sequences[15], "DHB2_HUMAN")
        self.assertEqual(record.sequences[16], "BDH_HUMAN")
        self.assertEqual(record.sequences[17], "BA72_EUBSP")
        self.assertEqual(record.sequences[18], "FVT1_HUMAN")
        self.assertEqual(record.sequences[19], "GUTD_ECOLI")
        self.assertEqual(record.sequences[20], "DHB3_HUMAN")
        self.assertEqual(record.sequences[21], "3BHD_COMTE")
        self.assertEqual(record.sequences[22], "LIGD_PSEPA")
        self.assertEqual(record.sequences[23], "DHES_HUMAN")
        self.assertEqual(record.sequences[24], "RFBB_NEIGO")
        self.assertEqual(record.sequences[25], "BPHB_PSEPS")
        self.assertEqual(record.sequences[26], "YURA_MYXXA")
        self.assertEqual(record.sequences[27], "PCR_PEA")
        self.assertEqual(record.sequences[28], "DHCA_HUMAN")
        self.assertEqual(record.sequences[29], "ADH_DROME")
        self.assertEqual(record.sequences[30], "MAS1_AGRRA")
        self.assertEqual(record.sequences[31], "FABI_ECOLI")
        self.assertEqual(record.sequences[32], "CSGA_MYXXA")
        self.assertEqual(record.diagrams["BUDC_KLETE"], "2_[2]_120_[1]_61")
        self.assertEqual(record.diagrams["YRTP_BACSU"], "6_[2]_119_[1]_55")
        self.assertEqual(record.diagrams["AP27_MOUSE"], "7_[2]_112_[1]_67")
        self.assertEqual(record.diagrams["HDE_CANTR"], "8_[2]_125_[1]_131_[2]_115_[1]_411")
        self.assertEqual(record.diagrams["HDHA_ECOLI"], "11_[2]_74_[1]_15_[1]_68")
        self.assertEqual(record.diagrams["DHII_HUMAN"], "34_[2]_119_[1]_81")
        self.assertEqual(record.diagrams["FIXR_BRAJA"], "36_[2]_123_[1]_61")
        self.assertEqual(record.diagrams["DHGB_BACME"], "7_[2]_123_[1]_74")
        self.assertEqual(record.diagrams["NODG_RHIME"], "6_[2]_116_[1]_65")
        self.assertEqual(record.diagrams["RIDH_KLEAE"], "14_[2]_116_[1]_61")
        self.assertEqual(record.diagrams["YINL_LISMO"], "5_[2]_75_[2]_15_[1]_66")
        self.assertEqual(record.diagrams["DHMA_FLAS1"], "14_[2]_121_[1]_77")
        self.assertEqual(record.diagrams["HMTR_LEIMA"], "6_[2]_157_[1]_66")
        self.assertEqual(record.diagrams["2BHD_STREX"], "6_[2]_116_[1]_75")
        self.assertEqual(record.diagrams["ENTA_ECOLI"], "5_[2]_109_[1]_76")
        self.assertEqual(record.diagrams["DHB2_HUMAN"], "82_[2]_120_[1]_127")
        self.assertEqual(record.diagrams["BDH_HUMAN"], "55_[2]_123_[1]_107")
        self.assertEqual(record.diagrams["BA72_EUBSP"], "6_[2]_121_[1]_64")
        self.assertEqual(record.diagrams["FVT1_HUMAN"], "32_[2]_124_[1]_118")
        self.assertEqual(record.diagrams["GUTD_ECOLI"], "2_[2]_122_[1]_77")
        self.assertEqual(record.diagrams["DHB3_HUMAN"], "48_[2]_120_[1]_84")
        self.assertEqual(record.diagrams["3BHD_COMTE"], "6_[2]_115_[1]_74")
        self.assertEqual(record.diagrams["LIGD_PSEPA"], "6_[2]_121_[1]_120")
        self.assertEqual(record.diagrams["DHES_HUMAN"], "2_[2]_50_[2]_44_[1]_144")
        self.assertEqual(record.diagrams["RFBB_NEIGO"], "6_[2]_129_[1]_153")
        self.assertEqual(record.diagrams["BPHB_PSEPS"], "5_[2]_118_[1]_94")
        self.assertEqual(record.diagrams["YURA_MYXXA"], "65_[2]_22_[2]_14_[1]_70")
        self.assertEqual(record.diagrams["PCR_PEA"], "25_[1]_32_[2]_284")
        self.assertEqual(record.diagrams["DHCA_HUMAN"], "4_[2]_159_[1]_55")
        self.assertEqual(record.diagrams["ADH_DROME"], "6_[2]_116_[1]_75")
        self.assertEqual(record.diagrams["MAS1_AGRRA"], "245_[2]_74_[1]_14_[1]_56")
        self.assertEqual(record.diagrams["FABI_ECOLI"], "6_[2]_123_[1]_75")
        self.assertEqual(record.diagrams["CSGA_MYXXA"], "51_[2]_7_[1]_50")
        handle.close()

    def test_mast_parser_3(self):
        """Test if Motif can parse MAST output files (third test)
        """
        from Bio.Alphabet import IUPAC
        from Bio.Motif.Parsers import MAST
        handle = open("Motif/mast.protein.tcm.txt")
        record = MAST.read(handle)
        self.assertEqual(record.version, "3.0")
        self.assertEqual(record.database, "farntrans5.s")
        self.assertEqual(record.alphabet, IUPAC.protein)
        self.assertEqual(len(record.motifs), 2)
        self.assertEqual(record.motifs[0].alphabet, IUPAC.protein)
        self.assertEqual(record.motifs[0].length, 30)
        self.assertEqual(record.motifs[0].name, "1")
        self.assertEqual(record.motifs[1].alphabet, IUPAC.protein)
        self.assertEqual(record.motifs[1].length, 14)
        self.assertEqual(record.motifs[1].name, "2")
        self.assertEqual(len(record.sequences), 5)
        self.assertEqual(record.sequences[0], "BET2_YEAST")
        self.assertEqual(record.sequences[1], "RATRABGERB")
        self.assertEqual(record.sequences[2], "CAL1_YEAST")
        self.assertEqual(record.sequences[3], "PFTB_RAT")
        self.assertEqual(record.sequences[4], "RAM1_YEAST")
        self.assertEqual(record.diagrams["BET2_YEAST"], "6_[2]_3_[1]_1_[2]_4_[1]_4_[2]_3_[1]_1_[2]_3_[1]_21_[1]_1_[2]_4_[1]_24")
        self.assertEqual(record.diagrams["RATRABGERB"], "65_[2]_3_[1]_1_[2]_3_[1]_1_[2]_3_[1]_18_[1]_1_[2]_4_[1]_26")
        self.assertEqual(record.diagrams["CAL1_YEAST"], "125_[2]_50_[2]_1_[1]_4_[2]_22_[1]_22_[1]_5_[2]_1")
        self.assertEqual(record.diagrams["PFTB_RAT"], "120_[2]_3_[1]_4_[2]_3_[1]_1_[2]_3_[1]_1_[2]_4_[1]_14_[2]_4_[1]_60")
        self.assertEqual(record.diagrams["RAM1_YEAST"], "144_[1]_5_[2]_4_[1]_1_[2]_4_[1]_1_[2]_4_[1]_4_[2]_5_[1]_35_[2]_4")
        handle.close()


class MotifTestPWM(unittest.TestCase):
    def setUp(self):
        from Bio import Seq
        self.m=Motif.read(open("Motif/SRF.pfm"),"jaspar-pfm")
        self.s=Seq.Seq("ACGTGTGCGTAGTGCGT",self.m.alphabet)

    def test_simple(self):
        """Test if Motif PWM scoring works."""
        result = self.m.scanPWM(self.s)
        self.assertEqual(6, len(result))
        # The fast C-code in Bio/Motif/_pwm.c stores all results as 32-bit
        # floats; the slower Python code in Bio/Motif/_Motif.py uses 64-bit
        # doubles. The C-code and Python code results will therefore not be
        # exactly equal. Test the first 5 decimal places only to avoid either
        # the C-code or the Python code to inadvertently fail this test.
        self.assertAlmostEqual(result[0], -29.18363571, places=5)
        self.assertAlmostEqual(result[1], -38.3365097, places=5)
        self.assertAlmostEqual(result[2], -29.17756271, places=5)
        self.assertAlmostEqual(result[3], -38.04542542, places=5)
        self.assertAlmostEqual(result[4], -20.3014183, places=5)
        self.assertAlmostEqual(result[5], -25.18009186, places=5)


if __name__ == "__main__":
    runner = unittest.TextTestRunner(verbosity = 2)
    unittest.main(testRunner=runner)
