/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <openobex/obex.h>

#include "btobexmain.h"
#include "btobexserver.h"


/* 
 * For retaining state while using the custom data transport 
 * for running an obex server
 */
typedef struct {
    int serverfd;
    int fd;
    struct sockaddr_rc rfcomm;
    uint8_t inputbuf[500];     /* is this a good size? */
} lb_customdata_t;


typedef struct {
	int serverdone;
	/*uint8_t *putfilename;*/
	int current_op;
	
	int last_evt;
	int last_cmd;
	
	btobexserver_usrconfig_t *usrconfig;
} obexconfig_t;



int btobexserver_trans_connect(obex_t *handle, void * customdata)
{
	/* do nothing, we're only running custom transport for servers, and
	this only needs to be implemented for clients */
	return 0;
}

int btobexserver_trans_disconnect(obex_t *handle, void * customdata)
{
	/* do nothing, we're only running custom transport for servers, and
		this only needs to be implemented for clients */
	return 0;
}    

int btobexserver_trans_write(obex_t *handle, void * customdata, uint8_t *buf, int buflen)
{
	lb_customdata_t *data;
	int actual;
	
	if (!handle) return -1;
	if (!customdata) return -1;    
	
	/* pass on received data to the client socket */
	data = customdata;
	actual = write(data->fd, buf, buflen);
	return actual;
}    

int btobexserver_trans_listen(obex_t *handle, void *customdata)
{
	/* do nothing, our server that's passed from Python should already be 
	bound and listening (and this method shouldn't get called anyway) */
	return 0;
}

int btobexserver_trans_accept(lb_customdata_t *customdata)
{
	uint32_t addrlen;
	if (!customdata) return -1;
	
	// First accept the connection and get the new client socket.
	addrlen = sizeof(struct sockaddr_rc);
	customdata->fd = accept(customdata->serverfd, (struct sockaddr *) &customdata->rfcomm,
							&addrlen);
	
	if (customdata->fd < 0) {
		DEBUG("[btobexserver_trans_accept] Invalid file descriptor for accepted client socket\n");
		return -1;
	}
	return 0;
}

// modified from openobex obex_transport.c obex_transport_handle_input
int btobexserver_trans_handleinput(obex_t *handle, void *customdata, int timeout)
{
	struct timeval time;
	lb_customdata_t *data;
	fd_set fdset;
	int highestfd = 0;
	int ret = -1;
	
	if (!handle) return -1;
	if (!customdata) return -1;
	
	data = customdata;
	
	/* Check if we have any fd's to do select on. */
	if(data->fd < 0 && data->serverfd < 0) {
		DEBUG("[btobexserver_trans_handleinput] No valid socket is open\n");
		return -1;
	}
	
	time.tv_sec = timeout;
	time.tv_usec = 0;
	
	/* Add the fd's to the set. */
	FD_ZERO(&fdset);
	if (data->fd >= 0) {
		FD_SET(data->fd, &fdset);
		highestfd = data->fd;
	}
	if (data->serverfd >= 0) {
		FD_SET(data->serverfd, &fdset);
		if (data->serverfd > highestfd)
			highestfd = data->serverfd;
	}
	
	/* Wait for input */
	ret = select(highestfd+1, &fdset, NULL, NULL, &time);
	
	/* Check if this is a timeout (0) or error (-1) */
	if (ret <= 0) return ret;
	
	if( (data->fd >= 0) && FD_ISSET(data->fd, &fdset)) {
			
		int len;            
			
		len = read(data->fd, &data->inputbuf, sizeof(data->inputbuf));
		if (len <= 0) return len;
		
		ret = OBEX_CustomDataFeed(handle, data->inputbuf, len);
		
	} else if( (data->serverfd >= 0) && FD_ISSET(data->serverfd, &fdset)) {
		
		//DEBUG("Data available on server socket\n");
			
		// accept a new client socket
		ret = btobexserver_trans_accept(data);
		
		if (ret < 0) {
			DEBUG("[btobexserver_trans_handleinput] Could not accept new socket\n");
			return ret;
		}
		
		} else {
			ret = -1;
		}
		
	return ret;
}

/*
 * Called when a PUT request is completed.
 */
static void btobexserver_put_done(obex_t *handle)
{
	obexconfig_t *config;
	config = OBEX_GetUserData(handle);	
	
	/* callback - PUT is complete */
	if (config->usrconfig->put_complete) {
		config->usrconfig->put_complete(handle);
	}
	config->current_op = -1;	
}


/* 
 * Handle a PUT request.
 */
static void btobexserver_put(obex_t *handle, obex_object_t *object)
{
	obex_headerdata_t hv;
	uint8_t hi;
	uint32_t hv_size;
	
	uint8_t *filename = NULL;	
	const uint8_t *body = NULL;
	int body_size = -1;
	
	int isfirstpacket = FALSE;
	
	obexconfig_t *config;
	
	DEBUG("[btobexserver_put] entry.\n");
	
	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hv_size)) {	
		switch (hi) {
			case OBEX_HDR_NAME:
				/*DEBUG("[btobexserver_put] Found name header\n");*/
				if ( (filename = malloc(hv_size / 2)) ) {
					OBEX_UnicodeToChar(filename, hv.bs, hv_size);
				}
				break;				
			case OBEX_HDR_BODY_END:
				/*DEBUG("[btobexserver_put] Found end-of-body header\n");*/
				body = hv.bs;
				body_size = hv_size;
				break;			
			case OBEX_HDR_BODY:
				/*DEBUG("[btobexserver_put] Found body header\n");*/
				body = hv.bs;
				body_size = hv_size;
				break;
			default:
				DEBUG("[btobexserver_put] Skipped header %d\n", hi);
		}
	}

	config = OBEX_GetUserData(handle);	
	
	/* set flag for multipacket PUT transfers */
	if (config->current_op != OBEX_CMD_PUT) {
		isfirstpacket = TRUE;
		config->current_op = OBEX_CMD_PUT;
	}
	
	/* callback - a PUT has started */	
	/* filename is NULL if client didn't give one, or we couldn't read it */
	if (isfirstpacket) {
		DEBUG("[btobexserver_put] This is 1st PUT packet\n");
		if (config->usrconfig->put_requested) {
			if ( !(config->usrconfig->put_requested(handle, (char*)filename)) ) {
				DEBUG("[btobexserver_put] User denied PUT request, responding 'forbidden'\n");
				OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
				return;
			}
		}
	}
	
	/* callback - PUT received some data */
	/* the body data is NULL if client did not send body data */
	if (body != NULL && config->usrconfig->put_progressed) {
		if ( !(config->usrconfig->put_progressed(handle, body, body_size)) ) {
			DEBUG("[btobexserver_put] User denied PUT request, responding 'forbidden'\n");
			OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
			return;
		}		
	}	
}


static void btobexserver_done(obex_t *handle, obex_object_t *object, int obex_cmd, int obex_rsp)
{
	obexconfig_t *config;	
	DEBUG("[btobexserver_put] Completed client request (0x%02x)\n", obex_cmd);
	
	switch (obex_cmd) {
		case OBEX_CMD_CONNECT:
			config = OBEX_GetUserData(handle);
			if (config->usrconfig->connect_complete) {
				config->usrconfig->connect_complete(handle);
			}
			break;
		case OBEX_CMD_DISCONNECT:
			config = OBEX_GetUserData(handle);
			if (config->usrconfig->disconnect_complete) {
				config->usrconfig->disconnect_complete(handle);
			}
			break;	
		case OBEX_CMD_PUT:
			btobexserver_put_done(handle);
			break;		
	}		
}

/*
 * Called when a new client request is arriving. The request can be rejected
 * here before it is processed.
 */
static void btobexserver_reqhinted(obex_t *handle, obex_object_t *object, int event, int cmd)
{
	DEBUG("[btobexserver_reqhinted] Incoming request (0x%02x)\n", cmd);
	
	switch(cmd)	{
		case OBEX_CMD_CONNECT:
		case OBEX_CMD_DISCONNECT:
		case OBEX_CMD_PUT:
			OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
			break;
		default:
			/* should implement OBEX_CMD_ABORT! */
			
			DEBUG("[btobexserver_reqhinted] Denied %02x request (we haven't implemented this)\n", cmd);
			OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED, OBEX_RSP_NOT_IMPLEMENTED);
			break;
	}
}

/*
 * Called when an obex event occurs.
 */
void btobexserver_event(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
	obexconfig_t *config;
	config = OBEX_GetUserData(handle);
	config->last_evt = event;	
	config->last_cmd = obex_cmd;
	
	switch (event)  {
		case OBEX_EV_PROGRESS:
			break;
		case OBEX_EV_ABORT:
			/* should abort saving/getting file on my server */
			break;
		case OBEX_EV_REQHINT:
			/* a new request is arriving */
			btobexserver_reqhinted(handle, object, event, obex_cmd);
			break;
		case OBEX_EV_REQ:
			/* an incoming request is here */
			switch (obex_cmd) {
				case OBEX_CMD_PUT:
					btobexserver_put(handle, object);
					break;
			}
			break;
		case OBEX_EV_REQDONE:
			/* a request is complete */
			btobexserver_done(handle, object, obex_cmd, obex_rsp);
			break;
		case OBEX_EV_LINKERR:
			/* transport broken */
			/*OBEX_TransportDisconnect(handle);*/
			DEBUG("[btobexserver_event] Transport link broken!\n");
			
			if (config->usrconfig->transport_broken) {
				config->usrconfig->transport_broken(handle);
			}
			break;
		default:
			DEBUG("[btobexserver_event] Unknown event (%d)\n", event);
			break;
	}
}

void btobexserver_initconfig(obexconfig_t *config) {
	config->serverdone = FALSE;
	config->current_op = -1;
	config->usrconfig = NULL;	
}

obex_t *btobexserver_init(int serverfd, unsigned int flags)
{
	obex_t *handle;
	obexconfig_t *config;
	lb_customdata_t *customdata;
	int err;
	
	/* Create obex handle with custom transport, not bluetooth transport.
	The reason for using the custom transport is because I want to 
	use an existing server; if I use the openobex built-in bluetooth 
	transport instead it will try to create a new server socket.	*/
	handle = OBEX_Init(OBEX_TRANS_CUSTOM, btobexserver_event, flags);
	if (!handle) return NULL;
	
	// set server config data
	config = malloc(sizeof(obexconfig_t));
	if (!config) {
		DEBUG("[btobexserver_init] Error allocating internal server config data\n");
		return NULL;
	}
	btobexserver_initconfig(config);
	OBEX_SetUserData(handle, config);
		
	// set the custom transport (tell it which functions to call)
	obex_ctrans_t btobexserver_trans;
	btobexserver_trans.listen = &btobexserver_trans_listen;
	btobexserver_trans.handleinput = &btobexserver_trans_handleinput;
	btobexserver_trans.disconnect = &btobexserver_trans_disconnect;
	btobexserver_trans.connect = &btobexserver_trans_connect;
	btobexserver_trans.write = &btobexserver_trans_write;
	
	err = OBEX_RegisterCTransport(handle, &btobexserver_trans);
	if (err < 0) {
		DEBUG("[btobexserver_init] Error registering custom OBEX transport\n");
		return NULL;    
	}	
		
	// Don't call BtOBEX_ServerRegister because that tries to make
	// a new server socket, but we want to use the existing one we've
	// been given.
		
	// set some custom data, noting the server fd
	// (the custom data is passed to all the custom transport functions)
	customdata = malloc(sizeof(lb_customdata_t));
	if (!customdata) {
		DEBUG("[btobexserver_init] Error allocating internal server custom transport data\n");
		return NULL;
	}	
	customdata->serverfd = serverfd;
	customdata->fd = 0;
	OBEX_SetCustomData(handle, customdata);	
	
	return handle;
}

void btobexserver_cleanup(obex_t *handle)
{
	obexconfig_t *config;
	lb_customdata_t *customdata;
	
	DEBUG("[btobexserver_cleanup] entry.\n");
	
	config = OBEX_GetUserData(handle);
	if (config) free(config);
	OBEX_SetUserData(handle, NULL);	
	
	customdata = OBEX_GetCustomData(handle);
	if (customdata) free(customdata);
	OBEX_SetCustomData(handle, NULL);		

	DEBUG("[btobexserver_cleanup] exit.\n");
}


void btobexserver_setusrconfig(obex_t *handle, btobexserver_usrconfig_t *usrconfig)
{
	obexconfig_t *config;
	
	config = OBEX_GetUserData(handle);
	config->usrconfig = usrconfig;
}

btobexserver_usrconfig_t *btobexserver_getusrconfig(obex_t *handle)
{
	obexconfig_t *config;
	
	config = OBEX_GetUserData(handle);
	return (config) ? config->usrconfig : NULL;
}

/* Returns -1 on error, 0 on timeout, or positive on success. */
int btobexserver_run(obex_t *handle)
{
	int err = 1;
	obexconfig_t *config;
	config = OBEX_GetUserData(handle);
	
	config->serverdone = FALSE;
	DEBUG("[btobexserver_run] Running OBEX server\n");
	
	while (!config->serverdone) {
        	err = OBEX_HandleInput(handle, config->usrconfig->timeout);
		if (err <= 0) {
			if (config->usrconfig->error_handling_input) {
				config->usrconfig->error_handling_input(handle, 
									err,
									config->last_evt, 
								        config->last_cmd);
			}
		}
	}
	DEBUG("[btobexserver_run] Stopped OBEX server\n");
	
	DEBUG("[btobexserver_cleanup] Closing transport\n");
	
	// close the transport connection
	OBEX_TransportDisconnect(handle);
	
	return err;
}

void btobexserver_stop(obex_t *handle)
{
	obexconfig_t *config;
	config = OBEX_GetUserData(handle);
	config->serverdone = TRUE;	
}


