/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "Python.h"     // includes stdio, string, errno, stdlib

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <unistd.h>

#include <sys/socket.h>
//#include <sys/types.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>

#include <openobex/obex.h>

#include "btobexmain.h"
#include "btobexclient.h"
#include "btobexserver.h"

#define DEBUG_NAME "[lightblue_obex_util]"


typedef struct {
	FILE *putfile;
	int got_file;
} lb_usrdata;


/**************************************************/
/*** Client operations ***/

/**
 * Connects an obex client to a remote device.
 */
int lb_obexclient_connect(obex_t *handle, bdaddr_t *bdaddr, uint8_t channel)
{
    int err = 0;
    
    Py_BEGIN_ALLOW_THREADS
    err = btobexclient_connect(handle, bdaddr, channel);
    Py_END_ALLOW_THREADS
    
    if (err) {
        PyErr_SetString(PyExc_IOError, "Error connecting OBEX session");
        return err;
    }
    return 0;
}

/**
 * Disconnects an obex client.
 */
int lb_obexclient_disconnect(obex_t *handle)
{
    int err = 0;
    
    Py_BEGIN_ALLOW_THREADS
    err = btobexclient_disconnect(handle);
    Py_END_ALLOW_THREADS

    if (err) {
	PyErr_SetString(PyExc_IOError, "Error disconnecting OBEX session");
	return err;
    }
    return 0;
}

/**
 * Performs a PUT on an obex client.
 */
int lb_obexclient_put(obex_t *handle, char *name, FILE *file, uint32_t filesize)
{
    int err = 0;
        
    Py_BEGIN_ALLOW_THREADS
    err = btobexclient_put(handle, name, file, filesize);
    Py_END_ALLOW_THREADS
    
    if (err) {
        PyErr_SetString(PyExc_IOError, "Error sending file");
        return err;
    }
    return 0;    
}

void lb_client_cleanup(obex_t *handle)
{
	btobexclient_cleanup(handle);
	
	/* delete the OBEX handle */
	OBEX_Cleanup(handle);
}

/**
 * (method for Python interface)
 * Sends a file to a remote device.
 * Arguments:
 *  - bluetooth device address (string)
 *  - channel (uint8_t / unsigned char)
 *  - file object containing data to be sent
 *  - size of data to be sent 
 */
static PyObject* lb_send_file(PyObject *self, PyObject *args)
{
	char *bdaddrstr;
	uint8_t channel;
	char *putname;
	PyObject *fileobj;
	uint32_t filesize;
	FILE *putfile;
	
	obex_t *handle;
	bdaddr_t bdaddr;
	int err = 0;
	
	if (!PyArg_ParseTuple(args, "sBsOI", &bdaddrstr, &channel, &putname, &fileobj, &filesize)) {
		return NULL;
		PyErr_Clear();
		if (!PyFile_Check(fileobj))
			return NULL;
	}    
	
	putfile = PyFile_AsFile(fileobj);
	if (!putfile) {
		PyErr_SetString(PyExc_IOError, "Error getting file object as C FILE*");
		return NULL;
	}			
	
	// create obex handle
	if( !(handle = btobexclient_init(0)) ) {
		PyErr_SetString(PyExc_IOError, "Error creating OBEX client handle");
		return NULL;
	}
	
	// connect
	str2ba(bdaddrstr, &bdaddr);
	err = lb_obexclient_connect(handle, &bdaddr, channel);
	if (err) {
		PyErr_SetString(PyExc_IOError, "Error connecting to server");
		lb_client_cleanup(handle);
		return NULL;
	}
	
	// put file
	err = lb_obexclient_put(handle, putname, putfile, filesize);
	if (err) {
		// Disconnect before returning.
		// Fetch & restore error in case disconnection causes an error.
		PyObject *ptype;
		PyObject *pvalue;
		PyObject *ptraceback;
		PyErr_Fetch(&ptype, &pvalue, &ptraceback);
		
		lb_obexclient_disconnect(handle);
		
		PyErr_Restore(ptype, pvalue, ptraceback);
		
		lb_client_cleanup(handle);
		return NULL;
	}
	
	// disconnect
	err = lb_obexclient_disconnect(handle);
	if (err) {
		// since file has already been sent, don't raise exception,
		// just print warning
		DEBUG(DEBUG_NAME " Warning: could not disconnect after sending file.\n");
		PyErr_Clear();      // clear error data from disconnection error
	}
	
	lb_client_cleanup(handle);
	return Py_BuildValue("");
}


/**************************************************/
/*** Server operations ***/


static void lb_server_cleanup(obex_t *handle) 
{
	btobexserver_cleanup(handle);	
	
	/* delete the OBEX handle */
	OBEX_Cleanup(handle);	
}

/** 
 * called when a client has connected 
 */
void lb_server_connect_complete(obex_t *handle)
{
	btobexserver_usrconfig_t *config;
	config = btobexserver_getusrconfig(handle);
	
	/* Before connecting the timeout was set to 0 (no timeout). 
	Now that a client has connected, set a timeout value */
	config->timeout = 30.0;
}

/** 
 * called when a client has disconnected 
 */
void lb_server_disconnect_complete(obex_t *handle)
{
	/* stop the server now */
	DEBUG(DEBUG_NAME " Client disconnected, stopping server.\n"); 
	btobexserver_stop(handle);
}


/**
 * Called when a PUT is first requested by a client.
 * Given name may be NULL if client did not provide one.
 */
int lb_server_put_requested(obex_t *handle, char *name)
{
	btobexserver_usrconfig_t *config;
	lb_usrdata *usrdata;
	
	config = btobexserver_getusrconfig(handle);
	usrdata = config->usrdata;
	
	if (usrdata->got_file) {
		DEBUG(DEBUG_NAME " Already received a file, rejecting PUT.\n");
		return FALSE;
	}	
	
	if (!usrdata->putfile) {
		DEBUG(DEBUG_NAME " usrdata FILE pointer is NULL, rejecting PUT.\n");
		return FALSE;
	}
	
	DEBUG(DEBUG_NAME " Allowing PUT request\n");
	return TRUE;
}

/**
 * Called when body data has been received in a PUT.
 * Given buf could be NULL if client did not send body data.
 */
int lb_server_put_progressed(obex_t *handle, const uint8_t *buf, uint32_t buflen)
{
	btobexserver_usrconfig_t *config;
	lb_usrdata *usrdata;
	FILE *file;
	size_t actual;
	
	DEBUG(DEBUG_NAME " lb_server_put_progressed. buflen: %d\n", buflen);
	
	config = btobexserver_getusrconfig(handle);
	usrdata = config->usrdata;
	
	file = usrdata->putfile;
	
	if (file != NULL && buf != NULL && buflen > 0) {	
		actual = fwrite(buf, 1, buflen, file);
		if (actual != buflen) {
			DEBUG(DEBUG_NAME " Error writing to file, denying further PUT packets\n");
			return FALSE;
		}
	}
	
	return TRUE;
}

/**
 * Called when a PUT is completed on the server.
 */
/*void lb_server_put_complete(obex_t *handle, char *filename)*/
void lb_server_put_complete(obex_t *handle)
{
	btobexserver_usrconfig_t *config;
	lb_usrdata *usrdata;
	
	config = btobexserver_getusrconfig(handle);
	usrdata = config->usrdata;
	
	usrdata->got_file = TRUE;
}

void lb_server_error(obex_t *handle, int error, int last_evt, int last_cmd)
{
	if (error < 0) {
		DEBUG(DEBUG_NAME " Server error (%d). (Last event = %d, last command = 0x%02x)\n", 
		       error, last_evt, last_cmd);
		/* stop the server now */
		btobexserver_stop(handle);
	} else if (error == 0) {
		btobexserver_usrconfig_t *config;
		lb_usrdata *usrdata;
	
		config = btobexserver_getusrconfig(handle);
		usrdata = config->usrdata;
		
		if (usrdata->got_file) {
			/* server timed out, but we've already received a file
			and that's all we wanted, so stop the server now 
			(i.e. maybe client sent a file but hasn't disconnected)*/
			DEBUG(DEBUG_NAME " Timeout after received file OK, stopping server.\n");
			btobexserver_stop(handle);
		}
	}
}

void lb_server_transport_broken(obex_t *handle)
{
	DEBUG(DEBUG_NAME "Transport broken, stopping server.\n");
	btobexserver_stop(handle);
}


/**
 * (method for Python interface)
 * Receives a file from a remote device. i.e. starts an OBEX server and 
 * stops the server immediately after it receives a file.
 *
 * Note the arguments are (fileno, filename-or-object) not like the python 
 * lightblue.obex.recvfile method that takes (socket-object, filename)
 * 
 * Arguments:
 * - fileno: the file descriptor for a server socket on which to run the
 *   OBEX server. The socket must be already bound and listening.
 * - dest: the file path or file object to use to store the received file.
 */
static PyObject* lb_recv_file(PyObject *self, PyObject *args)
{
	int err = 0;
	
	int serverfd = -1;
	PyObject *fileobj = NULL;
	
	obex_t *handle;
	btobexserver_usrconfig_t serverconfig;
	lb_usrdata usrdata;	
	
	
	if (!PyArg_ParseTuple(args, "iO", &serverfd, &fileobj)) {
		return NULL;
		PyErr_Clear();
		if (!PyFile_Check(fileobj))
			return NULL;
	}
	
	
	/* check given fd */
	if (serverfd < 0) {
		PyErr_SetString(PyExc_ValueError, "Given fileno is negative");
		return NULL;
	}
	 
	    	
	/* setup user data - set to use file object */
	usrdata.putfile = PyFile_AsFile(fileobj);
	if ( !(usrdata.putfile) ) {
		PyErr_SetString(PyExc_IOError, "Error getting file object as C FILE*");
		return NULL;
	}
	
	/* setup user config data, including callback functions */
	serverconfig.timeout = 0;	/* wait forever for a client connection */
	usrdata.got_file = FALSE;
	
	serverconfig.connect_complete = lb_server_connect_complete;	
	serverconfig.disconnect_complete = lb_server_disconnect_complete;	
	serverconfig.put_requested = lb_server_put_requested;	
	serverconfig.put_progressed = lb_server_put_progressed;
	serverconfig.put_complete = lb_server_put_complete;
	serverconfig.error_handling_input = lb_server_error;
	serverconfig.transport_broken = lb_server_transport_broken;
	
	/* set the user data in the server config */
	serverconfig.usrdata = &usrdata;
	
	/* create obex server */
	if ( !(handle = btobexserver_init(serverfd, 0)) ) {
		PyErr_SetString(PyExc_IOError, "Error creating OBEX server handle");
		return NULL;
	}
	OBEX_SetTransportMTU(handle, 256, 256);
		
	/* set server user config data */
	btobexserver_setusrconfig(handle, &serverconfig);
	
	/* run OBEX server */
	err = btobexserver_run(handle);
	if (err <= 0) {
		/* if server has already successfully received a file, 
		just print debug message, otherwise set an error */
		char *msg = (err == 0)? "Operation timed out" : 
				"Error while running OBEX server";
		if (usrdata.got_file) {
			DEBUG(DEBUG_NAME "%s (after file received OK)\n", msg);
		} else {
			lb_server_cleanup(handle);
			PyErr_SetString(PyExc_IOError, msg);
			return NULL;			
		}
	}
    
    DEBUG(DEBUG_NAME "Finished running server.\n");
	lb_server_cleanup(handle);
	return Py_BuildValue("");
}



/**************************************************/


/* list of all functions in this module */
static PyMethodDef obexmethods[] = {
    {"send_file", lb_send_file, METH_VARARGS},
    {"recv_file", lb_recv_file, METH_VARARGS},
    { NULL, NULL }  /* sentinel */
};

/* module initialization functions */
void init_obexutil(void) {
    Py_InitModule("_obexutil", obexmethods);
}
