# $Id: menu.py,v 1.14 2007/07/14 07:38:13 mwm Exp $
#
# menu.py -- Screen mixin to provide menus.
#
#    Copyright (C) 2001  Mike Meyer <mwm@mired.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"menu - a mixin to provide menus for plwm screens."

from Xlib import X
from keys import KeyGrabKeyboard, KeyHandler, allmap
from message import Message

class MenuKeyHandler(KeyGrabKeyboard):
    """Template for handling a menu.

    MenuKeyHandler defines the following event handler methods:

    _up, _down - move the cursor up or down the menu.

    _do - perform the currently selected menu action.
    _abort - exit the menu with no actions."""

    timeout = None

    def __init__(my, menu):
        KeyGrabKeyboard.__init__(my, menu.window, X.CurrentTime)
        my.menu = menu

    def _up(my, event): my.menu.up()
    def _down(my, event): my.menu.down()

    def _do(my, event):
        my._cleanup()
        my.menu.do()

    def _abort(my, event):
        my._cleanup()
        my.menu.close()


class MenuCharHandler(MenuKeyHandler):
    """MenuCharHandler allows "one-key access" to menu entries.

    This adds the _goto method, which goes to the first menu label
    that starts with a character >= the event keycode, then the latin1
    symbols to that method."""

    def _goto(my, event):
        char = chr(my.wm.display.keycode_to_keysym(event.detail, 0))
        if event.state:
            char = char.upper()
        return my.menu.goto(char)
allmap(MenuCharHandler, MenuCharHandler._goto)


class MenuCharSelecter(MenuCharHandler):
    """A MenuCharHandler with intanst selection.

    Just like MenuCharHandler, except when you hit the character,
    it not only takes you there, it issues the command if the character
    is in the menu. Otherwise, it acts like MenuCharHandler."""

    def _goto(my, event):
        if MenuCharHandler._goto(my, event):
            my._do(event)
allmap(MenuCharSelecter, MenuCharSelecter._goto)

class Menu(Message):
    "Holds and manipulates the menu."

    def setup(my, labels, align = 'center'):
        "Initialize the menu window, gc and font"

        width, height = Message.setup(my, labels, align)
        my.high = height / len(my.lines)
        my.current = 0
        return width, height

    def start(my, x, y, action, handler, timeout = 0):
        """Start it up...

           Passing x,y = -1,-1 will cause the menu to be centred on the
           screen.
        """

        if x==-1 and y==-1:
            x = my.wm.current_screen.root_x + \
                my.wm.current_screen.root_width/2-my.width/2
            y = my.wm.current_screen.root_x + \
                my.wm.current_screen.root_height/2-my.height/2

        Message.display(my, x, y, timeout)
        my.window.get_focus(X.CurrentTime)
        my.action = action
        handler(my)

    def up(my):
        "Move the menu selection up."

        my.current = my.current - 1
        if my.current < 0: my.current = len(my.lines) - 1
        my.redraw()

    def down(my):
        "Move the menu selection down"

        my.current = my.current + 1
        if my.current >= len(my.lines): my.current = 0
        my.redraw()
        
    def goto(my, char):
        """Goto the first entry with a label that starts >= char

        returns true if entry actually starts with char."""

        length = len(my.lines)
        lc = char.lower()
        for i in range(length):
            if (lc, char) <= (my.lines[i].name[0].lower(), my.lines[i].name[0]):
                break
        if i < length: my.current = i
        else: my.current = length - 1
        my.redraw()
        return char == my.lines[i].name[0]

    def do(my):
        "Run it!"

        my.close()
        my.wm.display.sync()
        my.action(my.lines[my.current].name)

    def redraw(my, event = None):
        "Redraw the window, with highlights"

        Message.redraw(my)
        my.window.fill_rectangle(my.gc, 0, my.current * my.high,
                                        my.width, my.high)


class screenMenu:
    """PLWM Screen mixin to provide a per-screen menu.

    This mixin requires the color and font mixins be in the screen class."""

    menu_font = "9x15Bold"
    menu_foreground = "black"
    menu_background = "white"
    menu_bordercolor = "black"
    menu_borderwidth = 3
    menu_seconds = 0
    menu_draw = X.GXinvert

    def menu_make(my, labels, align = 'center'):
        """Create a menu of labels.

        Returns the width and height of the resulting menu."""

        my.menu = Menu(my, my.menu_font, my.menu_draw,
                             my.menu_foreground, my.menu_background,
                             my.menu_bordercolor, my.menu_borderwidth,
                             my.menu_seconds)
        return my.menu.setup(labels, align)

    def menu_run(my, x, y, action):
        "Instantiate the menu, and return the label or None."

        my.menu.start(x, y, action, my.menu_handler)
