"""
see also examples/oofun/*.py files for other oofun benefits

Let
F(y) = y^2 - 0.05
G(y) = y^4 + y^2 - 0.01
H(y,z) = y + 2*z - 0.15

Consider the problem
(x0-0)^2 + (x1-1)^2 + (x2-2)^2 +... +(x[n-1]-(n-1))^2 -> min
n=15

subjected to some constraints (probably with some nested code):

c0(x) = 1e-2*x[11]**2 + 4e-1*x[12]**2 + 0.5*x[13] ** 2 + x[14] ** 2 - 1 <= 0 (Python indexing starts from zero, hence x[14] is last coord)

F(c0(x)) <= 0
G(c0(x)) <=0
H(c0(x), F(c0(x))) <= 0
# and probably some other, see below.

Providing user-supplied analitical derivative for each constraint and/or objective function
sometimes is not an ordinary task (if possible at all).
Still, for some parts of code it can be done easily.
For example, for nested code F(G(x)) one can provide derivatives for F, and openopt
will search for dG/dx via finite-differences approximation and then substitute it for d(F(G(x)))/dx

It's very convenient for a number of deeply merged recursive funcs: F(G(H(...(Z(x)))))

The situation is rather common for engineering problems: for example,
Z(x) is mass of spoke,
H(Z, other data) is mass of wheel,
G(H, other data) is mass of bicycle
"""

from numpy import arange, ones, asarray, zeros, array, hstack
from scikits.openopt import NLP, oofun

n= 15

x0 = ones(n)
f = lambda x: ((x - arange(n)) ** 2).sum()
df = lambda x: 2 * (x-arange(n))

c0 = oofun(lambda x: 1e-2*x[-4]**2 + 4e-1*x[-3]**2 + 0.5*x[-2] ** 2 + x[-1] ** 2 - 1, name = 'constraint 0') # parameter "name" is optional
c0.dep = [11, 12, 13, 14] # optional, will not be required when oovar will be implemented and used
# all user-provided derivatives *.d are optional
c0.d = lambda x: array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2e-2*x[-4], 8e-1*x[-3], x[-2], 2*x[-1]])

c1 = oofun(lambda c: c**2 - 0.05, input = c0, name = 'constraint 1')
c1.d = lambda c: 2*c

c2 = oofun(lambda c: c**4 + c**2 - 0.01, input = c1, name='constraint 2')
c2.d = lambda c: 2*c + 4*c**3

# 2 inputs, 1 output
c21 = oofun(lambda y, z: y + 2*z - 0.15, input = [c0, c2], name = 'constraint 21')
c21.d = lambda y, z: array([1, 2])

# also, several inputs <-> several inputs oofuncs can be used as well:

# 1 input, 3 outputs
c13 = oofun(lambda y: [y - 0.15, 4*(y-0.14), 5*(y+0.18)], input = c0, name = 'constraint 13')
c13.d = lambda y: array((1, 4, 5))

# 2 inputs, 3 outputs
c23 = oofun(lambda y, z: [y + 2*z - 0.15,  10*y + 21*z - 1.5, 100*y + 201*z - 15], input = [c0, c2], name = 'constraint 23')
c23.d = lambda y, z: array([[1, 2], [10, 21], [100, 201]])

# gather all non-linear inequality constraints:
c = [c2, c0, c1, c23, c21, c13] # ORDER: any

p = NLP(f,  x0,  df=df, c=c)
r = p.solve('ralg')
#Solver:   Time Elapsed = 0.82 	CPU Time Elapsed = 0.8
#objFunValue: 497.79286 (feasible, max constraint =  8e-07)



