# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
Python X2go is a python module that implements X2go client support for 
the free X2go server project (U{http://wiki.x2go.org}) in Python.

Introduction
============
    With Python X2go you can write your own X2go clients or embed X2go client
    features into already existing application environments.

API Concept
===========

    Python X2go consists of quite a few classes. Furthermore,
    Python X2go is quite heavily taking advantage of Python\'s
    threading features. When providing a library like Python
    X2go, it is always quite a task to keep the library code
    compatible with former versions of the same library. This is
    intended for Python X2go, but with some restraints.

    Python X2go only offers 5 public API classes. With the release of
    version 0.1.0.0, we will try to keep these 5 public API classes
    of future releases as compatible as possible with versions of Python X2go
    greater/equal than v0.1.0.0.

    The 4 public API classes are:

        - L{X2goClient} --- a whole X2go client API
        - L{X2goSession} --- management of an individual X2go 
        session started from an L{X2goClient} instance
        - L{X2goClientSettings} --- provide access to X2go (global and 
        user) configuration node »settings«
        - L{X2goClientPrinting} --- provide access to X2go (global and 
        user) configuration node »printing«
        - L{X2goSessionProfiles} --- provide access to X2go (global and 
        user) configuration node »sessions«

    Any other of the Python X2go classes may be subject to internal changes
    and the way of addressing these classes in code may vary between different
    versions of Python X2go. If you directly use other than the 5 public API 
    classes in your own applications, so please be warned.


API Structure
=============

    When using Python X2go in your applications, the basic idea is that you
    create your own class and inherit the X2goClient class in that::

        import x2go
        class MyX2goClient(x2go.X2goClient):

            ...

    The L{X2goClient} class flattens the complex structure of Python X2go into
    many L{X2goClient} methods that you can use in your own C{MyX2goClient} instances.

    However, it might be handy to retrieve a whole X2go session instance 
    from the L{X2goClient} instance. This can be achieved by the 
    L{X2goClient.register_session()} method::

        import x2go
        my_x2gocli = MyX2goClient()
        reg_session_instance = my_x2gocli.register_session(<options>, return_object=True)

    Whereas <options> can be as simple as::

         »profile_name=<PROFILE_NAME_IN_SESSIONS_FILE>«

    or contain a whole set of L{X2goSession} parameters that can be used to start a 
    session manually (i.e. a session that is based on a pre-configured session profile 
    in either of the »sessions« config files).

    The L{X2goClient.register_session()} method---in object-retrieval-mode---returns
    an L{X2goSession} instance. With this instance you can then manage
    your X2go session::

        import gevent, getpass
        pw=getpass.getpass()
        # authenticate
        reg_session_instance.connect(password=pw, <further_options>)
        # launch the session window
        reg_session_instance.start()
        # or resume the youngest session for 
        gevent.sleep(60)
        reg_session_instance.suspend()
        # or alternatively:
        reg_session_instance.terminate()

    How to access---especially how to modify---the X2go client configuration
    files »settings«, »printing«, »sessions« and »xconfig« (Windows only)
    is explained in detail with each class declaration in this API documentation. 
    Please refer to the class docs of L{X2goClientSettings}, L{X2goClientPrinting},
    L{X2goSessionProfiles} and L{X2goXServer}.

Configuration and Session Management
====================================

    Python X2go strictly separates configuration management from
    session management. The classes needed for session management
    / administration are supposed to only gain »read access« to the 
    classes handling the X2go client configuration nodes.

    A configuration node in Python X2go can be a file, a gconf database
    section, a section in the Windows registry, etc.

    NOTE: Each configuration node will be re-read whenever it is needed 
    by an X2go sesion or the X2goClient class itself.

    Conclusively, any change to either of the configuration nodes
    will be reflected as a change in your X2go client behaviour:

      - L{X2goSessionProfiles}: changes to a session profile in
      the »sessions« node will be available for the next registered
      L{X2goSession} instance
      - L{X2goClientPrinting}: on each incoming X2go print job the
      »printing« configuration node will be re-read, thus you can 
      change your X2go client's print setup during a running session
      - L{X2goClientSettings}: also the configuration node »settings« 
      is re-read whenever needed in the course of X2go session management
      - L{X2goXServer} (Windows only): this class will only be initialized 
      once (starting the XServer on Windows platforms) on construction
      of an L{X2goClient} instance

Dependencies
============
    Python X2go takes advantage of the libevent/greenlet implementation 
    gevent (http://www.gevent.org). The least needed version of Python gevent
    is 0.13.0 (or above).

    Python X2go (because of gevent) requires at least Python 2.6. Further recent
    information on Python X2go is available at: 
    U{http://wiki.x2go.org/python-x2go}

Contact
=======
    If you have any questions concerning Python X2go, please sign up for the
    x2go-dev list (https://lists.berlios.de/mailman/listinfo/x2go-dev) and post
    your questions, requests and feedbacks there.
"""

__NAME__    = 'python-x2go'
__VERSION__ = '0.1.1.3'

from gevent import monkey
monkey.patch_all()

import utils

import guardian
import signal as _signal
_signal.signal (_signal.SIGTERM, guardian._sigterm_handle )
_signal.signal (_signal.SIGINT, guardian._sigterm_handle )

from client import X2goClient
from backends.profiles import X2goSessionProfiles
from backends.printing import X2goClientPrinting
from backends.settings import X2goClientSettings
from session import X2goSession
from sshproxy import X2goSSHProxy
from x2go_exceptions import *
from log import *

from cleanup import x2go_cleanup

from defaults import X2GOCLIENT_OS
from defaults import CURRENT_LOCAL_USER
from defaults import LOCAL_HOME
from defaults import X2GO_CLIENT_ROOTDIR
from defaults import X2GO_SESSIONS_ROOTDIR
from defaults import X2GO_SSH_ROOTDIR

if X2GOCLIENT_OS == 'Windows':
    from xserver import X2goClientXConfig, X2goXServer

