# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goControlSessionSTDOUT class - core functions for handling your individual X2go sessions.

This backend handles X2go server implementations that respond via server-side STDOUT.

"""
__NAME__ = 'x2gocontrolsession-pylib'

# modules
import os
import types
import paramiko
import gevent

import copy
import binascii

import string
import random

# Python X2go modules
import x2go.sshproxy as sshproxy
import x2go.log as log
import x2go.utils as utils
import x2go.x2go_exceptions as x2go_exceptions
import x2go.defaults as defaults
import x2go.checkhosts as checkhosts

from x2go.backends.terminal import X2goTerminalSession as _X2goTerminalSession
from x2go.backends.info import X2goServerSessionInfo as _X2goServerSessionInfo
from x2go.backends.info import X2goServerSessionList as _X2goServerSessionList
from x2go.backends.proxy import X2goProxy as _X2goProxy

def _rerewrite_blanks(cmd):
    # X2go run command replace X2GO_SPACE_CHAR string with blanks
    if cmd:
        cmd = cmd.replace("X2GO_SPACE_CHAR", " ")
    return cmd

def _rewrite_password(cmd, user=None, password=None):

    # if there is a ,,-u X2GO_USER'' parameter in RDP options then we will replace 
    # it by our X2go session password
    if cmd and user:
        cmd = cmd.replace('X2GO_USER', user)
    # if there is a ,,-p X2GO_PASSWORD'' parameter in RDP options then we will replace 
    # it by our X2go session password
    if cmd and password:
        cmd = cmd.replace('X2GO_PASSWORD', password)
    return cmd


class X2goControlSessionSTDOUT(paramiko.SSHClient):
    """\
    STILL UNDOCUMENTED

    @param logger: you can pass an L{X2goLogger} object to the
        L{X2goControlSessionSTDOUT} constructor
    @type logger: L{X2goLogger} instance
    @param loglevel: if no L{X2goLogger} object has been supplied a new one will be
        constructed with the given loglevel
    @type loglevel: int

    """
    associated_terminals = None

    def __init__(self,
                 profile_name='UNKNOWN',
                 add_to_known_hosts=False,
                 known_hosts=None,
                 terminal_backend=_X2goTerminalSession,
                 info_backend=_X2goServerSessionInfo,
                 list_backend=_X2goServerSessionList,
                 proxy_backend=_X2goProxy,
                 client_rootdir=os.path.join(defaults.LOCAL_HOME, defaults.X2GO_CLIENT_ROOTDIR),
                 sessions_rootdir=os.path.join(defaults.LOCAL_HOME, defaults.X2GO_SESSIONS_ROOTDIR),
                 ssh_rootdir=os.path.join(defaults.LOCAL_HOME, defaults.X2GO_SSH_ROOTDIR),
                 logger=None, loglevel=log.loglevel_DEFAULT,
                 *args, **kwargs):
        """\
        Initialize an X2go session. With the L{X2goControlSessionSTDOUT} class you can start
        new X2go sessions, resume suspended sessions or suspend resp. terminate
        currently running sessions on a connected X2go server.

        """
        self.associated_terminals = {}
        self.terminated_terminals = []

        self.profile_name = profile_name
        self.add_to_known_hosts = add_to_known_hosts
        self.known_hosts = known_hosts

        self.hostname = None
        self.port = None

        self.sshproxy_session = None

        self._session_auth_rsakey = None
        self._remote_home = None
        self._remote_group = {}

        self.locked = False

        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        self._terminal_backend = terminal_backend
        self._info_backend = info_backend
        self._list_backend = list_backend
        self._proxy_backend = proxy_backend

        self.client_rootdir = client_rootdir
        self.sessions_rootdir = sessions_rootdir
        self.ssh_rootdir = ssh_rootdir

        paramiko.SSHClient.__init__(self, *args, **kwargs)
        if self.add_to_known_hosts:
            self.set_missing_host_key_policy(paramiko.AutoAddPolicy())

        self.session_died = False

    def load_session_host_keys(self):
        if self.known_hosts is not None:
            utils.touch_file(self.known_hosts)
            self.load_host_keys(self.known_hosts)

    def __del__(self):

        self.disconnect()

    def _x2go_sftp_put(self, local_path, remote_path):

        self.logger('sFTP-put: %s -> %s:%s' % (os.path.normpath(local_path), self.get_transport().getpeername(), remote_path), loglevel=log.loglevel_DEBUG)
        self.sftp_client.put(os.path.normpath(local_path), remote_path)

    def _x2go_sftp_write(self, remote_path, content):

        self.logger('sFTP-write: opening remote file %s on host %s for writing' % (remote_path, self.get_transport().getpeername()), loglevel=log.loglevel_DEBUG)
        remote_fileobj = self.sftp_client.open(remote_path, 'w')
        self.logger('sFTP-write: writing content: %s' % content, loglevel=log.loglevel_DEBUG_SFTPXFER)
        remote_fileobj.write(content)
        remote_fileobj.close()

    def _x2go_sftp_remove(self, remote_path):

        self.logger('sFTP-write: removing remote file %s on host %s' % (remote_path, self.get_transport().getpeername()), loglevel=log.loglevel_DEBUG)
        self.sftp_client.remove(remote_path)

    def _x2go_exec_command(self, cmd_line, loglevel=log.loglevel_INFO, **kwargs):

        while self.locked:
            gevent.sleep(.1)

        self.locked = True
        _retval = None

        if type(cmd_line) == types.ListType:
            cmd = " ".join(cmd_line)
        else:
            cmd = cmd_line
        if self.get_transport() is not None:

            timeout = gevent.Timeout(20)
            timeout.start()
            try:
                self.logger("executing command on X2go server ,,%s'': %s" % (self.profile_name, _rerewrite_blanks(cmd)), loglevel)
                _retval = self.exec_command(_rewrite_password(cmd, user=self.get_transport().get_username(), password=self._session_password), **kwargs)
            except AttributeError:
                self.session_died = True
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                self.locked = False
                raise x2go_exceptions.X2goControlSessionException('the X2go control session has died unexpectedly')
            except EOFError:
                self.session_died = True
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                self.locked = False
                raise x2go_exceptions.X2goControlSessionException('the X2go control session has died unexpectedly')
            except x2go_exceptions.SSHException:
                self.session_died = True
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                self.locked = False
                raise x2go_exceptions.X2goControlSessionException('the X2go control session has died unexpectedly')
            except gevent.timeout.Timeout:
                self.session_died = True
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                self.locked = False
                raise x2go_exceptions.X2goControlSessionException('the X2go control session command timed out')
            finally:
                self.locked = False
                timeout.cancel()

        else:
            self.locked = False
            raise x2go_exceptions.X2goControlSessionException('the X2go control session is not connected')
        return _retval

    @property
    def _x2go_remote_home(self):

        if self._remote_home is None:
            (stdin, stdout, stderr) = self._x2go_exec_command('echo $HOME')
            self._remote_home = stdout.read().split()[0]
            self.logger('remote user\' home directory: %s' % self._remote_home, loglevel=log.loglevel_DEBUG)
            return self._remote_home
        else:
            return self._remote_home

    def _x2go_remote_group(self, group):

        if not self._remote_group.has_key(group):
            (stdin, stdout, stderr) = self._x2go_exec_command('getent group %s | cut -d":" -f4' % group)
            self._remote_group[group] = stdout.read().split('\n')[0].split(',')
            self.logger('remote %s group: %s' % (group, self._remote_group[group]), loglevel=log.loglevel_DEBUG)
            return self._remote_group[group]
        else:
            return self._remote_group[group]

    def is_x2gouser(self, username):
        ###
        ### FIXME:
        ###
        # discussion about server-side access restriction based on posix group membership or similar currently 
        # in process (as of 20110517, mg)
        #return username in self._x2go_remote_group('x2gousers')
        return True

    def remote_username(self):
        """\
        Returns the control session's remote username.

        """
        if self.get_transport() is not None:
            return self.get_transport().get_username()
        else:
            return None

    @property
    def _x2go_session_auth_rsakey(self):
        if self._session_auth_rsakey is None:
            self._session_auth_rsakey = paramiko.RSAKey.generate(defaults.RSAKEY_STRENGTH)
        return self._session_auth_rsakey

    def set_profile_name(self, profile_name):
        self.profile_name = profile_name

    def check_host(self, hostname, port=22):
        """\
        Wraps around a Paramiko/SSH host key check.

        """
        return checkhosts.check_ssh_host_key(self, hostname, port=port)

    def connect(self, hostname, port=22, username='', password='', pkey=None,
                use_sshproxy=False, sshproxy_host='', sshproxy_user='', sshproxy_password='',
                sshproxy_key_filename='', sshproxy_tunnel='',
                key_filename=None, timeout=None, allow_agent=False, look_for_keys=False,
                session_instance=None,
                add_to_known_hosts=False, force_password_auth=False):
        """\
        Connect to an X2go server and authenticate to it. This method is directly
        inherited from the paramiko.SSHClient module. The features of the Paramiko 
        SSH client connect method are recited here. The parameters C{add_to_known_hosts}
        and C{force_password_auth} have been added as a parameter for X2go.

        The server's host key
        is checked against the system host keys (see C{load_system_host_keys})
        and any local host keys (C{load_host_keys}).  If the server's hostname
        is not found in either set of host keys, the missing host key policy
        is used (see C{set_missing_host_key_policy}).  The default policy is
        to reject the key and raise an C{SSHException}.

        Authentication is attempted in the following order of priority:

            - The C{pkey} or C{key_filename} passed in (if any)
            - Any key we can find through an SSH agent
            - Any "id_rsa" or "id_dsa" key discoverable in C{~/.ssh/}
            - Plain username/password auth, if a password was given

        If a private key requires a password to unlock it, and a password is
        passed in, that password will be used to attempt to unlock the key.

        @param hostname: the server to connect to
        @type hostname: str
        @param port: the server port to connect to
        @type port: int
        @param username: the username to authenticate as (defaults to the
            current local username)
        @type username: str
        @param password: a password to use for authentication or for unlocking
            a private key
        @type password: str
        @param pkey: an optional private key to use for authentication
        @type pkey: C{PKey}
        @param key_filename: the filename, or list of filenames, of optional
            private key(s) to try for authentication
        @type key_filename: str or list(str)
        @param timeout: an optional timeout (in seconds) for the TCP connect
        @type timeout: float
        @param allow_agent: set to False to disable connecting to the SSH agent
        @type allow_agent: C{bool}
        @param look_for_keys: set to False to disable searching for discoverable
            private key files in C{~/.ssh/}
        @type look_for_keys: C{bool}
        @param add_to_known_hosts: non-paramiko option, if C{True} paramiko.AutoAddPolicy() 
            is used as missing-host-key-policy. If set to C{False} paramiko.RejectPolicy() 
            is used
        @type add_to_known_hosts: C{bool}
        @param force_password_auth: non-paramiko option, disable pub/priv key authentication 
            completely, even if the C{pkey} or the C{key_filename} parameter is given
        @type force_password_auth: C{bool}
        @param session_instance: an instance L{X2goSession} using this L{X2goControlSessionSTDOUT}
            instance.
        @type session_instance: C{instance}

        @raise BadHostKeyException: if the server's host key could not be
            verified
        @raise AuthenticationException: if authentication failed
        @raise SSHException: if there was any other error connecting or
            establishing an SSH session
        @raise socket.error: if a socket error occurred while connecting

        """
        if use_sshproxy and sshproxy_host and sshproxy_user:
            try:
                self.sshproxy_session = sshproxy.X2goSSHProxy(known_hosts=self.known_hosts,
                                                              sshproxy_host=sshproxy_host,
                                                              sshproxy_user=sshproxy_user,
                                                              sshproxy_password=sshproxy_password,
                                                              sshproxy_key_filename=sshproxy_key_filename,
                                                              sshproxy_tunnel=sshproxy_tunnel,
                                                              session_instance=session_instance,
                                                              logger=self.logger,
                                                             )

            except:
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                self.sshproxy_session = None
                raise

            if self.sshproxy_session is not None:
                self.sshproxy_session.start()

                # divert port to sshproxy_session's local forwarding port (it might have changed due to 
                # SSH connection errors
                gevent.sleep(.1)
                port = self.sshproxy_session.get_local_proxy_port()

        if not add_to_known_hosts and session_instance:
            self.set_missing_host_key_policy(checkhosts.X2goInteractiveAddPolicy(caller=self, session_instance=session_instance))

        if add_to_known_hosts:
            self.set_missing_host_key_policy(paramiko.AutoAddPolicy())

        # disable pub/priv key authentication if forced
        if force_password_auth:
            key_filename = None
            pkey = None

        self.logger('connecting to [%s]:%s' % (hostname, port), loglevel=log.loglevel_NOTICE)

        self.load_session_host_keys()

        _hostname = hostname
        # enforce IPv4 for localhost address
        if _hostname == 'localhost':
            _hostname = '127.0.0.1'

        if (key_filename and os.path.exists(os.path.normpath(key_filename))) or pkey:
            try:
                self.logger('trying SSH pub/priv key authentication with server', loglevel=log.loglevel_DEBUG)
                paramiko.SSHClient.connect(self, _hostname, port=port, username=username, pkey=pkey,
                                           key_filename=key_filename, timeout=timeout, allow_agent=allow_agent, 
                                           look_for_keys=look_for_keys)

            except paramiko.AuthenticationException, e:
                self.close()
                if password:
                    self.logger('next auth mechanism we\'ll try is keyboard-interactive authentication', loglevel=log.loglevel_DEBUG)
                    try:
                        paramiko.SSHClient.connect(self, _hostname, port=port, username=username, password=password,
                                                   timeout=timeout, allow_agent=allow_agent, 
                                                   look_for_keys=look_for_keys)
                    except paramiko.AuthenticationException, e:
                        self.close()
                        if self.sshproxy_session:
                            self.sshproxy_session.stop_thread()
                        raise e
                    except:
                        self.close()
                        if self.sshproxy_session:
                            self.sshproxy_session.stop_thread()
                        raise
                else:
                    self.close()
                    if self.sshproxy_session:
                        self.sshproxy_session.stop_thread()
                    raise(e)

            except:
                self.close()
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                raise

        # if there is not private key, we will use the given password, if any
        else:
            # create a random password if password is empty to trigger host key validity check
            if not password:
                password = "".join([random.choice(string.letters+string.digits) for x in range(1, 20)])
            self.logger('performing SSH keyboard-interactive authentication with server', loglevel=log.loglevel_DEBUG)
            try:
                paramiko.SSHClient.connect(self, _hostname, port=port, username=username, password=password,
                                           timeout=timeout, allow_agent=allow_agent, look_for_keys=look_for_keys)
            except paramiko.AuthenticationException, e:
                self.close()
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                raise e
            except:
                self.close()
                if self.sshproxy_session:
                    self.sshproxy_session.stop_thread()
                raise

        self.set_missing_host_key_policy(paramiko.RejectPolicy())

        self.hostname = hostname
        self.port = port

	# if we succeed, we immediately grab us an sFTP client session
        try:
            self.sftp_client = self.open_sftp()
        except:
            raise x2go_exceptions.X2goControlSessionException('could not invoke server-side SFTP subsystem')

        # preparing reverse tunnels
        ssh_transport = self.get_transport()
        ssh_transport.reverse_tunnels = {}

        # mark Paramiko/SSH transport as X2goControlSession
        ssh_transport._x2go_session_marker = True
        self._session_password = password

        if self.get_transport():
            self.session_died = False
        return (self.get_transport() is not None)

    def dissociate(self, terminal_session):
        """\
        STILL UNDOCUMENTED

        """
        for t_name in self.associated_terminals.keys():
            if self.associated_terminals[t_name] == terminal_session:
                del self.associated_terminals[t_name]
                if self.terminated_terminals.has_key(t_name):
                    del self.terminated_terminals[t_name]

    def disconnect(self):
        """\
        STILL UNDOCUMENTED

        """
        if self.associated_terminals:
            t_names = self.associated_terminals.keys()
            for  t_obj in self.associated_terminals.values():
                try:
                    if not self.session_died:
                        t_obj.suspend()
                except x2go_exceptions.X2goTerminalSessionException:
                    pass
                except x2go_exceptions.X2goControlSessionException:
                    pass
                t_obj.__del__()
            for t_name in t_names:
                try:
                    del self.associated_terminals[t_name]
                except KeyError:
                    pass

        self._remote_home = None
        self._remote_group = {}

        self._session_auth_rsakey = None

        try:
            if self.get_transport() is not None:
                still_active = self.get_transport().is_active()
                self.close()
                if self.sshproxy_session is not None:
                    self.sshproxy_session.stop_thread()
                return still_active
            return False
        except AttributeError:
            # if the Paramiko _transport object has not yet been initialized, ignore it
            # but state that this method call did not close the SSH client, but was already closed
            return False


    def is_alive(self):
        if self._x2go_exec_command('echo', loglevel=log.loglevel_DEBUG):
            return True
        return False

    def start(self, **kwargs):
        """\
        Start a new X2go session. 

        The L{X2goControlSessionSTDOUT.start()} method accepts any parameter
        that can be passed to any of the C{X2goTerminalSession} backend class
        constructors.

        """
        return self.resume(**kwargs)

    def resume(self, session_name=None, session_instance=None, **kwargs):
        """\
        Resume a running/suspended X2go session. 

        The L{X2goControlSessionSTDOUT.resume()} method accepts any parameter
        that can be passed to any of the C{X2goTerminalSession} backend class constructors.

        @return: True if the session could be successfully resumed
        @rtype: C{bool}

        """
        if not self.is_x2gouser(self.get_transport().get_username()):
            raise x2go_exceptions.X2goUserException('remote user %s is not allowed to run X2go commands' % self.get_transport().get_username())

        if session_name is not None:
            session_info = self.list_sessions()[session_name]
        else:
            session_info = None

        _terminal = self._terminal_backend(self,
                                           profile_name=self.profile_name,
                                           session_info=session_info,
                                           info_backend=self._info_backend,
                                           list_backend=self._list_backend,
                                           proxy_backend=self._proxy_backend,
                                           client_rootdir=self.client_rootdir,
                                           session_instance=session_instance,
                                           sessions_rootdir=self.sessions_rootdir,
                                           **kwargs)

        _success = False
        if session_name is not None:
            try:
                _success = _terminal.resume()
            except x2go_exceptions.X2goFwTunnelException:
                pass

        else:
            try:
                _success = _terminal.start()
            except x2go_exceptions.X2goFwTunnelException:
                pass

        if _success:
            while not _terminal.ok():
                gevent.sleep(.2)

            if _terminal.ok():
                self.associated_terminals[_terminal.get_session_name()] = _terminal
                self.get_transport().reverse_tunnels[_terminal.get_session_name()] = {
                    'sshfs': (0, None),
                    'snd': (0, None),
                }

                return _terminal or None

        return None

    def share_desktop(self, desktop=None, user=None, display=None, share_mode=0, **kwargs):
        """\
        Share another already running desktop session. Desktop sharing can be run
        in two different modes: view-only and full-access mode.

        @param desktop: desktop ID of a sharable desktop in format <user>@<display>
        @type desktop: C{str}
        @param user: user name and display number can be given separately, here give the
            name of the user who wants to share a session with you.
        @type user: C{str}
        @param display: user name and display number can be given separately, here give the
            number of the display that a user allows you to be shared with.
        @type display: C{str}
        @param share_mode: desktop sharing mode, 0 is VIEW-ONLY, 1 is FULL-ACCESS.
        @type share_mode: C{int}

        @return: True if the session could be successfully shared.
        @rtype: C{bool}

        """
        if desktop:
            user = desktop.split('@')[0]
            display = desktop.split('@')[1]
        if not (user and display):
            raise x2go_exceptions.X2goDesktopSharingException('Need user name and display number of sharable desktop.')

        cmd = '%sXSHAD%sXSHAD%s' % (share_mode, user, display)

        kwargs['cmd'] = cmd
        kwargs['session_type'] = 'shared'

        return self.start(**kwargs)

    def list_desktops(self, raw=False, maxwait=20):
        """\
        List all desktop-like sessions of current user (or of users that have 
        granted desktop sharing) on the connected server.

        @param raw: if C{True}, the raw output of the server-side X2go command 
            C{x2godesktopsharing} is returned.
        @type raw: C{bool}

        @return: a list of X2go desktops available for sharing
        @rtype: C{list}

        """
        if raw:
            (stdin, stdout, stderr) = self._x2go_exec_command("export HOSTNAME && x2golistdesktops")
            return stdout.read(), stderr.read()

        else:

            # this _success loop will catch errors in case the x2golistsessions output is corrupt
            # this should not be needed and is a workaround for the current X2go server implementation

            timeout = gevent.Timeout(maxwait)
            timeout.start()
            try:
                (stdin, stdout, stderr) = self._x2go_exec_command("export HOSTNAME && x2golistdesktops")
                _stdout_read = stdout.read()
                _listdesktops = _stdout_read.split('\n')
            except gevent.timeout.Timeout:
                # if we do not get a reply here after <maxwait> seconds we will raise a time out, we have to
                # make sure that we catch this at places where we want to ignore timeouts (e.g. in the 
                # desktop list cache)
                raise x2go_exceptions.X2goTimeOutException('x2golistdesktop command timed out')
            finally:
                timeout.cancel()

            return _listdesktops

    def list_sessions(self, raw=False):
        """\
        List all sessions of current user on the connected server.

        @param raw: if C{True}, the raw output of the server-side X2go command 
            C{x2golistsessions} is returned.
        @type raw: C{bool}

        @return: normally an instance of a C{X2goServerSessionList} backend Bis returned. However,
            if the raw argument is set, the plain text output of the x2golistsessions 
            command is returned
        @rtype: C{X2goServerSessionList} instance or str

        """
        if raw:
            (stdin, stdout, stderr) = self._x2go_exec_command("export HOSTNAME && x2golistsessions")
            return stdout.read(), stderr.read()

        else:

            # this _success loop will catch errors in case the x2golistsessions output is corrupt
            # this should not be needed and is a workaround for the current X2go server implementation
            _listsessions = {}
            _success = False
            _count = 0
            _maxwait = 20

            # we will try this 20 times before giving up... we might simply catch the x2golistsessions
            # output in the middle of creating a session in the database...
            while not _success and _count < _maxwait:
                _count += 1
                try:
                    (stdin, stdout, stderr) = self._x2go_exec_command("export HOSTNAME && x2golistsessions")
                    _stdout_read = stdout.read()
                    _listsessions = self._list_backend(_stdout_read, info_backend=self._info_backend).sessions
                    _success = True
                except KeyError:
                    gevent.sleep(1)
                except IndexError:
                    gevent.sleep(1)
                except ValueError:
                    gevent.sleep(1)

            if _count >= _maxwait:
                raise x2go_exceptions.X2goControlSessionException('x2golistsessions command failed after we have tried 20 times')

            # update internal variables when list_sessions() is called
            for _session_name, _session_info in self.associated_terminals.items():
                if _session_name not in _listsessions.keys():
                    del self.associated_terminals[_session_name]
                    self.terminated_terminals.append(_session_name)
                elif _session_info.is_suspended():
                    del self.associated_terminals[_session_name]

            return _listsessions

    def clean_sessions(self):
        """\
        Find X2go terminals that have previously been started by the
        connected user on the remote X2go server and terminate them.

        """
        session_list = self.list_sessions()
        for session_name in session_list.keys():
            self.terminate(session_name=session_name)

    def is_connected(self):
        """\
        Returns C{True} if this X2go session is connected to the remote server (that
        is if it has a valid Paramiko Transport object).

        @return: X2go session connected?
        @rtype: C{bool}

        """
        return self.get_transport() is not None and self.get_transport().is_authenticated()

    def is_running(self, session_name):
        """\
        Returns C{True} if the given X2go session is in running state,
        C{False} else.

        @param session_name: X2go name of the session to be queried
        @type session_name: str

        @return: X2go session running?
        @rtype: C{bool}

        """
        session_infos = self.list_sessions()
        if session_name in session_infos.keys():
            return session_infos[session_name].is_running()
        return None

    def is_suspended(self, session_name):
        """\
        Returns C{True} if the given X2go session is in suspended state,
        C{False} else.

        @return: X2go session suspended?
        @rtype: C{bool}

        """
        session_infos = self.list_sessions()
        if session_name in session_infos.keys():
            return session_infos[session_name].is_suspended()
        return None

    def has_terminated(self, session_name):
        """\
        Returns C{True} if this X2go session is not in the session list on the 
        connected server, C{False} else.

        Of course, if this command is called before session startup, it will also
        return C{True}.

        @return: X2go session has terminate?
        @rtype: C{bool}

        """
        session_infos = self.list_sessions()

        if session_name not in session_infos.keys():
            if session_name in self.terminated_terminals:
                return True
            else:
                # do a post-mortem tidy up
                if session_name in self.associated_terminals.keys():
                    self.terminate(session_name)
                return True

        return False

    def suspend(self, session_name):
        """\
        Suspend either this or another available X2go session on the connected
        server.

        If L{session_name} is given, L{X2goControlSessionSTDOUT.suspend()} tries to suspend the
        corresponding session.

        @param session_name: X2go name of the session to be suspended
        @type session_name: str

        @return: True if the session could be successfully suspended
        @rtype: C{bool}

        """
        _ret = False
        _session_names = [ t.get_session_name() for t in self.associated_terminals.values() ]
        if session_name in _session_names:

            self.logger('suspending associated terminal session: %s' % session_name, loglevel=log.loglevel_DEBUG)
            (stdin, stdout, stderr) = self._x2go_exec_command("x2gosuspend-session %s" % session_name, loglevel=log.loglevel_DEBUG)
            dummy_stdout = stdout.read()
            dummy_stderr = stderr.read()
            if self.associated_terminals.has_key(session_name):
                if self.associated_terminals[session_name] is not None:
                    self.associated_terminals[session_name].__del__()
                try: del self.associated_terminals[session_name]
                except KeyError: pass
            _ret = True

        else:

            self.logger('suspending non-associated terminal session: %s' % session_name, loglevel=log.loglevel_DEBUG)
            (stdin, stdout, stderr) = self._x2go_exec_command("x2gosuspend-session %s" % session_name, loglevel=log.loglevel_DEBUG)
            dummy_stdout = stdout.read()
            dummy_stderr = stderr.read()
            _ret = True

        return _ret

    def terminate(self, session_name):
        """\
        Terminate either this or another available X2go session on the connected
        server.

        If L{session_name} is given, L{X2goControlSessionSTDOUT.terminate()} tries to terminate the
        corresponding session.

        @param session_name: X2go name of the session to be terminated
        @type session_name: str

        @return: True if the session could be successfully terminate
        @rtype: C{bool}

        """

        _ret = False
        _session_names = [ t.get_session_name() for t in self.associated_terminals.values() ]
        if session_name in _session_names:

            self.logger('terminating associated session: %s' % session_name, loglevel=log.loglevel_DEBUG)
            (stdin, stdout, stderr) = self._x2go_exec_command("x2goterminate-session %s" % session_name, loglevel=log.loglevel_DEBUG)
            dummy_stdout = stdout.read()
            dummy_stderr = stderr.read()
            if self.associated_terminals.has_key(session_name):
                if self.associated_terminals[session_name] is not None:
                    self.associated_terminals[session_name].__del__()
                try: del self.associated_terminals[session_name]
                except KeyError: pass
            self.terminated_terminals.append(session_name)
            _ret = True

        else:

            self.logger('terminating non-associated session: %s' % session_name, loglevel=log.loglevel_DEBUG)
            (stdin, stdout, stderr) = self._x2go_exec_command("x2goterminate-session %s" % session_name, loglevel=log.loglevel_DEBUG)
            dummy_stdout = stdout.read()
            dummy_stderr = stderr.read()
            _ret = True

        return _ret


