# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goSession class - a public API of Python X2go, handling standalone X2go 
sessions.

This class is normally embedded into the context of an L{X2goClient}
instance, but it is also possible to address L{X2goSession}s directly via this
class.

"""
__NAME__ = 'x2gosession-pylib'

import os
import copy
import types
import uuid
import time
import threading
import gevent

# Python X2go modules
import log
import utils
import session
from x2go_exceptions import *

from x2go.backends.control import X2goControlSession as _X2goControlSession
from x2go.backends.terminal import X2goTerminalSession as _X2goTerminalSession
from x2go.backends.info import X2goServerSessionInfo as _X2goServerSessionInfo
from x2go.backends.info import X2goServerSessionList as _X2goServerSessionList
from x2go.backends.proxy import X2goProxy as _X2goProxy
from x2go.backends.profiles import X2goSessionProfiles as _X2goSessionProfiles
from x2go.backends.settings import X2goClientSettings as _X2goClientSettings
from x2go.backends.printing import X2goClientPrinting as _X2goClientPrinting

from defaults import LOCAL_HOME as _LOCAL_HOME
from defaults import X2GO_CLIENT_ROOTDIR as _X2GO_CLIENT_ROOTDIR
from defaults import X2GO_SESSIONS_ROOTDIR as _X2GO_SESSIONS_ROOTDIR
from defaults import X2GO_SSH_ROOTDIR as _X2GO_SSH_ROOTDIR

from defaults import SUPPORTED_SOUND, SUPPORTED_PRINTING, SUPPORTED_FOLDERSHARING, SUPPORTED_MIMEBOX

# options of the paramiko.SSHClient().connect()
_X2GO_SESSION_PARAMS = ('geometry', 'depth', 'link', 'pack',
                        'cache_type', 'kblayout', 'kbtype',
                        'session_type', 'snd_system', 'snd_port',
                        'cmd',
                        'rdp_server', 'rdp_options',
                        'xdmcp_server',
                        'rootdir', 'loglevel', 'profile_name', 'profile_id',
                        'print_action', 'print_action_args',
                        'convert_encoding', 'client_encoding', 'server_encoding',
                        'proxy_options', 
                        'logger',
                        'control_backend', 'terminal_backend', 'proxy_backend',
                        'profiles_backend', 'settings_backend', 'printing_backend',
                       )
"""A list of allowed X2go session parameters."""
_X2GO_SSHPROXY_PARAMS = ('sshproxy_host', 'sshproxy_user', 'sshproxy_password',
                         'sshproxy_key_filename', 'sshproxy_pkey', 'sshproxy_tunnel',
                        )
"""A list of allowed X2go SSH proxy parameters."""


class X2goSession(object):
    """\
    Public API class for launching X2go sessions. Recommended is to manage X2go sessions from
    within an L{X2goClient} instance. However, Python X2go is designed in a way that it also
    allows the management of singel L{X2goSession} instance.

    Thus, you can use the L{X2goSession} class to manually set up X2go sessions without 
    L{X2goClient} context (session registry, session list cache, auto-registration of new
    sessions etc.).

    """
    def __init__(self, server=None, control_session=None,
                 use_sshproxy=False,
                 profile_id=None, profile_name='UNKNOWN',
                 session_name=None,
                 printing=False,
                 allow_mimebox=False,
                 mimebox_extensions=[],
                 mimebox_action='OPEN',
                 allow_share_local_folders=False,
                 share_local_folders=[],
                 control_backend=_X2goControlSession,
                 terminal_backend=_X2goTerminalSession,
                 info_backend=_X2goServerSessionInfo,
                 list_backend=_X2goServerSessionList,
                 proxy_backend=_X2goProxy,
                 settings_backend=_X2goClientSettings,
                 printing_backend=_X2goClientPrinting,
                 client_rootdir=os.path.join(_LOCAL_HOME, _X2GO_CLIENT_ROOTDIR),
                 sessions_rootdir=os.path.join(_LOCAL_HOME, _X2GO_SESSIONS_ROOTDIR),
                 ssh_rootdir=os.path.join(_LOCAL_HOME, _X2GO_SSH_ROOTDIR),
                 keep_controlsession_alive=False,
                 add_to_known_hosts=False,
                 known_hosts=None,
                 logger=None, loglevel=log.loglevel_DEFAULT,
                 connected=False, virgin=True, running=None, suspended=None, terminated=None, faulty=None,
                 client_instance=None,
                 **params):
        """\
        @param server: hostname of X2go server
        @type server: C{str}
        @param control_session: an already initialized C{X2goControlSession*} instance
        @type control_session: C{X2goControlSession*} instance
        @param use_sshproxy: for communication with X2go server use an SSH proxy host
        @type use_sshproxy: C{bool}
        @param profile_id: profile ID
        @type profile_id: C{str}
        @param profile_name: profile name
        @type profile_name: C{str}
        @param session_name: session name (if available)
        @type session_name: C{str}
        @param printing: enable X2go printing
        @type printing: C{bool}
        @param allow_mimebox: enable X2go MIME box support
        @type allow_mimebox: C{bool}
        @param mimebox_extensions: whitelist of allowed X2go MIME box extensions
        @type mimebox_extensions: C{list}
        @param mimebox_action: action for incoming X2go MIME box files
        @type mimebox_action: C{X2goMimeBoxAction*} or C{str}
        @param allow_share_local_folders: enable local folder sharing support
        @type allow_share_local_folders: C{bool}
        @param share_local_folders: list of local folders to share with the remote X2go session
        @type share_local_folders: C{list}
        @param control_backend: X2go control session backend to use
        @type control_backend: C{class}
        @param terminal_backend: X2go terminal session backend to use
        @type terminal_backend: C{class}
        @param info_backend: X2go session info backend to use
        @type info_backend: C{class}
        @param list_backend: X2go session list backend to use
        @type list_backend: C{class}
        @param proxy_backend: X2go proxy backend to use
        @type proxy_backend: C{class}
        @param settings_backend: X2go client settings backend to use
        @type settings_backend: C{class}
        @param printing_backend: X2go client printing backend to use
        @type printing_backend: C{class}
        @param client_rootdir: client base dir (default: ~/.x2goclient)
        @type client_rootdir: C{str}
        @param sessions_rootdir: sessions base dir (default: ~/.x2go)
        @type sessions_rootdir: C{str}
        @param ssh_rootdir: ssh base dir (default: ~/.ssh)
        @type ssh_rootdir: C{str}
        @param keep_controlsession_alive: On last L{X2goSession.disconnect()} keep the associated C{X2goControlSession*} instance alive?
        @ŧype keep_controlsession_alive: C{bool}
        @param add_to_known_hosts: Auto-accept server host validity?
        @type add_to_known_hosts: C{bool}
        @param known_hosts: the underlying Paramiko/SSH systems C{known_hosts} file
        @type known_hosts: C{str}
        @param connected: manipulate session state »connected« by giving a pre-set value
        @type connected: C{bool}
        @param virgin: manipulate session state »virgin« by giving a pre-set value
        @type virgin: C{bool}
        @param running: manipulate session state »running« by giving a pre-set value
        @type running: C{bool}
        @param suspended: manipulate session state »suspended« by giving a pre-set value
        @type suspended: C{bool}
        @param terminated: manipulate session state »terminated« by giving a pre-set value
        @type terminated: C{bool}
        @param faulty: manipulate session state »faulty« by giving a pre-set value
        @type faulty: C{bool}
        @param client_instance: if available, the underlying L{X2goClient} instance
        @type client_instance: C{X2goClient} instance
        @param params: further control session, terminal session and SSH proxy class options
        @type params: C{dict}

        """
        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        self._keep = None

        self.uuid = uuid.uuid1()
        self.connected = connected

        self.virgin = virgin
        self.running = running
        self.suspended = suspended
        self.terminated = terminated
        self.faulty = faulty
        self.keep_controlsession_alive = keep_controlsession_alive

        self.profile_id = profile_id
        self.profile_name = profile_name
        self.session_name = session_name
        self.server = server

        self._last_status = None

        self.locked = False

        self.printing = printing
        self.allow_share_local_folders = allow_share_local_folders
        self.share_local_folders = share_local_folders
        self.allow_mimebox = allow_mimebox
        self.mimebox_extensions = mimebox_extensions
        self.mimebox_action = mimebox_action
        self.control_backend = control_backend
        self.terminal_backend = terminal_backend
        self.info_backend = info_backend
        self.list_backend = list_backend
        self.proxy_backend = proxy_backend
        self.settings_backend = settings_backend
        self.printing_backend = printing_backend
        self.client_rootdir = client_rootdir
        self.sessions_rootdir = sessions_rootdir
        self.ssh_rootdir = ssh_rootdir
        self.control_session = control_session

        self.control_params = {}
        self.terminal_params = {}
        self.sshproxy_params = {}
        self.update_params(params)

        self.session_environment = {}

        try: del self.control_params['server']
        except: pass

        self.client_instance = client_instance

        if self.logger.get_loglevel() & log.loglevel_DEBUG:
            self.logger('X2go control session parameters for profile %s:' % profile_name, loglevel=log.loglevel_DEBUG)
            for p in self.control_params:
                self.logger('    %s: %s' % (p, self.control_params[p]), log.loglevel_DEBUG)
            self.logger('X2go terminal session parameters for profile %s:' % profile_name, loglevel=log.loglevel_DEBUG)
            for p in self.terminal_params:
                self.logger('    %s: %s' % (p,self.terminal_params[p]), log.loglevel_DEBUG)
            self.logger('X2go sshproxy parameters for profile %s:' % profile_name, loglevel=log.loglevel_DEBUG)
            for p in self.sshproxy_params:
                self.logger('    %s: %s' % (p,self.sshproxy_params[p]), loglevel=log.loglevel_DEBUG)

        self.add_to_known_hosts = add_to_known_hosts
        self.known_hosts = known_hosts
        self.use_sshproxy = use_sshproxy

        self._current_status = {
            'timestamp': time.time(),
            'server': self.server,
            'virgin': self.virgin,
            'connected': self.connected,
            'running': self.running,
            'suspended': self.suspended,
            'terminated': self.terminated,
            'faulty': self.faulty,
        }

        self.init_control_session()
        self.terminal_session = None

    def HOOK_rforward_request_denied(self, server_port=0):
        """\
        HOOK method: called if a reverse port forwarding request has been denied.

        @param server_port: remote server port (starting point of reverse forwarding tunnel)
        @type server_port: C{str}

        """
        if self.client_instance:
            self.client_instance.HOOK_rforward_request_denied(profile_name=self.profile_name, session_name=self.session_name, server_port=server_port)
        else:
            self.logger('HOOK_rforward_request_denied: TCP port (reverse) forwarding request for session %s to server port %s has been denied by server %s. This is a common issue with SSH, it might help to restart the server\'s SSH daemon.' % (self.session_name, server_port, self.profile_name), loglevel=log.loglevel_WARN)

    def HOOK_forwarding_tunnel_setup_failed(self, chain_host='UNKNOWN', chain_port=0):
        """\
        HOOK method: called if a port forwarding tunnel setup failed.

        @param chain_host: hostname of chain host (forwarding tunnel end point)
        @type chain_host: C{str}
        @param chain_port: port of chain host (forwarding tunnel end point)
        @type chain_port: C{str}

        """
        # mark session as faulty
        self.faulty = True

        if self.client_instance:
            self.client_instance.HOOK_forwarding_tunnel_setup_failed(profile_name=self.profile_name, session_name=self.session_name, chain_host=chain_host, chain_port=chain_port)
        else:
            self.logger('HOOK_forwarding_tunnel_setup_failed: Forwarding tunnel request to [%s]:%s for session %s (%s) was denied by remote X2go/SSH server. Session startup failed.' % (chain_host, chain_port, self.session_name, self.profile_name), loglevel=log.loglevel_WARN)

        # get rid of the faulty session...
        self.terminate()

    def HOOK_check_host_dialog(self, host, port, fingerprint='no fingerprint', fingerprint_type='RSA'):
        """\
        HOOK method: called if a host check is requested. This hook has to either return C{True} (default) or C{False}.

        @param host: SSH server name to validate
        @type host: C{str}
        @param port: SSH server port to validate
        @type port: C{int}
        @param fingerprint: the server's fingerprint
        @type fingerprint: C{str}
        @param fingerprint_type: finger print type (like RSA, DSA, ...)
        @type fingerprint_type: C{str}
        @return: if host validity is verified, this hook method should return C{True}
        @rtype: C{bool}

        """
        if self.client_instance:
            return self.client_instance.HOOK_check_host_dialog(profile_name=self.profile_name, host=host, port=port, fingerprint=fingerprint, fingerprint_type=fingerprint_type)
        else:
            self.logger('HOOK_check_host_dialog: host check requested for [%s]:%s with %s fingerprint: ,,%s.\'\'. Automatically adding host as known host.' % (host, port, fingerprint_type, fingerprint), loglevel=log.loglevel_WARN)
            return True

    def init_control_session(self):
        """\
        Initialize a new control session (C{X2goControlSession*}).

        """
        if self.control_session is None:
            self.logger('initializing X2goControlSession', loglevel=log.loglevel_DEBUG)
            self.control_session = self.control_backend(profile_name=self.profile_name,
                                                        add_to_known_hosts=self.add_to_known_hosts,
                                                        known_hosts=self.known_hosts,
                                                        terminal_backend=self.terminal_backend,
                                                        info_backend=self.info_backend,
                                                        list_backend=self.list_backend,
                                                        proxy_backend=self.proxy_backend,
                                                        client_rootdir=self.client_rootdir,
                                                        sessions_rootdir=self.sessions_rootdir,
                                                        ssh_rootdir=self.ssh_rootdir,
                                                        logger=self.logger)

    def set_server(self, server):
        """\
        Modify server name after L{X2goSession} has already been initialized.

        @param server: new server name
        @type server: C{str}

        """
        self.server = server

    def set_profile_name(self, profile_name):
        """\
        Modify session profile name after L{X2goSession} has already been initialized.

        @param profile_name: new session profile name
        @type profile_name: C{str}

        """
        self.profile_name = profile_name
        self.control_session.set_profile_name(profile_name)

    def __str__(self):
        return self.__get_uuid()

    def __repr__(self):
        result = 'X2goSession('
        for p in dir(self):
            if '__' in p or not p in self.__dict__ or type(p) is types.InstanceType: continue
            result += p + '=' + str(self.__dict__[p]) + ', '
        return result + ')'

    def __call__(self):
        return self.__get_uuid()

    def __del__(self):
        """\
        Class destructor.

        """
        if self.has_control_session() and self.has_terminal_session():
            self.get_control_session().dissociate(self.get_terminal_session())

        if self.has_control_session():
            if self.keep_controlsession_alive:
                # regenerate this session instance for re-usage if this is the last session for a certain session profile
                # and keep_controlsession_alive is set to True...
                self.virgin = True
                self.connected = self.is_connected()
                self.running = None
                self.suspended = None
                self.terminated = None
                self._current_status = {
                    'timestamp': time.time(),
                    'server': self.server,
                    'virgin': self.virgin,
                    'connected': self.connected,
                    'running': self.running,
                    'suspended': self.suspended,
                    'terminated': self.terminated,
                    'faulty': self.faulty,
                }
                self._last_status = None
                self.session_name = None

            else:
                self.get_control_session().__del__()
                self.control_session = None

        if self.has_terminal_session():
            self.get_terminal_session().__del__()
            self.terminal_session = None

    def update_params(self, params):
        """\
        This method can be used to modify L{X2goSession} parameters after the
        L{X2goSession} instance has already been initialized.

        @param params: a Python dictionary with L{X2goSession} parameters
        @type params: C{dict}

        """
        try: del params['server'] 
        except KeyError: pass
        try: del params['profile_name']
        except KeyError: pass
        try: del params['profile_id'] 
        except KeyError: pass
        try:
            self.printing = params['printing']
            del params['printing'] 
        except KeyError: pass
        try: 
            self.allow_share_local_folders = params['allow_share_local_folders']
            del params['allow_share_local_folders']
        except KeyError: pass
        try:
            self.share_local_folders = params['share_local_folders']
            del params['share_local_folders'] 
        except KeyError: pass
        try:
            self.allow_mimebox = params['allow_mimebox']
            del params['allow_mimebox']
        except KeyError: pass
        try:
            self.mimebox_extensions = params['mimebox_extensions']
            del params['mimebox_extensions']
        except KeyError: pass
        try: 
            self.mimebox_action = params['mimebox_action']
            del params['mimebox_action']
        except KeyError: pass
        try:
            self.use_sshproxy = params['use_sshproxy']
            del params['use_sshproxy']
        except KeyError: pass

        _terminal_params = copy.deepcopy(params)
        _control_params = copy.deepcopy(params)
        _sshproxy_params = copy.deepcopy(params)
        for p in params.keys():
            if p in session._X2GO_SESSION_PARAMS:
                del _control_params[p]
                del _sshproxy_params[p]
            elif p in session._X2GO_SSHPROXY_PARAMS:
                del _control_params[p]
                del _terminal_params[p]
            else:
                del _sshproxy_params[p]
                del _terminal_params[p]

        self.control_params.update(_control_params)
        self.terminal_params.update(_terminal_params)
        self.sshproxy_params.update(_sshproxy_params)

    def get_uuid(self):
        """\
        Retrieve session UUID hash for this L{X2goSession}.

        """
        return str(self.uuid)
    __get_uuid = get_uuid

    def get_username(self):
        """\
        After a session has been set up you can query the
        username the sessions runs as.

        @return: the remote username the X2go session runs as
        @rtype: C{str}

        """
        # try to retrieve the username from the control session, if already connected
        try:
            return self.control_session.get_transport().get_username()
        except AttributeError:
            return self.control_params['username']
    __get_username = get_username


    def user_is_x2gouser(self, username=None):
        """\
        Check if a given user is valid server-side X2go user.

        @param username: username to check validity for
        @type username: C{str}
        @return: return C{True} if the username is allowed to launch X2go sessions
        @rtype: C{bool}

        """
        if username is None:
            username = self.__get_username()
        return self.control_session.is_x2gouser(username)
    __user_is_x2gouser = user_is_x2gouser

    def get_password(self):
        """\
        After a session has been setup up you can query the
        username's password from the session.

        @return: the username's password
        @rtype: C{str}

        """
        return self.control_session._session_password
    __get_password = get_password

    def get_server_peername(self):
        """\
        After a session has been setup up you can query the
        peername of the host this session is connected to (or
        about to connect to).

        @return: the address of the server the X2go session is
            connected to (as an C{(addr,port)} tuple)
        @rtype: C{tuple}

        """
        return self.control_session.get_transport().getpeername()
    __get_server_peername = get_server_peername

    def get_server_hostname(self):
        """\
        After a session has been setup up you can query the
        hostname of the host this session is connected to (or
        about to connect to).

        @return: the hostname of the server the X2go session is
            connected to / about to connect to
        @rtype: C{str}

        """
        self.server = self.control_session.hostname
        return self.server
    __get_server_hostname = get_server_hostname

    def get_server_port(self):
        """\
        After a session has been setup up you can query the
        IP socket port used for connecting the remote X2go server.

        @return: the server-side IP socket port that is used by the X2go session to
            connect to the server
        @rtype: C{str}

        """
        return self.control_session.port
    __get_server_port = get_server_port

    def get_session_name(self):
        """\
        Retrieve the server-side X2go session name for this session.

        @return: X2go session name
        @rtype: C{str}

        """
        return self.session_name
    __get_session_name = get_session_name

    def get_session_cmd(self):
        """\
        Retrieve the server-side command that is used to start a session
        on the remote X2go server.

        @return: server-side session command
        @rtype: C{str}

        """
        if self.terminal_params.has_key('cmd'):
            return self.terminal_params['cmd']
        return None
    __get_session_cmd = get_session_cmd

    def get_control_session(self):
        """\
        Retrieve the control session (C{X2goControlSession*} backend) of this L{X2goSession}.

        @return: the L{X2goSession}'s control session
        @rtype: C{X2goControlSession*} instance
        """
        return self.control_session
    __get_control_session = get_control_session

    def has_control_session(self):
        """\
        Check if this L{X2goSession} instance has an associated control session.

        @return: returns C{True} if this L{X2goSession} has a control session associated to itself
        @rtype: C{bool}

        """
        return self.control_session is not None
    __has_control_session = has_control_session

    def get_terminal_session(self):
        """\
        Retrieve the terminal session (C{X2goTerminalSession*} backend) of this L{X2goSession}.

        @return: the L{X2goSession}'s terminal session
        @rtype: C{X2goControlTerminal*} instance

        """
        if self.terminal_session == 'PENDING':
            return None
        return self.terminal_session
    __get_terminal_session = get_terminal_session

    def has_terminal_session(self):
        """\
        Check if this L{X2goSession} instance has an associated terminal session.

        @return: returns C{True} if this L{X2goSession} has a terminal session associated to itself
        @rtype: C{bool}


        """
        return self.terminal_session not in (None, 'PENDING')
    __has_terminal_session = has_terminal_session

    def check_host(self):
        """\
        Provide a host check mechanism. This method basically calls the L{HOOK_check_host_dialog()} method
        which by itself calls the L{X2goClient.HOOK_check_host_dialog()} method. Make sure you
        override any of these to enable user interaction on X2go server validity checks.

        @return: returns C{True} if an X2go server host is valid for authentication
        @rtype: C{bool}

        """
        if self.connected:
            return True

        _port = self.control_params['port']
        (_valid, _host, _port, _fingerprint, _fingerprint_type) = self.control_session.check_host(self.server, port=_port)
        return _valid or self.HOOK_check_host_dialog(host=_host, port=_port, fingerprint=_fingerprint, fingerprint_type=_fingerprint_type)
    __check_host = check_host

    def can_auto_connect(self):
        """\
        Check if a session is configured adequately to be able to auto-connect to the X2go
        server (e.g. public key authentication).

        @return: returns C{True} if the session can auto-connect, C{False} otherwise, C{None}
            if no control session has been set up yet.
        @rtype: C{bool}

        """

        def _can_sshproxy_autoconnect():

            if self.use_sshproxy:
                if self.sshproxy_params.has_key('sshproxy_key_filename') and self.sshproxy_params['sshproxy_key_filename'] and os.path.exists(os.path.normpath(self.sshproxy_params['sshproxy_key_filename'])):
                    return True
                elif self.sshproxy_params.has_key('sshproxy_pkey') and self.sshproxy_params['sshproxy_pkey']:
                    return True
                else:
                    return False
            else:
                return True

        # do we have a key file passed as control parameter?
        if self.control_params.has_key('key_filename') and self.control_params['key_filename'] and os.path.exists(os.path.normpath(self.control_params['key_filename'])):
            return _can_sshproxy_autoconnect()

        # or a private key?
        elif self.control_params.has_key('pkey') and self.control_params['pkey']:
            return _can_sshproxy_autoconnect()

        else:
            return False
    __can_auto_connect = can_auto_connect

    def connect(self, username='', password='', add_to_known_hosts=False, force_password_auth=False,
                use_sshproxy=False, sshproxy_user='', sshproxy_password=''):
        """\
        Connects to the L{X2goSession}'s server host. This method basically wraps around 
        the C{X2goControlSession*.connect()} method.

        @param username: the username for the X2go server that is going to be
            connected to (as a last minute way of changing the session username)
        @type username: C{str}
        @param password: the user's password for the X2go server that is going to be
            connected to
        @type password: C{str}
        @param add_to_known_hosts: non-paramiko option, if C{True} paramiko.AutoAddPolicy()
            is used as missing-host-key-policy. If set to C{False} paramiko.RejectPolicy()
            is used
        @type add_to_known_hosts: C{bool}
        @param force_password_auth: disable SSH pub/priv key authentication mechanisms
            completely
        @type force_password_auth: C{bool}
        @param use_sshproxy: use an SSH proxy host for connecting the target X2go server
        @type use_sshproxy: C{bool}
        @param sshproxy_user: username for authentication against the SSH proxy host
        @type sshproxy_user: C{str}
        @param sshproxy_password: password for authentication against the SSH proxy host
        @type sshproxy_password: C{str}

        @return: returns C{True} is the connection to the X2go server has been successful
        @rtype C{bool}

        """
        if self.control_session and self.control_session.is_connected():
            self.logger('control session is already connected, skipping authentication', loglevel=log.loglevel_DEBUG)
            self.connected = True
        else:
            if username:
                self.control_params['username'] = username
            if add_to_known_hosts is not None:
                self.control_params['add_to_known_hosts'] = add_to_known_hosts
            if force_password_auth is not None:
                self.control_params['force_password_auth'] = force_password_auth
            if sshproxy_user:
                self.sshproxy_params['sshproxy_user'] = sshproxy_user
            if sshproxy_password:
                self.sshproxy_params['sshproxy_password'] = sshproxy_password
            self.control_params['password'] = password

            _params = {}
            _params.update(self.control_params)
            _params.update(self.sshproxy_params)

            try:
                self.connected = self.control_session.connect(self.server,
                                                              use_sshproxy=self.use_sshproxy, 
                                                              session_instance=self, 
                                                              **_params)
            except X2goControlSessionException, e:
                raise X2goSessionException(str(e))
            except:
                # remove credentials immediately
                self.control_params['password'] = ''
                if self.sshproxy_params and self.sshproxy_params.has_key('sshproxy_password'):
                    del self.sshproxy_params['sshproxy_password']
                raise
            finally:
                # remove credentials immediately
                self.control_params['password'] = ''
                if self.sshproxy_params and self.sshproxy_params.has_key('sshproxy_password'):
                    del self.sshproxy_params['sshproxy_password']

            if not self.connected:
                # then tidy up...
                self.disconnect()

            _dummy = self.get_server_hostname()

        if self.connected:
            self.update_status()

        return self.connected
    __connect = connect

    def disconnect(self):
        """\
        Disconnect this L{X2goSession} instance.

        @return: returns C{True} if the disconnect operation has been successful
        @rtype: C{bool}

        """
        self.connected = False
        self.running = None
        self.suspended = None
        self.terminated = None
        self.faults = None
        try:
            self.update_status(force_update=True)
        except X2goControlSessionException:
            pass
        retval = self.control_session.disconnect()
        return retval
    __disconnect = disconnect

    def set_print_action(self, print_action, **kwargs):
        """\
        If X2go client-side printing is enable within this X2go session you can use
        this method to alter the way how incoming print spool jobs are handled/processed.

        For further information, please refer to the documentation of the L{X2goClient.set_session_print_action()}
        method.

        @param print_action: one of the named above print actions, either as string or class instance
        @type print_action: C{str} or C{instance}
        @param kwargs: additional information for the given print action (print 
            action arguments), for possible print action arguments and their values see each individual
            print action class
        @type kwargs: C{dict}

        """
        if type(print_action) is not types.StringType:
            return False
        self.terminal_session.set_print_action(print_action, **kwargs)
    __set_print_action = set_print_action

    def is_alive(self):
        """\
        Find out if this X2go session is still alive (that is: connected to the server).

        @return: returns C{True} if the server connection is still alive
        @rtype: C{bool}

        """
        self.connected = self.control_session.is_alive()
        if not self.connected:
            self._X2goSession__disconnect()
        return self.connected
    __is_alive = is_alive

    def clean_sessions(self):
        """\
        Clean all running sessions for the authenticated user on the remote X2go server.

        """
        if self.is_alive():
            self.control_session.clean_sessions()
        else:
            self._X2goSession__disconnect()
    __clean_sessions = clean_sessions

    def list_sessions(self, raw=False):
        """\
        List all sessions on the remote X2go server that are owned by the authenticated user 

        @param raw: if C{True} the output of this method equals
            the output of the server-side C{x2golistsessions} command
        @type raw: C{bool}

        @return: a session list (as data object or list of strings when called with C{raw=True} option)
        @rtype: C{X2goServerSessionList*} instance or C{list}

        """
        try:
            return self.control_session.list_sessions(raw=raw)
        except X2goControlSessionException:
            self._X2goSession_disconnect()
            return None
    __list_sessions = list_sessions

    def list_desktops(self, raw=False):
        """\
        List X2go desktops sessions available for desktop sharing on the remote X2go server.

        @param raw: if C{True} the output of this method equals
            the output of the server-side C{x2golistdesktops} command
        @type raw: C{bool}

        @return: a list of strings representing available desktop sessions
        @rtype: C{list}

        """
        try:
            return self.control_session.list_desktops(raw=raw)
        except X2goDesktopSharingException:
            if raw:
                return ('','')
            else:
                return []
        except X2goControlSessionException:
            self._X2goSession_disconnect()
            return None
    __list_desktops = list_desktops

    def update_status(self, session_list=None, force_update=False):
        """\
        Update the current session status. The L{X2goSession} instance uses an internal
        session status cache that allows to query the session status without the need
        of retrieving data from the remote X2go server for each query.

        The session status (if initialized properly with the L{X2goClient} constructor gets
        updated in regularly intervals.

        In case you use the L{X2goSession} class in standalone instances (that is: without
        being embedded into an L{X2goSession} context) then run this method in regular
        intervals to make sure the L{X2goSession}'s internal status cache information
        is always up-to-date.

        @param session_list: provide an C{X2goServerSessionList*} that refers to X2go sessions we want to update.
            This option is mainly for reducing server/client traffic.
        @type session_list: C{X2goServerSessionList*} instance
        @param force_update: force a session status update, if if the last update is less then 1 second ago
        @type force_update: C{bool}

        """
        if not force_update and self._last_status is not None:
            _status_update_timedelta = time.time() - self._last_status['timestamp']

            # skip this session status update if not longer than a second ago...
            if  _status_update_timedelta < 1:
                self.logger('status update interval too short (%s), skipping status about this time...' % _status_update_timedelta, loglevel=log.loglevel_DEBUG)
                return False

        e = None
        self._last_status = copy.deepcopy(self._current_status)
        if session_list is None:
            try:
                session_list = self.control_session.list_sessions()
                self.connected = True
            except X2goControlSessionException, e:
                self.connected = False
                self.running = None
                self.suspended = None
                self.terminated = None
                self.faulty = None

        if self.connected:
            try:
                _session_name = self.get_session_name()
                _session_info = session_list[_session_name]
                self.running = _session_info.is_running()
                self.suspended = _session_info.is_suspended()
                if not self.virgin:
                    self.terminated = not (self.running or self.suspended)
                else:
                    self.terminated = None
            except KeyError:
                self.running = False
                self.suspended = False
                if not self.virgin:
                    self.terminated = True
            self.faulty = not (self.running or self.suspended or self.terminated or self.virgin)


        self._current_status = {
            'timestamp': time.time(),
            'server': self.server,
            'virgin': self.virgin,
            'connected': self.connected,
            'running': self.running,
            'suspended': self.suspended,
            'terminated': self.terminated,
            'faulty': self.faulty,
        }

        if (not self.connected or self.faulty) and e:
            raise e

        return True

    __update_status = update_status

    def resume(self, session_name=None):
        """\
        Resume or continue a suspended / running X2go session on the
        remote X2go server.

        @param session_name: the server-side name of an X2go session
        @type session_name: C{str}

        @return: returns C{True} if resuming the session has been successful, C{False} otherwise
        @rtype: C{bool}

        """
        self.terminal_session = 'PENDING'
        _new_session = False
        if self.session_name is None:
            self.session_name = session_name

        if self.is_alive():
            _control = self.control_session

            # FIXME: normally this part gets called if you suspend a session that is associated to another client
            # we do not have a possibility to really check if SSH has released port forwarding channels or
            # sockets, thus  we plainly have to wait a while
            if self.is_running():
                self.suspend()
                gevent.sleep(10)

            self.terminal_session = _control.resume(session_name=self.session_name,
                                                    session_instance=self,
                                                    logger=self.logger, **self.terminal_params)

            if self.session_name is None:
                _new_session = True
                try:
                    self.session_name = self.terminal_session.session_info.name
                except AttributeError:
                    raise X2goSessionException('start of new X2go session failed')

            if self.has_terminal_session() and not self.faulty:

                if SUPPORTED_SOUND and self.terminal_session.params.snd_system is not 'none':
                    self.terminal_session and not self.faulty and self.terminal_session.start_sound()

                if (SUPPORTED_PRINTING and self.printing) or \
                   (SUPPORTED_MIMEBOX and self.allow_mimebox) or \
                   (SUPPORTED_FOLDERSHARING and self.allow_share_local_folders):
                    self.terminal_session and not self.faulty and self.terminal_session.start_sshfs()

                try:
                    if SUPPORTED_PRINTING and self.printing:
                        self.terminal_session and not self.faulty and self.terminal_session.start_printing()
                        self.terminal_session and not self.faulty and self.session_environment.update({'X2GO_SPOOLDIR': self.terminal_session.get_printing_spooldir(), })
                except X2goUserException:
                    pass

                if SUPPORTED_MIMEBOX and self.allow_mimebox:
                    self.terminal_session and not self.faulty and self.terminal_session.start_mimebox(mimebox_extensions=self.mimebox_extensions, mimebox_action=self.mimebox_action)
                    self.session_environment.update({'X2GO_MIMEBOX': self.terminal_session.get_mimebox_spooldir(), })

                if SUPPORTED_FOLDERSHARING and self.share_local_folders:
                    if _control.get_transport().reverse_tunnels[self.terminal_session.get_session_name()]['sshfs'][1] is not None:
                        for _folder in self.share_local_folders:
                            self.share_local_folder(_folder)

                # only run the session startup command if we do not resume...
                if _new_session:
                    self.terminal_session.run_command(env=self.session_environment)

                self.virgin = False
                self.suspended = False
                self.running = True
                self.terminated = False
                self.faulty = False

                return True

            else:
                self.terminal_session = None
                return False

            return self.running
        else:
            self._X2goSession__disconnect()
            return False

    __resume = resume

    def start(self):
        """\
        Start a new X2go session on the remote X2go server.

        @return: returns C{True} if starting the session has been successful, C{False} otherwise
        @rtype: C{bool}

        """
        self.session_name = None
        return self.resume()
    __start = start

    def share_desktop(self, desktop=None, user=None, display=None, share_mode=0, check_desktop_list=True):
        """\
        Share an already running X2go session on the remote X2go server locally. The shared session may be either
        owned by the same user or by a user that grants access to his/her desktop session by the local user.

        @param desktop: desktop ID of a sharable desktop in format <user>@<display>
        @type desktop: C{str}
        @param user: user name and display number can be given separately, here give the
            name of the user who wants to share a session with you.
        @type user: C{str}
        @param display: user name and display number can be given separately, here give the
            number of the display that a user allows you to be shared with.
        @type display: C{str}
        @param share_mode: desktop sharing mode, 0 is VIEW-ONLY, 1 is FULL-ACCESS.
        @type share_mode: C{int}
        @param check_desktop_list: check if the given desktop is available on the X2go server; handle with care as
            the server-side C{x2golistdesktops} command might block client I/O.
        @type check_desktop_list: C{bool}

        @return: returns C{True} if starting the session has been successful, C{False} otherwise
        @rtype: C{bool}

        """
        self.terminal_session = 'PENDING'

        _desktop = desktop or '%s@%s' % (user, display)
        if check_desktop_list:
            if not _desktop in self._X2goSession__list_desktops():
                _orig_desktop = _desktop
                _desktop = '%s.0' % _desktop
                if not _desktop in self._X2GoSession__list_desktops():
                    raise X2goDesktopSharingException('No such desktop ID: %s' % _orig_desktop)

        _session_owner = _desktop.split('@')[0]
        _display = _desktop.split('@')[1]

        if self.is_alive():
            if self.get_username() != _session_owner:
                self.logger('waiting for user ,,%s\'\' to interactively grant you access to his/her desktop session...' % _session_owner, loglevel=log.loglevel_NOTICE)
                self.logger('THIS MAY TAKE A WHILE!', loglevel=log.loglevel_NOTICE)

            _control = self.control_session
            try:
                self.terminal_session = _control.share_desktop(desktop=_desktop, share_mode=share_mode,
                                                               logger=self.logger, **self.terminal_params)
            except ValueError:
                # x2gostartagent output parsing will result in a ValueError. This one we will catch
                # here and change it into an X2goSessionException
                raise X2goSessionException('the session on desktop %s is seemingly dead' % _desktop)

            if self.has_terminal_session():
                self.session_name = self.terminal_session.session_info.name

                # shared desktop sessions get their startup command set by the control
                # session, run this pre-set command now...
                self.terminal_session.run_command(env=self.session_environment)

                self.virgin = False
                self.suspended = False
                self.running = True
                self.terminated = False
                self.faulty = False

                return self.running
            else:
                self.terminal_session = None
                return False

        else:
            self._X2goSession__disconnect()
            return False
    __share_desktop = share_desktop

    def suspend(self):
        """\
        Suspend this X2go session.

        @return: returns C{True} if suspending the session has been successful, C{False} otherwise
        @rtype: C{bool}

        """
        if self.is_alive():
            if self.has_terminal_session():
                if self.terminal_session.suspend():

                    self.running = False
                    self.suspended = True
                    self.terminated = False
                    self.faults = False
                    self.session_cleanup()
                    return True

            elif self.has_control_session() and self.session_name:
                if self.control_session.suspend(session_name=self.session_name):

                    self.running = False
                    self.suspended = True
                    self.terminated = False
                    self.faulty = False
                    self.session_cleanup()
                    return True

            else:
                raise X2goClientException('cannot suspend session')

        else:
            self._X2goSession__disconnect()

        return False
    __suspend = suspend

    def terminate(self):
        """\
        Terminate this X2go session.

        @return: returns C{True} if terminating the session has been successful, C{False} otherwise
        @rtype: C{bool}

        """
        if self.is_alive():
            if self.has_terminal_session():
                if self.terminal_session.terminate():

                    self.running = False
                    self.suspended = False
                    self.terminated = True
                    self.faulty = False
                    self.session_cleanup()
                    return True

            elif self.has_control_session() and self.session_name:
                if self.control_session.terminate(session_name=self.session_name):

                    self.running = False
                    self.suspended = False
                    self.terminated = True
                    self.faulty = False
                    self.session_cleanup()
                    return True
            else:
                raise X2goClientException('cannot terminate session')

        else:
            self._X2goSession__disconnect()

        return False
    __terminate = terminate

    def get_profile_name(self):
        """\
        Retrieve the profile name of this L{X2goSession} instance.

        @return: X2go client profile name of the session
        @rtype: C{str}

        """
        return self.profile_name
    __get_profile_name = get_profile_name

    def get_profile_id(self):
        """\
        Retrieve the profile ID of this L{X2goSession} instance.

        @return: the session profile's id
        @rtype: C{str}

        """
        return self.profile_id
    __get_profile_id = get_profile_id

    ###
    ### QUERYING INFORMATION
    ###

    def session_ok(self):
        """\
        Test if this C{X2goSession} is
        in a healthy state.

        @return: C{BTrue} if session is ok, C{False} otherwise
        @rtype: C{bool}

        """
        if self.has_terminal_session():
            return self.terminal_session.ok()
        return False
    __session_ok = session_ok

    def color_depth_from_session_name(self):
        """\
        Extract color depth from session name.

        @return: the session's color depth (as found in the session name)
        @rtype: C{str}

        """
        return int(self.get_session_name().split('_')[2][2:])
    __color_depth_from_session_name = color_depth_from_session_name

    def is_color_depth_ok(self):
        """\
        Check if this session will display properly with the local screen's color depth.

        @return: C{True} if the session will display on this client screen, False otherwise. If no terminal session is yet registered with this session, C{None} is returned.
        @rtype C{bool}

        """
        return utils.is_color_depth_ok(depth_session=self.color_depth_from_session_name(), depth_local=utils.local_color_depth())
        __is_color_depth_ok = is_color_depth_ok

    def is_connected(self):
        """\
        Test if the L{X2goSession}'s control session is connected to the 
        remote X2go server.

        @return: C{True} if session is connected, C{False} otherwise
        @rtype: C{bool}

        """
        self.connected = bool(self.control_session and self.control_session.is_connected())
        if not self.connected:
            self.running = None
            self.suspended = None
            self.terminated = None
            self.faulty = None
        return self.connected
    __is_connected = is_connected

    def is_running(self):
        """\
        Test if the L{X2goSession}'s terminal session is up and running.

        @return: C{True} if session is running, C{False} otherwise
        @rtype: C{bool}

        """
        if self.is_connected():
            self.running = self.control_session.is_running(self.get_session_name())
            if self.running:
                self.suspended = False
                self.terminated = False
                self.faulty = False
            if self.virgin and not self.running:
                self.running = None
        return self.running
    __is_running = is_running

    def is_suspended(self):
        """\
        Test if the L{X2goSession}'s terminal session is in suspended state.

        @return: C{True} if session is suspended, C{False} otherwise
        @rtype: C{bool}

        """
        if self.is_connected():
            self.suspended = self.control_session.is_suspended(self.get_session_name())
            if self.suspended:
                self.running = False
                self.terminated = False
                self.faulty = False
            if self.virgin and not self.suspended:
                self.suspended = None
        return self.suspended
    __is_suspended = is_suspended

    def has_terminated(self):
        """\
        Test if the L{X2goSession}'s terminal session has terminated.

        @return: C{True} if session has terminated, C{False} otherwise
        @rtype: C{bool}

        """
        if self.is_connected():
            self.terminated = not self.virgin and self.control_session.has_terminated(self.get_session_name())
            if self.terminated:
                self.running = False
                self.suspended = False
                self.faulty = False
            if self.virgin and not self.terminated:
                self.terminated = None
        return self.has_terminated
    __has_terminated = has_terminated

    def share_local_folder(self, folder_name):
        """\
        Share a local folder with this registered X2go session.

        @param folder_name: the full path to an existing folder on the local
            file system
        @type folder_name: C{str}

        @return: returns C{True} if the local folder has been successfully mounted within
            this X2go session
        @rtype: C{bool}

        """
        if self.has_terminal_session():
            if self.allow_share_local_folders:
                return self.terminal_session.share_local_folder(folder_name=folder_name)
            else:
                self.logger('local folder sharing is disabled for this session profile', loglevel=log.loglevel_WARN)
        else:
            raise X2goSessionException('this X2goSession object does not have any associated terminal')
    __share_local_folder = share_local_folder

    def is_locked(self):
        """\
        Query session if it is locked by some command being processed.

        @return: return C{True} is the session is locked, C{False} if not; returns None, if there is no
            control session yet.
        @rtype: C{bool}

        """
        if self.control_session is not None:
            return self.control_session.locked or self.locked
        return None

    def session_cleanup(self):
        """\
        Clean up X2go session.

        """
        if self.has_terminal_session():
            self.terminal_session.release_proxy()
        if self.has_terminal_session():
            self.terminal_session.__del__()
        self.terminal_session = None
