# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goSessionGuardian class - a guardian thread that controls X2go session threads
and their sub-threads (like reverse forwarding tunnels, Paramiko transport threads,
etc.).

"""
__NAME__ = 'x2goguardian-pylib'

# modules
import gevent
import threading
import copy

# Python X2go modules
from cleanup import x2go_cleanup
import x2go_exceptions
import log

_sigterm_received = False
def _sigterm_handle(s, f):
    global _sigterm_received
    _sigterm_received = True
    return 0

class X2goSessionGuardian(threading.Thread):
    """\
    L{X2goSessionGuardian} thread controls X2go session threads and their sub-threads (like 
    reverse forwarding tunnels, Paramiko transport threads, etc.). Its main function is
    to tidy up once a session gets interrupted (SIGTERM, SIGINT). 

    There is one L{X2goSessionGuardian} for each L{X2goClient} instance (thus: for normal
    setups there should be _one_ L{X2goClient} and _one_ L{X2goSessionGuardian} in use).

    """

    def __init__(self, client_instance, 
                 auto_update_listsessions_cache=False, 
                 auto_update_listdesktops_cache=False, 
                 auto_update_sessionregistry=False,
                 auto_register_sessions=False,
                 refresh_interval=5,
                 logger=None, loglevel=log.loglevel_DEFAULT):
        """\
        @param auto_update_listsessions_cache: let L{X2goSessionGuardian} refresh the session list cache for all L{X2goSession} objects
        @type auto_update_listsessions_cache: C{bool}
        @param auto_update_listdesktops_cache: let L{X2goSessionGuardian} refresh desktop lists in the session list cache for all L{X2goSession} objects
        @type auto_update_listdesktops_cache: C{bool}
        @param auto_update_sessionregistry: if set to C{True} the session status will be updated in regular intervals
        @type auto_update_sessionregistry: C{bool}
        @param auto_register_sessions: register new sessions automatically once they appear in the X2go session (e.g. 
            instantiated by another client that is connected to the same X2go server under same user ID)
        @type auto_register_sessions: C{bool}
        @param refresh_interval: refresh cache and session registry every <refresh_interval> seconds
        @type refresh_interval: C{int}
        @param logger: you can pass an L{X2goLogger} object to the L{X2goSessionGuardian} constructor
        @type logger: C{instance}
        @param loglevel: if no L{X2goLogger} object has been supplied a new one will be
            constructed with the given loglevel
        @type loglevel: C{int}

        """
        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        self.client_instance = client_instance
        self.auto_update_listsessions_cache = auto_update_listsessions_cache
        self.auto_update_listdesktops_cache = auto_update_listdesktops_cache
        self.auto_update_sessionregistry = auto_update_sessionregistry
        self.auto_register_sessions = auto_register_sessions
        self.refresh_interval = refresh_interval

        threading.Thread.__init__(self, target=self.guardian)
        self.daemon = True
        self.start()

    def guardian(self):
        """\
        The handler of this L{X2goSessionGuardian} thread.

        """
        global _sigterm_received

        seconds = 0
        self._keepalive = True
        while not _sigterm_received and self._keepalive:
            gevent.sleep(1)
            seconds += 1

            if seconds % self.refresh_interval == 0:

                self.logger('Entering X2go Guardian client management loop...', loglevel=log.loglevel_DEBUG)


                if self.auto_update_listsessions_cache:
                    self.client_instance.update_cache_all_profiles(update_sessions=self.auto_update_listsessions_cache, 
                                                                   update_desktops=self.auto_update_listdesktops_cache,
                                                                  )

                if self.auto_update_sessionregistry and not self.auto_register_sessions:
                    self.client_instance.update_sessionregistry_status_all_profiles()

                # session auto-registration will automatically trigger an update of the session registry status
                if self.auto_register_sessions:
                    self.client_instance.register_available_server_sessions_all_profiles()

        self.logger('X2go session guardian thread waking up after %s seconds' % seconds, loglevel=log.loglevel_DEBUG)

        for session_uuid in self.client_instance.session_registry.keys():
            session_summary = self.client_instance.get_session_summary(session_uuid)
            self.logger('calling session cleanup on profile %s for terminal session: %s' % (session_summary['profile_name'], session_summary['session_name']), loglevel=log.loglevel_DEBUG)
            x2go_cleanup(threads=session_summary['active_threads'])

    def stop_thread(self):
        """\
        Stop this L{X2goSessionGuardian} thread.

        """
        self._keepalive = False


