/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: SeaOfGates.java
 * Routing tool: Sea of Gates control
 * Written by: Steven M. Rubin
 *
 * Copyright (c) 2007, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.electric.tool.routing;

import com.sun.electric.database.EditingPreferences;
import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.topology.ArcInst;
import com.sun.electric.database.topology.Geometric;
import com.sun.electric.database.topology.NodeInst;
import com.sun.electric.database.variable.EditWindow_;
import com.sun.electric.database.variable.UserInterface;
import com.sun.electric.database.variable.Variable;
import com.sun.electric.technology.ArcProto;
import com.sun.electric.technology.Technology;
import com.sun.electric.technology.technologies.Generic;
import com.sun.electric.tool.Job;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngine;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngineFactory.SeaOfGatesEngineType;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesHandlers;
import com.sun.electric.util.ElapseTimer;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Class to control sea-of-gates routing.
 */
public class SeaOfGates
{
    /**
     * Method to run Sea-of-Gates routing on the current cell.
     * Presumes that it is inside of a Job.
     * @param justSubCells true to run the router on all sub-cells of the current cell
     * (but not on the current cell itself).
     */
    public static void seaOfGatesRoute(boolean justSubCells)
    {
    	if (justSubCells)
    	{
            // get cell
            UserInterface ui = Job.getUserInterface();
            Cell cell = ui.needCurrentCell();
            if (cell == null) return;
            Set<Cell> subCells =  new HashSet<Cell>();
            for(Iterator<NodeInst> it = cell.getNodes(); it.hasNext(); )
            {
            	NodeInst ni = it.next();
            	if (ni.isCellInstance()) subCells.add((Cell)ni.getProto());
            }
            for(Cell subCell : subCells)
            {
                List<ArcInst> selected = new ArrayList<ArcInst>();
                for (Iterator<ArcInst> it = subCell.getArcs(); it.hasNext(); )
                {
                	ArcInst ai = it.next();
                    if (ai.getProto() != Generic.tech().unrouted_arc) continue;
                    selected.add(ai);
                }
                if (!selected.isEmpty())
                {
                    // Run seaOfGatesRoute on subcell
                    SeaOfGatesHandlers.startInJob(subCell, selected, SeaOfGatesEngineType.defaultVersion);
                }
            }
    	} else
    	{
    		seaOfGatesRoute(SeaOfGatesEngineType.defaultVersion);
    	}
    }

    /**
     * Method to run Sea-of-Gates routing on the current cell, using a specified routing engine type.
     * Presumes that it is inside of a Job.
     */
    public static void seaOfGatesRoute(SeaOfGatesEngineType version)
    {
        // get cell and network information
        UserInterface ui = Job.getUserInterface();
        Cell cell = ui.needCurrentCell();
        if (cell == null) return;

        // get list of selected nets
        List<ArcInst> selected = getSelected();
        if (selected == null) return;

        // make sure there is something to route
        if (selected.isEmpty())
        {
            ui.showErrorMessage("There are no Unrouted Arcs in this cell", "Routing Error");
            return;
        }

        // Run seaOfGatesRoute on selected unrouted arcs in a separate job
        SeaOfGatesHandlers.startInJob(cell, selected, version);
    }
    
    /**
     * Method to run Sea-of-Gates routing on the current cell, using a specified routing engine.
     * Presumes that it is inside of a Job.
     */
    public static void seaOfGatesRoute(EditingPreferences ep, SeaOfGatesEngine router)
    {
        if (router == null) {
            throw new NullPointerException();
        }

        // get cell and network information
        UserInterface ui = Job.getUserInterface();
        Cell cell = ui.needCurrentCell();
        if (cell == null) return;

        // get list of selected nets
        List<ArcInst> selected = getSelected();

        // make sure there is something to route
        if (selected.isEmpty())
        {
            ui.showErrorMessage("There are no Unrouted Arcs in this cell", "Routing Error");
            return;
        }

        // Run seaOfGatesRoute on selected unrouted arcs
        Job job = Job.getRunningJob();
		router.routeIt(SeaOfGatesHandlers.getDefault(cell, job, ep), cell, selected);
    }

    private static List<ArcInst> getSelected() {
        EditWindow_ wnd = Job.getUserInterface().getCurrentEditWindow_();
        if (wnd == null) return null;
        Cell cell = wnd.getCell();
        if (cell == null) return null;
        
        // get list of selected nets
        List<ArcInst> selected = new ArrayList<ArcInst>();
        List<Geometric> highlighted = wnd.getHighlightedEObjs(false, true);
        for(Geometric h : highlighted)
        {
            ArcInst ai = (ArcInst)h;
            if (ai.getProto() != Generic.tech().unrouted_arc) continue;
            selected.add(ai);
        }
        if (selected.isEmpty())
        {
            for (Iterator<ArcInst> it = cell.getArcs(); it.hasNext(); )
            {
            	ArcInst ai = it.next();
                if (ai.getProto() != Generic.tech().unrouted_arc) continue;
                selected.add(ai);
            }
        }
        return selected;
    }

     /**
     * Class to hold preferences during Sea-of-Gates routing run.
     */
    public static class SeaOfGatesOptions implements Serializable
    {
        public boolean useParallelFromToRoutes;
        public boolean useParallelRoutes;
        public double maxArcWidth;
        public int complexityLimit;
        public boolean useGlobalRouter, reRunFailedRoutes;
        public int forcedNumberOfThreads;
        public ElapseTimer theTimer;

        public SeaOfGatesOptions()
        {
            useParallelFromToRoutes = true;
            useParallelRoutes = false;
            maxArcWidth = 10;
            complexityLimit = 200000;
            useGlobalRouter = false;
            reRunFailedRoutes = false;
            forcedNumberOfThreads = 0;
        }

        public void getOptionsFromPreferences()
        {
            useParallelFromToRoutes = Routing.isSeaOfGatesUseParallelFromToRoutes();
            useParallelRoutes = Routing.isSeaOfGatesUseParallelRoutes();
            maxArcWidth = Routing.getSeaOfGatesMaxWidth();
            complexityLimit = Routing.getSeaOfGatesComplexityLimit();
            useGlobalRouter = Routing.isSeaOfGatesUseGlobalRouting();
            reRunFailedRoutes = Routing.isSeaOfGatesRerunFailedRoutes();
            forcedNumberOfThreads = Routing.getSeaOfGatesForcedProcessorCount();
        }
    }

    /**
     * Class to define Sea-of-Gates routing parameters that apply only to a specific Cell.
     */
    public static class SeaOfGatesCellParameters implements Serializable
	{
    	private Cell cell;
    	private boolean steinerDone, favorHorVer, horEven;
    	private Map<ArcProto,String> gridSpacing;
    	private Set<ArcProto> preventedArcs, favoredArcs;

		/** key of Variable holding SOG parameters. */	private static final Variable.Key ROUTING_SOG_PARAMETERS_KEY = Variable.newKey("ATTR_ROUTING_SOG_PARAMETERS");

		public SeaOfGatesCellParameters(Cell cell)
		{
			this.cell = cell;
			this.steinerDone = false;
			this.favorHorVer = true;
			this.horEven = false;
			this.gridSpacing = new HashMap<ArcProto,String>();
			this.preventedArcs = new HashSet<ArcProto>();
			this.favoredArcs = new HashSet<ArcProto>();
			Variable var = cell.getVar(ROUTING_SOG_PARAMETERS_KEY);
			if (var != null)
			{
				String[] lines = (String[])var.getObject();
				for(int i=0; i<lines.length; i++)
				{
					String[] parts = lines[i].split(" ");
					if (parts.length <= 0) continue;
					if (parts[0].startsWith(";")) continue;
					if (parts[0].equalsIgnoreCase("SteinerTreesDone"))
					{
						steinerDone = true;
						continue;
					}
					if (parts[0].equalsIgnoreCase("IgnoreHorVer"))
					{
						favorHorVer = false;
						continue;
					}
					if (parts[0].equalsIgnoreCase("HorizontalEven"))
					{
						horEven = true;
						continue;
					}
					if (parts[0].equalsIgnoreCase("ArcGrid") && parts.length >= 3)
					{
						int colonPos = parts[1].indexOf(':');
						if (colonPos >= 0)
						{
							String techName = parts[1].substring(0, colonPos);
							String layerName = parts[1].substring(colonPos+1);
							Technology tech = Technology.findTechnology(techName);
							ArcProto ap = tech.findArcProto(layerName);
							gridSpacing.put(ap, parts[2]);
						}
						continue;
					}
					if (parts[0].equalsIgnoreCase("ArcAvoid") && parts.length >= 2)
					{
						int colonPos = parts[1].indexOf(':');
						if (colonPos >= 0)
						{
							String techName = parts[1].substring(0, colonPos);
							String layerName = parts[1].substring(colonPos+1);
							Technology tech = Technology.findTechnology(techName);
							ArcProto ap = tech.findArcProto(layerName);
							preventedArcs.add(ap);
						}
						continue;
					}
					if (parts[0].equalsIgnoreCase("ArcFavor") && parts.length >= 2)
					{
						int colonPos = parts[1].indexOf(':');
						if (colonPos >= 0)
						{
							String techName = parts[1].substring(0, colonPos);
							String layerName = parts[1].substring(colonPos+1);
							Technology tech = Technology.findTechnology(techName);
							ArcProto ap = tech.findArcProto(layerName);
							favoredArcs.add(ap);
						}
						continue;
					}
				}
			}
		}

		public void saveParameters(EditingPreferences ep)
		{
			List<String> strings = new ArrayList<String>();

			// header
			strings.add("; Parameters for Cell " + cell.describe(false));

			// steiner trees
			if (steinerDone) strings.add("SteinerTreesDone");

			// miscellaneous
			if (!favorHorVer) strings.add("IgnoreHorVer");
			if (horEven) strings.add("HorizontalEven");

			// ArcProto information
			for(ArcProto ap : gridSpacing.keySet())
			{
				String grid = gridSpacing.get(ap);
				strings.add("ArcGrid " + ap.getTechnology().getTechName() + ":" + ap.getName() + " " + grid);
			}
			for(ArcProto ap : preventedArcs)
			{
				strings.add("ArcAvoid " + ap.getTechnology().getTechName() + ":" + ap.getName());
			}
			for(ArcProto ap : favoredArcs)
			{
				strings.add("ArcFavor " + ap.getTechnology().getTechName() + ":" + ap.getName());
			}

			String[] paramArray = new String[strings.size()];
			for(int i=0; i<strings.size(); i++) paramArray[i] = strings.get(i);
			cell.newVar(ROUTING_SOG_PARAMETERS_KEY, paramArray, ep);
		}

		public void setSteinerDone(boolean sd) { steinerDone = sd; }

		public boolean isSteinerDone() { return steinerDone; }

		public void setFavorHorVer(boolean f) { favorHorVer = f; }

		public boolean isFavorHorVer() { return favorHorVer; }

		public void setHorizontalEven(boolean he) { horEven = he; }

		public boolean isHorizontalEven() { return horEven; }

		public void setPrevented(ArcProto ap, boolean prevent)
		{
			if (prevent) preventedArcs.add(ap); else
				preventedArcs.remove(ap);
		}

		public boolean isPrevented(ArcProto ap) { return preventedArcs.contains(ap); }

		public void setFavored(ArcProto ap, boolean prevent)
		{
			if (prevent) favoredArcs.add(ap); else
				favoredArcs.remove(ap);
		}

		public boolean isFavored(ArcProto ap) { return favoredArcs.contains(ap); }

		public String getGrid(ArcProto ap)
		{
			String v = gridSpacing.get(ap);
			return v;
		}

		public void setGrid(ArcProto ap, String grid)
		{
			if (grid == null) gridSpacing.remove(ap); else
				gridSpacing.put(ap, grid);
		}
	}

}
