/* -*- c++ -*- */
/*
 * Copyright 2002,2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <gr_nco.h>
#include <gr_fxpt_nco.h>

#define ITERATIONS	10000000

static double
timeval_to_double (const struct timeval *tv)
{
  return (double) tv->tv_sec + (double) tv->tv_usec * 1e-6;
}


static void
benchmark (void test (float *x, float *y), const char *implementation_name)
{
  struct rusage	rusage_start;
  struct rusage	rusage_stop;
  float	x, y;

  // get starting CPU usage

  if (getrusage (RUSAGE_SELF, &rusage_start) < 0){
    perror ("getrusage");
    exit (1);
  }

  // do the actual work

  test (&x, &y);
  
  // get ending CPU usage

  if (getrusage (RUSAGE_SELF, &rusage_stop) < 0){
    perror ("getrusage");
    exit (1);
  }

  // compute results

  double user =
    timeval_to_double (&rusage_stop.ru_utime)
    - timeval_to_double (&rusage_start.ru_utime);

  double sys =
    timeval_to_double (&rusage_stop.ru_stime)
    - timeval_to_double (&rusage_start.ru_stime);

  double total = user + sys;

  printf ("%16s:  cpu: %6.3f  steps/sec: %10.3e\n",
	  implementation_name, total, ITERATIONS / total);
}

// ----------------------------------------------------------------

void native_sincos (float *x, float *y)
{
  gr_nco<float,float>	nco;
  
  nco.set_freq (2 * M_PI / 5003.123);
  
  for (int i = 0; i < ITERATIONS; i++){
    nco.sincos (x, y);
    nco.step ();
  }
}

void fxpt_sincos (float *x, float *y)
{
  gr_fxpt_nco	nco;
  
  nco.set_freq (2 * M_PI / 5003.123);
  
  for (int i = 0; i < ITERATIONS; i++){
    nco.sincos (x, y);
    nco.step ();
  }
}

// ----------------------------------------------------------------

void native_sin (float *x, float *y)
{
  gr_nco<float,float>	nco;
  
  nco.set_freq (2 * M_PI / 5003.123);
  
  for (int i = 0; i < ITERATIONS; i++){
    *x = nco.sin ();
    nco.step ();
  }
}

void fxpt_sin (float *x, float *y)
{
  gr_fxpt_nco	nco;
  
  nco.set_freq (2 * M_PI / 5003.123);
  
  for (int i = 0; i < ITERATIONS; i++){
    *x = nco.sin ();
    nco.step ();
  }
}

// ----------------------------------------------------------------

void nop_fct (float *x, float *y)
{
}

void nop_loop (float *x, float *y)
{
  for (int i = 0; i < ITERATIONS; i++){
    nop_fct (x, y);
  }
}

int 
main (int argc, char **argv)
{
  benchmark (nop_loop, "nop loop");
  benchmark (native_sin, "native sine");
  benchmark (fxpt_sin, "fxpt sine");
  benchmark (native_sincos, "native sin/cos");
  benchmark (fxpt_sincos, "fxpt sin/cos");
}
