# manager shell
#  Copyright (C) 1992-2000 Michigan State University
#
#  The CAPA system is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 2 of the
#  License, or (at your option) any later version.
#
#  The CAPA system is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#
#  You should have received a copy of the GNU General Public
#  License along with the CAPA system; see the file COPYING.  If not,
#  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
#  Boston, MA 02111-1307, USA.
#
#  As a special exception, you have permission to link this program
#  with the TtH/TtM library and distribute executables, as long as you
#  follow the requirements of the GNU GPL in regard to all of the
#  software in the executable aside from TtH/TtM.

###########################################################
# manager.tcl - 
# Created Guy Albertelli II 1998
###########################################################
set gTclVer 1.0

###########################################################
# createControlWindow
###########################################################
# Creates the menu window 
###########################################################
# Arguments: none
# Returns: nothing
# Globals: gWindowMenu - set to the name of Menu for the windows
#                        menu
###########################################################
proc createControlWindow {} {
    global gWindowMenu gCapaConfig
    
    after 500 { dateUpdate }
    after 1000 { cleanWindowList }

    set gCapaConfig(Printer_selected) "0"
    set gCapaConfig(lprOneSided_command) "lpr "
    set gCapaConfig(lprTwoSided_command) ""
    set gCapaConfig(printer_option) ""
    
    wm withdraw .

    set menuFrame [menu .main -tearoff 0 -type tearoff -font 8x13bold \
	  -disabledforeground grey85 ]
    
    wm title $menuFrame "Manager"

    $menuFrame post 0 0

    wm geometry $menuFrame "+0+20"
    $menuFrame add command -label "Manager" -foreground grey85 -background \
	    black -state disabled 
    $menuFrame add command -label "Info..." -command { createInfoWindow }
    $menuFrame add cascade -label "Actions" -menu $menuFrame.actions
    $menuFrame add cascade -label "Windows" -menu $menuFrame.windows
    $menuFrame add cascade -label "Print" -menu $menuFrame.print
    $menuFrame add command -label "Remap..." -command { createRemapWindow }
    $menuFrame add command -label "Quit" -command { quit }

    set action  [menu $menuFrame.actions -tearoff 1 -font 8x13bold]
    set windows [menu $menuFrame.windows -tearoff 1 -font 8x13bold]
    set print   [menu $menuFrame.print -tearoff 1  -font 8x13bold]
    set score   [menu $menuFrame.scoring -tearoff 1 -font 8x13bold]
    set gWindowMenu $windows
    
    $action add command -label "Manage Classl" -command \
 	    { specifyClass "Pick classl file" "Classl" "classl" createClasslEditor }
#    $action add command -label "Edit capa.config" -command \
	    { specifyClass "Pick a capa.config file" "CAPA configuration" \
		  "capa.config" editCapaConfig }
    $action add cascade -label "Scoring" -menu $menuFrame.scoring
    $action add command -label "Generate Stats" -command \
	    { specifyClass "Select the capa.config in the class directory" \
		  "CAPA configuration" "capa.config" runCapaTools }
    $action add command -label "Send Class Email" -command \
	{ specifyClass "Select the capa.config in the class directory" \
	      "CAPA configuration" "capa.config" runGroupEmail }
    $action add command -label "Randomize Seating File" \
	-command { RSstart [incr gUniqueNumber] }

    $score add command -label "Run Scorer" -command \
	{ specifyClass "Pick set.qz file" "QZ file" "*.qz" runScorer }
    $score add command -label "ReScore a scorer.output" -command \
	{ specifyClass "Pick scorer.output file" "scorer.output file" \
	      "scorer.output.*" reScore }

    $print add command -label "Print Assignment" -command \
	{ specifyClass "Select the capa.config in the class directory" \
	      "CAPA configuration" "capa.config" printAssignment  }

    bind $menuFrame <Destroy> { quit 0 }
}

###########################################################
# createInfoWindow
###########################################################
# creates the Information window
###########################################################
# Arguments: None
# Returns: Nothing
# Globals: gDate - the variable containg the current date 
#          gWindowMenu - used to register the new window in the
#                        windows menu
#          gVer - Stores the current version of Grader (set in 
#                 C init code
###########################################################
proc createInfoWindow {} {
    global gDate gWindowMenu gVer gTclVer gCmd gCompileDate

    if { [winfo exists .about] } {
	capaRaise .about
	return 
    }

    set about [toplevel .about]

    $gWindowMenu add command -label "About" -command "capaRaise $about"

    wm title $about "About" 
    
    label $about.l1 -font 12x24 -text "Manager $gVer" -pady 20
    label $about.l4 -font 8x13 -text "Manager.tcl Version $gTclVer" -pady 20
    label $about.l6  -font 8x13 -text "$gCompileDate" 
    message $about.l2 -font 8x13 -text "Code by: Y. Tsai, G. Albertelli II Copyright Michigan State University Board of Trustees, 1992-2000, CAPA is released under to GNU GPL v2, and comes WITHOUT ANY WARRENTY, see COPYING for details" \
	-pady 20 -aspect 300
    label $about.l3 -font 8x13 -textvariable gDate 
    label $about.l5  -font 8x13 -textvariable gCmd

    button $about.close -text "Close" -command "destroy $about
                                                removeWindowEntry About"
    
    pack $about.l1 $about.l4 $about.l6 $about.l2 $about.l3 $about.l5 \
	    $about.close -side top 

    Centre_Dialog $about default
}

###########################################################
# quit
###########################################################
# called when the quit option is selected on the menu, unmaps
# all keys.
###########################################################
# Arguments: None
# Returns: Nothing
# Globals: None
###########################################################
proc quit { { ask 1} } {
    if { $ask && [makeSure "Are you sure you wish to quit?"] == "Cancel" } {
	return 
    }
    
    unmapAllKeys
    exit
}

###########################################################
# specifyClass
###########################################################
# runs tk_getOpenFile and creates the class window if a directory is chosen
###########################################################
# Arguments: None
# Returns: Nothing
# Globals: gClassDir - remembers the directory the class is in
###########################################################
proc specifyClass { title typename type followupCommand } {
#    set var [tk_getOpenFile]
    set var [tk_getOpenFile -title $title -filetypes \
		 [list  [list  $typename $type ] { {All Files} {"*"} } ] ]
    
    if { $var == "" } {	return }
    
    $followupCommand $var
}

###########################################################
# dateUpdate
###########################################################
# updates the gDate var, and register to run again in .8 seconds
###########################################################
# Arguments: None
# Returns: Nothing
# Globals: gDate - gets set to the current time and date
###########################################################
proc dateUpdate { } {
    global gDate
    set gDate [clock format [clock seconds]]
    after 800 dateUpdate
}

###########################################################
###########################################################
###########################################################
###########################################################
proc askToSave { msg cmd } {
    global gChanged gPrompt
    
    set dialog [toplevel .askToSavePrompt -borderwidth 10]
    wm title $dialog "Do you wish to Save"
    wm geo $dialog "+200+200"
    message $dialog.msg -text $msg -aspect 800
    
    set gPrompt(result) ""
    set buttonFrame [frame $dialog.buttons -bd 10]
    pack $dialog.msg $buttonFrame -side top -fill x
    
    bind $dialog <Destroy> { 
	set gPrompt(result) Cancel
	set gPrompt(yes) 0
    }

    button $buttonFrame.yes -text Yes -underline 0 -command {
	set gPrompt(yes) 1
    }
    button $buttonFrame.no -text No -underline 0 -command {
	set gPrompt(yes) 0
    } 
    button $buttonFrame.cancel -text Cancel  -underline 0 -command { 
	set gPrompt(yes) 0 
	set gPrompt(result) Cancel
    }
    pack $buttonFrame.yes $buttonFrame.no $buttonFrame.cancel -side left
    bind $dialog <Alt-Key> break

    Centre_Dialog $dialog default
    update

    focus $dialog
    capaRaise $dialog
    capaGrab $dialog
    vwait gPrompt(yes)
    capaGrab release $dialog
    bind $dialog <Destroy> ""
    destroy $dialog
    if {$gPrompt(yes)} {
	eval $cmd
    } else {
	return $gPrompt(result)
    }
}    

###########################################################
# printAssignment
###########################################################
###########################################################
###########################################################
proc printAssignment { classconfig } {
    global gCT gFile gUniqueNumber gCapaConfig

    set num [incr gUniqueNumber]
    set gFile($num) [file dirname $classconfig]
    getOneStudent "" $gFile($num) s_id s_name    
    if { $s_id == "" } { return }
    set s_id [string toupper $s_id]
    if { "" == [set setlist [getSetRange "" $gFile($num)]] } { return }
    set cmdnum [incr gUniqueNumber]
    if { "Yes" != [makeSure "Are you sure you want to print set(s) [linsert $setlist 1 to] for student: $s_name"] } { 
	return
    }
    displayStatus "Running qzparse" message $num
    parseCapaConfig $num $gFile($num)
    set command "$gCapaConfig($num.qzparse_command) -c $gFile($num) \
                 -set [lindex $setlist 0]:[lindex $setlist 1] -stu $s_id"
    eval "exec $command"
    set tex_file [file join $gFile($num) TeX $s_id.tex]
    set command "$gCapaConfig($num.latex_command) $tex_file"
    removeStatus $num
    if { "Yes" != [makeSure "Planning on running LaTeX, Continue?"] } { return }
    displayStatus "Running LaTeX" message $num
    set directory [pwd]
    cd [file join $gFile($num) TeX]
    eval "exec $command"
    cd $directory
    set dvi_file [file join $gFile($num) TeX $s_id.dvi]
    set ps_file [file join $gFile($num) TeX $s_id.ps]
    set command "$gCapaConfig($num.dvips_command) $dvi_file -o $ps_file >& /dev/null"
    removeStatus $num
    if { "Yes" != [makeSure "Planning on running dvips, Continue?"] } { return }
    displayStatus "Running dvips" message $num
    eval "exec $command"
    removeStatus $num
    if { "Cancel" == [set lprcmd [getLprCommand $ps_file $num]] } { return }
    if { [catch { eval "exec $lprcmd" } errors ] } {
	displayError "An error occurred while printing: $errors"
    }
    foreach name [array names gCapaConfig "$num.*"] {
	unset gCapaConfig($name)
    }
}

source common.tcl
source utils.tcl
source classl.tcl
source scorer.tcl
source scorer.anon.tcl
source scorer.errors.tcl
source capastats.tcl
source seating.tcl
source groupemail.tcl

set gUniqueNumber 0
set gMaxSet 99
set gMaxTries 99
option add *font 8x13bold
option add selectbackground #b4b4ff
createControlWindow
