/* cfengine for GNU
 
        Copyright (C) 1995
        Free Software Foundation, Inc.
 
   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/
 

#include "cf.defs.h"
#include "cf.extern.h"


/*********************************************************************/
/* Tidy toolkit                                                      */
/*********************************************************************/

RecursiveHomeTidy(name,level,tp)

char *name;
int level;
struct Tidy *tp;

{ struct stat statbuf;
  DIR *dirh;
  struct dirent *dirp;
  char pcwd[bufsize];
  time_t ticks;

if (strlen(name) == 0)
   {
   name = "/";
   }

Debug2("HomeTidy: Opening %s\n",name);

if ((dirh = opendir(name)) == NULL)
   {
   printf("cfengine: RecursiveTidy(): Can't open directory %s\n",name);
   return;
   }

if (level == 2)
   {
   strcpy(VLOGFILE,name);
   strcat(VLOGFILE,"/.cfengine.rm");

   if ((VLOGFP = fopen(VLOGFILE,"w")) == NULL)         /* log deleted files for each user */
      {
      printf("cfengine: Couldn't open a file %s\n",VLOGFILE);
      VLOGFP = stderr;
      }
   else
      {
      ticks = time((time_t *)NULL);
      fprintf(VLOGFP,"This file is generated by cfengine %s\n",VERSION);
      fprintf(VLOGFP,"It contains a log of the files which have been tidied.\n");
      fprintf(VLOGFP,"The time of writing is %s\n",ctime(&ticks));
      fprintf(VLOGFP,"If you have any questions about this, send them to %s.\n",VSYSADM);
      fprintf(VLOGFP,"-(Start transcript)---------------\n");
      }
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strcmp(".",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0)
      {
      continue;
      }

   if (IgnoreFile(name,dirp->d_name))
      {
      continue;
      }

   strcpy(pcwd,name);                                 /* Assemble pathname */
   AddSlash(pcwd);

   if (BufferOverflow(pcwd,dirp->d_name))
      {
      printf(" culprit: RecursiveHomeTidy()\n");
      return;
      }

   strcat(pcwd,dirp->d_name);

   if (TRAVLINKS)
      {
      if (stat(pcwd,&statbuf) == -1)
         {
         Verbose("cfengine: RecursiveHomeTidy(): Can't stat %s\n",pcwd);
         continue;
         }
      }
   else
      {
      if (lstat(pcwd,&statbuf) == -1)
         {
         if (DEBUG || D2 || VERBOSE)
            {
            printf("cfengine: RecursiveHomeTidy(): Can't stat %s\n",pcwd);
            if (readlink(pcwd,VBUFF,bufsize) != -1)
               {
               printf("          File is link to -> %s\n",VBUFF);
               }
            }
         continue;
         }
      }


   if (S_ISDIR(statbuf.st_mode))
      {
      if (IsMountedFileSystem(&statbuf,pcwd))
         {
         continue;
         }
      else
         {
         RecursiveHomeTidy(pcwd,level+1);
         }
      }
   else
      {
      TidyHomeFile(pcwd,dirp->d_name,&statbuf);
      }
   }

if (level == 2)
   {
   fclose(VLOGFP);
   chmod(VLOGFILE,DEFAULTMODE);
   }

closedir(dirh);
}


/*********************************************************************/

TidyHomeFile(path,name,statbuf)

char *path;
char *name;
struct stat *statbuf;

  /* Tidy a file if it's past its sell-by date in kB, and if
     it is greater than the specified size. Don't need an OR,
     since size age can just be set to zero.                 */

{ struct Tidy *tp;
  struct TidyPattern *tlp;
  short savetravlinks, savekilloldlinks;

for (tp = VTIDY; tp != NULL; tp=tp->next)
   {
   if (tp->tidylist == NULL)  /* used to eliminate non-home searches */
      {
      continue;
      }

   for (tlp = tp->tidylist; tlp != NULL; tlp=tlp->next)
      {
      if (IsExcluded(tlp->classes))
	 {
	 continue;
	 }

      savetravlinks = TRAVLINKS;
      savekilloldlinks = KILLOLDLINKS;

      if (tlp->travlinks == 'T')
         {
         TRAVLINKS = true;
         }
      else if (tlp->travlinks == 'F')
         {
         TRAVLINKS = false;
	 }
      else if (tlp->travlinks == 'K')
         {
         KILLOLDLINKS = true;
         }

      TRAVLINKS = savetravlinks;
      
      if (WildMatch(tlp->pattern,name) && CheckHomeSubDir(path,tp->path))
         {
         DoTidyFile(path,name,tlp,statbuf,CF_USELOGFILE);
	 }
      }
   }

TRAVLINKS = savetravlinks;
KILLOLDLINKS = savekilloldlinks;
}


/*********************************************************************/

RecursiveTidySpecialArea(name,tp,maxrecurse)

char *name;
struct Tidy *tp;
int maxrecurse;

{ struct stat statbuf;
  DIR *dirh;
  struct dirent *dirp;
  char pcwd[bufsize];
  int is_dir;

if (maxrecurse == -1)
   {
   Debug2("MAXRECURSE ran out, quitting at %s\n",name);
   return;
   }

if (IgnoreFile(name,""))
   {
   Debug2("cfengine: Ignoring directory %s\n",name);
   return;
   }

if (strlen(name) == 0)     /* Check for root dir */
   {
   name = "/";
   }

if ((dirh = opendir(name)) == NULL)
   {
   printf("cfengine: RecursiveTidySpecialArea(): Can't open directory [%s]\n",name);
   return;
   }

Verbose("Tidy: opening dir %s\n\n",name);

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strcmp(".",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0)
      {
      continue;
      }

   if (IgnoreFile(name,dirp->d_name))
      {
      continue;
      }

   strcpy(pcwd,name);                                   /* Assemble pathname */
   AddSlash(pcwd);

   if (BufferOverflow(pcwd,dirp->d_name))
      {
      printf(" culprit: RecursiveTidySpecialArea\n");
      return;
      }

   strcat(pcwd,dirp->d_name);

   if (stat(pcwd,&statbuf) == -1)
      {
      Debug("cfengine: RecursiveTidy(): Can't stat %s\n",pcwd);
      continue;
      }

   if (S_ISDIR(statbuf.st_mode))
      {
      is_dir =  true;
      }
   else
      {
      is_dir = false;
      }
   
   if (!TRAVLINKS)
      {
      if (lstat(pcwd,&statbuf) == -1)
         {
         if (DEBUG || D2 || VERBOSE)
            {
            printf("cfengine: RecursiveTidy(): Can't stat %s\n",pcwd);
            if (readlink(pcwd,VBUFF,bufsize) != -1)
               {
               printf("          File is link to -> %s\n",VBUFF);
               }
            }
         continue;
         }
      }

   if (S_ISDIR(statbuf.st_mode))
      {
      if (IsMountedFileSystem(&statbuf,pcwd))
         {
         continue;
         }
      else
         {
         RecursiveTidySpecialArea(pcwd,tp,maxrecurse-1);
         }
      }
   else
      { int level = tp->recurse - maxrecurse;

      TidyParticularFile(pcwd,dirp->d_name,tp,&statbuf,is_dir,level);
      }
   }

closedir(dirh);
}

/*********************************************************************/

TidyParticularFile(path,name,tp,statbuf,dirlink,level)

char *path, *name;
struct Tidy *tp;
struct stat *statbuf;
int level,dirlink;

{ struct TidyPattern *tlp;

Debug2("TidyParticularFile(%s,%s)\n",path,name);

if (tp->tidylist == NULL)
   {
   return;
   }

for (tlp = tp->tidylist; tlp != NULL; tlp=tlp->next)
   {
   if (dirlink && tlp->dirlinks == 'k')  /* Keep link directories */
      {
      continue;
      }
   
   if (level > tlp->recurse && tlp->recurse != INFINITERECURSE)
      {
      Debug2("[PATTERN %s RECURSE ENDED at %d(%d) BEFORE MAXVAL %d]\n",tlp->pattern,
		level,tlp->recurse,tp->recurse);
      continue;
      }
   
   if (IsExcluded(tlp->classes))
      {
      continue;
      }
   
   if (! WildMatch(tlp->pattern,name))
      {
      continue;
      }
   
   Debug2("Matched %s to %s in %s\n",name,tlp->pattern,path);
   DoTidyFile(path,name,tlp,statbuf,CF_NOLOGFILE);
   }
}

/*********************************************************************/

TidyPathExists(path)

char *path;

{ struct Tidy *tp;

Debug1("TidyPathExists(%s)\n",path);

for (tp = VTIDY; tp != NULL; tp=tp->next)
   {
   if (strcmp(tp->path,path) == 0)
      {
      return true;
      }
   }

return false;
}

/*********************************************************************/

InstallTidyPath(path,wild,rec,age,travlinks,tidysize,type,ldirs,classes)

char *wild, *path;
short age,tidysize;
int rec, travlinks;
char type, ldirs, *classes;

{ struct Tidy *ptr;
  char *pp, *sp;
  int no_of_links = 0;

if (!IsInstallable(classes))
   {
   Debug1("Not installing tidy path, no match\n");
   return;
   }

Debug1("InstallTidyItem(%s,%s,%d,%d,%c,%c)\n",path,wild,rec,age,travlinks,type,ldirs);

VBUFF[0]='\0';                                /* Expand any variables */
ExpandVarstring(path,VBUFF,"");
DeleteSlash(VBUFF);

if ((ptr = (struct Tidy *)malloc(sizeof(struct Tidy))) == NULL)
   {
   FatalError("Memory Allocation failed for InstallTidyItem() #1");
   }

if ((pp = malloc(strlen(VBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallTidyItem() #3");
   }

if (VTIDYTOP == NULL)                 /* First element in the list */
   {
   VTIDY = ptr;
   }
else
   {
   VTIDYTOP->next = ptr;
   }

if (rec != INFINITERECURSE && strncmp("home/",pp,5) == 0)     /* Is this a subdir of home wildcard? */
   {
   for (sp = pp; *sp != '\0'; sp++)                     /* Need to make recursion relative to start */
      {                                                    /* of the search, not relative to home */
      if (*sp == '/')
         {
         no_of_links++;
         }
      }
   }

strcpy (pp,VBUFF);

ptr->tidylist = NULL;
ptr->path = pp;
ptr->recurse = rec + no_of_links;
ptr->next = NULL;
VTIDYTOP = ptr;

AddTidyItem(path,wild,rec,age,travlinks,tidysize,type,ldirs,classes);

InitializeAction();
}

/*********************************************************************/

AddTidyItem(path,wild,rec,age,travlinks,tidysize,type,ldirs,classes)

char *wild, *path;
short age,tidysize;
int rec, travlinks;
char type, ldirs,*classes;

{ char varbuff[bufsize];
  struct Tidy *ptr;

Debug1("AddTidyItem(%s)\n",path);

if ( ! IsInstallable(CLASSBUFF))
   {
   Debug1("Not installing TidyItem no match\n");
   return;
   }

for (ptr = VTIDY; ptr != NULL; ptr=ptr->next)
   {
   varbuff[0] = '\0';
   ExpandVarstring(path,varbuff,"");

   if (strcmp(ptr->path,varbuff) == 0)
      {
      PrependTidy(&(ptr->tidylist),wild,rec,age,travlinks,tidysize,type,ldirs,classes);
      
      /* Must have the maximum recursion level here */
      if (rec == INFINITERECURSE || (ptr->recurse < rec && ptr->recurse != INFINITERECURSE))
	 {
         ptr->recurse = rec;
	 }
      return;
      }
   }
}

/*********************************************************************/
/* Level 2                                                           */
/*********************************************************************/

DoTidyFile(path,name,tlp,statbuf,logging_this)

char *path, *name;
struct TidyPattern *tlp;
struct stat *statbuf;
short logging_this;

{ time_t nowticks, fileticks;
  int size_match = false, age_match = false;

Debug2("DoTidyFile(%s,%s)\n",path,name);

nowticks = time((time_t *)NULL);             /* cmp time in days */

switch (tlp->searchtype)
   {
   case 'a': fileticks = statbuf->st_atime;
             break;
   case 'm': fileticks = statbuf->st_mtime;
	     break;
   case 'c': fileticks = statbuf->st_ctime;
	     break;
   default:  printf("cfengine: Internal error in DoTidyFile()\n");
             break;
   }

if (nowticks-fileticks < 0)                  /* shouldn't happen */
   {
   printf("cfengine: ALERT! access time for %s is in the future. Check system clock!\n",path);
   return;
   }

if (tlp->size == CF_EMPTYFILE)
   {
   if (statbuf->st_size == 0)
      {
      size_match = true;
      }
   else
      {
      size_match = false;
      }
   }
else
   {
   size_match = (tlp->size * 1024 <= statbuf->st_size);
   }

age_match = tlp->age*TICKSPERDAY <= (nowticks-fileticks);

if (age_match && size_match)
   {
   if (logging_this)
      {
      fprintf(VLOGFP,"cf: rm %s\n",path);
      }

   if (! DONTDO)
      {
      if (unlink(path) == -1)
	 {
         perror("unlink");
	 }

      Verbose("cfengine: Deleting %s\n - size=%d bytes, %c-age=%d days\n",
		path,statbuf->st_size,tlp->searchtype,(nowticks-fileticks)/TICKSPERDAY);
      }
   else
      {
      printf("cfengine: want to remove %s\n",path);
      }
   }
else
   {
   Debug2("(No age match)\n");
   }
}

/*********************************************************************/

PrependTidy(list,wild,rec,age,travlinks,tidysize,type,ldirs,classes)

struct TidyPattern **list;
char *wild;
short age,tidysize;
int rec, travlinks;
char type, ldirs,*classes;

{ struct TidyPattern *tp;
  char *spe,*sp;

if ((tp = (struct TidyPattern *)malloc(sizeof(struct TidyPattern))) == NULL)
   {
   perror("Can't allocate memory in PrependTidy()");
   FatalError("");
   }

if ((sp = malloc(strlen(wild)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for PrependTidy() #2");
   }

if ((classes!= NULL) && (spe = malloc(strlen(classes)+2)) == NULL)
   {
   perror("Can't allocate memory in PrependItem()");
   FatalError("");
   }

strcpy(sp,wild);

tp->size = tidysize;
tp->recurse = rec;
tp->age = age;
tp->searchtype = type;
tp->travlinks = travlinks;
tp->pattern = sp;
tp->next = *list;
tp->dirlinks = ldirs;
*list = tp;

if (classes != NULL)
   {
   strcpy(spe,classes);
   tp->classes = spe;
   }
else
   {
   tp->classes = NULL;
   }
}

/*********************************************************************/

DeleteTidyList(list)

struct TidyPattern *list;

{
if (list != NULL)
   {
   DeleteTidyList(list->next);
   list->next = NULL;

   if (list->classes != NULL)
      {
      free (list->classes);
      }

   free((char *)list);
   }
}
