;;; winnt.el --- Lisp routines for Windows NT.

;; Copyright (C) 1994 Free Software Foundation, Inc.

;; Author: Geoff Voelker (voelker@cs.washington.edu)

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; (August 12, 1993)
;; Created.

;; (November 21, 1994)
;; [C-M-backspace] defined.
;; mode-line-format defined to show buffer file type.
;; audio bell initialized.

;;; Code:

;; Map delete and backspace
(define-key function-key-map [backspace] "\177")
(define-key function-key-map [delete] "\C-d")
(define-key function-key-map [M-backspace] [?\M-\177])
(define-key function-key-map [C-M-backspace] [\C-\M-delete])

;; Show file type (text or binary) on modeline
(setq-default mode-line-format
  (list (purecopy "")
   'mode-line-modified
   'mode-line-buffer-identification
   (purecopy "   ")
   'global-mode-string
   (purecopy "   %[(")
   (purecopy "%t:")
   'mode-name 'mode-line-process 'minor-mode-alist
   (purecopy "%n")
   (purecopy ")%]--")
   (purecopy '(line-number-mode "L%l--"))
   (purecopy '(column-number-mode "C%c--"))
   (purecopy '(-3 . "%p"))
   (purecopy "-%-")))

;; Ignore case on file-name completion
(setq completion-ignore-case t)

;; The cmd.exe shell uses the "/c" switch instead of the "-c" switch
;; for executing its command line argument (from simple.el).
;; However, Emacs now includes cmdproxy which is used as the shell by
;; default; cmdproxy accepts "-c" and interprets as Unix shells do.
;(setq shell-command-switch "/c")

;; For appending suffixes to directories and files in shell completions.
(add-hook 'shell-mode-hook 
	  '(lambda () (setq comint-completion-addsuffix '("\\" . " "))))

;; Use ";" instead of ":" as a path separator (from files.el).
(setq path-separator ";")

;; Set the null device (for compile.el).
(setq grep-null-device "NUL")

;; Set the grep regexp to match entries with drive letters.
(setq grep-regexp-alist
  '(("^\\(\\([a-zA-Z]:\\)?[^:( \t\n]+\\)[:( \t]+\\([0-9]+\\)[:) \t]" 1 3)))

;; Taken from dos-fn.el ... don't want all that's in the file, maybe
;; separate it out someday.

(defvar auto-detect-buffer-file-type t
  "*Detect file type (text or binary) from the file contents.
If non-nil, insert-file-contents decides whether to convert DOS line
endings based on the content being inserted.  Text conversion is only
performed if all line endings (in the first 16KB if reverting a file)
are in DOS format.")

(defvar in-insert-file-contents nil
  "Bound to t inside insert-file-contents.")

(defvar file-name-buffer-file-type-alist
  '(
    ("[:/].*config.sys$" . nil)		; config.sys text
    ("\\.elc$" . t)			; emacs stuff
    ("\\.\\(obj\\|exe\\|com\\|lib\\|sym\\|sys\\|chk\\|out\\|bin\\|ico\\|pif\\|dos\\)$" . t)
					; MS-Dos stuff
    ("\\.\\(dll\\|drv\\|cpl\\|scr\\vbx\\|386\\|vxd\\|fon\\|fnt\\|fot\\|ttf\\|grp\\)$" . t)
					; Windows stuff
    ("\\.\\(hlp\\|bmp\\|wav\\|avi\\|mpg\\|jpg\\|tif\\mov\\au\\)" . t)
					; known binary data files
    ("\\.\\(arc\\|zip\\|pak\\|lzh\\|zoo\\|class\\)$" . t)
					; Packers
    ("\\.\\(a\\|o\\|tar\\|z\\|gz\\|taz\\|jar\\)$" . t)
					; Unix stuff
    ("\\.tp[ulpw]$" . t)
					; Borland Pascal stuff
    )
  "*Alist for distinguishing text files from binary files.
Each element has the form (REGEXP . TYPE), where REGEXP is matched
against the file name, and TYPE is nil for text, t for binary.
A new type `auto' is now supported which uses a heuristic to
decide whether the file is text or binary.  Other symbols are
interpreted as functions to call to return the type.")

(defun find-buffer-file-type (filename)
  "Determine the type (text or binary) of FILENAME.
This is called by insert-file-contents to determine whether to convert
DOS line endings.  Strictly speaking, the file type should only depend
on the file name, but if insert-file-contents-allow-replace and
in-insert-file-contents are both nil, then it is reasonably safe to
examine the buffer contents to determine the file type.  (That is, the
file contents will have been inserted at point before calling this
function.  However, the length of inserted data is not known.)"
  (if (and in-insert-file-contents auto-detect-buffer-file-type)
      ;; use auto detection for all files when inserting
      'auto
    ;; decide file type based on file name
    (let ((alist file-name-buffer-file-type-alist)
	  (found nil)
	  (code nil))
      (let ((case-fold-search t))
	(setq filename (file-name-sans-versions filename))
	(while (and (not found) alist)
	  (if (string-match (car (car alist)) filename)
	      (setq code (cdr (car alist))
		    found t))
	  (setq alist (cdr alist))))
      (if found
	  (cond((memq code '(nil t auto)) code)
	       ((and (symbolp code) (fboundp code))
		(funcall code filename)))
	default-buffer-file-type))))

(defun find-file-binary (filename) 
  "Visit file FILENAME and treat it as binary."
  (interactive "FFind file binary: ")
  (let ((file-name-buffer-file-type-alist '(("" . t)))
	(auto-detect-buffer-file-type nil))
    (find-file filename)))

(defun find-file-text (filename) 
  "Visit file FILENAME and treat it as a text file."
  (interactive "FFind file text: ")
  (let ((file-name-buffer-file-type-alist '(("" . nil)))
	(auto-detect-buffer-file-type nil))
    (find-file filename)))

(defun find-file-auto (filename) 
  "Visit file FILENAME, detecting type (text or binary)."
  (interactive "FFind file (detect type): ")
  (let ((file-name-buffer-file-type-alist '(("" . auto)))
	(auto-detect-buffer-file-type nil))
    (find-file filename)))

(defun find-file-not-found-set-buffer-file-type ()
  (save-excursion
    (set-buffer (current-buffer))
    (setq buffer-file-type (find-buffer-file-type (buffer-file-name))))
  nil)

;;; To set the default file type on new files.
(add-hook 'find-file-not-found-hooks 'find-file-not-found-set-buffer-file-type)

(defun toggle-buffer-file-type (arg)
  "Change whether current buffer is written as binary or text.
When changing from binary to text, all DOS line endings are converted to
Unix line endings unless prefix arg is less than -1.  With arg, change
buffer type to binary if arg is positive, otherwise change to text."
  (interactive "P")
  (let* ((old buffer-file-type)
	 (mod (buffer-modified-p))
	 (buffer-read-only nil))
    (setq buffer-file-type
	  (if arg
	      (> (prefix-numeric-value arg) 0)
	    (not buffer-file-type)))
    (if (and old
	     (not buffer-file-type)
	     (or (not arg)
		 (> (prefix-numeric-value arg) -2)))
	(save-excursion
	  (beginning-of-buffer)
	  (while (search-forward "\r\n" nil t)
	    (replace-match "\n" nil t))
	  (set-buffer-modified-p mod))))
  (force-mode-line-update))

;;; Map all versions of a filename (8.3, longname, mixed case) to the 
;;; same buffer.
(setq find-file-visit-truename t)

;;; insert-file-contents now sets the variables buffer-file-lines and
;;; buffer-file-dos-lines to the number of lines (ie. LF chars) and the
;;; number of lines in DOS format (ie. CRLF pairs) inserted into the
;;; buffer.  (It also resets buffer-file-type to t if
;;; find-buffer-file-type returned nil, but the entire file used Unix
;;; eol format.  Since it may be useful to inspect these variables some
;;; time after calling insert-file-contents we force them to be buffer
;;; local (and permanent).
(defvar buffer-file-lines 0
  "Number of text lines last inserted by insert-file-contents.")
(make-variable-buffer-local 'buffer-file-lines)
(put 'buffer-file-lines 'permanent-local t)
(defvar buffer-file-dos-lines 0
  "Number of text lines in DOS format last inserted by insert-file-contents.")
(make-variable-buffer-local 'buffer-file-dos-lines)
(put 'buffer-file-dos-lines 'permanent-local t)

;;; For using attached Unix filesystems.
(defun save-to-unix-hook ()
  (save-excursion
    (setq buffer-file-type t))
  nil)

(defun revert-from-unix-hook ()
  (save-excursion
    (setq buffer-file-type (find-buffer-file-type (buffer-file-name))))
  nil)

;; Really should provide this capability at the drive letter granularity.
(defun using-unix-filesystems (flag)
  "Read and write files without CR/LF translation, if FLAG is non-nil.
This is in effect assuming the files are on a remote Unix file system.
If FLAG is nil, resume using CR/LF translation as usual."
  (if flag
      (progn
	(add-hook 'write-file-hooks 'save-to-unix-hook)
	(add-hook 'after-save-hook 'revert-from-unix-hook))
    (progn
      (remove-hook 'write-file-hooks 'save-to-unix-hook)
      (remove-hook 'after-save-hook 'revert-from-unix-hook))))

;;; Avoid creating auto-save file names containing invalid characters
;;; (primarily "*", eg. for the *mail* buffer).
(fset 'original-make-auto-save-file-name
      (symbol-function 'make-auto-save-file-name))

(defun make-auto-save-file-name ()
  "Return file name to use for auto-saves of current buffer.
Does not consider `auto-save-visited-file-name' as that variable is checked
before calling this function.  You can redefine this for customization.
See also `auto-save-file-name-p'."
  (convert-standard-filename (original-make-auto-save-file-name)))

(defun convert-standard-filename (filename)
  "Convert a standard file's name to something suitable for the current OS.
This function's standard definition is trivial; it just returns the argument.
However, on some systems, the function is redefined
with a definition that really does change some file names."
  (let ((name (copy-sequence filename))
	(start 0))
    ;; leave ':' if part of drive specifier
    (if (eq (aref name 1) ?:)
	(setq start 2))
    ;; destructively replace invalid filename characters with !
    (while (string-match "[?*:<>|\"\000-\037]" name start)
      (aset name (match-beginning 0) ?!)
      (setq start (match-end 0)))
    name))

;;; Fix interface to (X-specific) mouse.el
(defun x-set-selection (type data)
  (or type (setq type 'PRIMARY))
  (put 'x-selections type data))

(defun x-get-selection (&optional type data-type)
  (or type (setq type 'PRIMARY))
  (get 'x-selections type))

;;; Check the configuration of the shell related variables.

(defun win32-system-shell-p (shell-name)
  (let ((system-shells '("cmd" "cmd.exe" "command" "command.com"
			 "4nt" "4nt.exe" "4dos" "4dos.exe"
			 "ndos" "ndos.exe")))
    (and shell-name
	 (member (downcase (file-name-nondirectory shell-name)) 
		 system-shells))))

(defvar win32-allow-system-shell nil
  "*Disable startup warning when using \"system\" shells.")

(defun win32-check-shell-configuration ()
  (interactive)
  (let ((prev-buffer (current-buffer))
	(buffer (get-buffer-create "*Shell Configuration*"))
	(system-shell))
    (set-buffer buffer)
    (erase-buffer)
    (if (win32-system-shell-p (getenv "SHELL"))
	(insert (format "Warning! The SHELL environment variable uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n" 
			(getenv "SHELL"))))
    (if (win32-system-shell-p shell-file-name)
	(insert (format "Warning! shell-file-name uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n" 
			shell-file-name)))
    (if (and (boundp 'explicit-shell-file-name)
	     (win32-system-shell-p explicit-shell-file-name))
	(insert (format "Warning! explicit-shell-file-name uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n"
			explicit-shell-file-name)))
    (setq system-shell (> (buffer-size) 0))

    ;; allow user to specify that they really do want to use one of the
    ;; "system" shells, despite the drawbacks, but still warn if
    ;; shell-command-switch doesn't match.
    (if win32-allow-system-shell
	(erase-buffer))

    (cond (system-shell
	   ;; System shells.
	   (if (string-equal "-c" shell-command-switch)
	       (insert "Warning! shell-command-switch is \"-c\".
You should set this to \"/c\" when using a system shell.\n\n"))
	   (if win32-quote-process-args
	       (insert "Warning! win32-quote-process-args is t.
You should set this to nil when using a system shell.\n\n")))
	  ;; Non-system shells.
	  (t
	   (if (string-equal "/c" shell-command-switch)
	       (insert "Warning! shell-command-switch is \"/c\".
You should set this to \"-c\" when using a non-system shell.\n\n"))
	   (if (not win32-quote-process-args)
	       (insert "Warning! win32-quote-process-args is nil.
You should set this to t when using a non-system shell.\n\n"))))
    (if (> (buffer-size) 0)
	(display-buffer buffer)
      (kill-buffer buffer))
    (set-buffer prev-buffer)))

(add-hook 'after-init-hook 'win32-check-shell-configuration)


;;; Basic support functions for managing Emacs' locale setting

(defvar win32-valid-locales nil
  "List of locale ids known to be supported.")

;;; This is the brute-force version; an efficient version is now
;;; built-in though.
(if (not (fboundp 'win32-get-valid-locale-ids))
    (defun win32-get-valid-locale-ids ()
      "Return list of all valid Windows locale ids."
      (let ((i 65535)
	    locales)
	(while (> i 0)
	  (if (win32-get-locale-info i)
	      (setq locales (cons i locales)))
	  (setq i (1- i)))
	locales)))

(defun win32-list-locales ()
  "List the name and id of all locales supported by Windows."
  (interactive)
  (if (null win32-valid-locales)
      (setq win32-valid-locales (win32-get-valid-locale-ids)))
  (switch-to-buffer-other-window (get-buffer-create "*Supported Locales*"))
  (erase-buffer)
  (insert "LCID\tAbbrev\tFull name\n\n")
  (insert (mapconcat
	   '(lambda (x)
	      (format "%d\t%s\t%s"
		      x
		      (win32-get-locale-info x)
		      (win32-get-locale-info x t)))
	   win32-valid-locales "\n"))
  (insert "\n")
  (goto-char (point-min)))


;;; Setup Info-default-directory-list to include the info directory
;;; near where Emacs executable was installed.  We used to set INFOPATH,
;;; but when this is set Info-default-directory-list is ignored.  We
;;; also cannot rely upon what is set in paths.el because they assume
;;; that configuration during build time is correct for runtime.
(defun win32-init-info ()
  (let* ((instdir (file-name-directory invocation-directory))
	 (dir1 (expand-file-name "../info/" instdir))
	 (dir2 (expand-file-name "../../../info/" instdir)))
    (if (file-exists-p dir1)
	(setq Info-default-directory-list 
	      (append Info-default-directory-list (list dir1)))
      (if (file-exists-p dir2)
	  (setq Info-default-directory-list
		(append Info-default-directory-list (list dir2)))))))

(add-hook 'before-init-hook 'win32-init-info)

;(fmakunbound 'font-menu-add-default)
;(global-unset-key [C-down-mouse-1])
;(global-unset-key [C-down-mouse-2])
;(global-unset-key [C-down-mouse-3])

;;; Set to a system sound if you want a fancy bell.
(set-message-beep nil)

;;; The "Windows" keys on newer keyboards bring up the Start menu
;;; whether you want it or not - make Emacs ignore these keystrokes
;;; rather than beep.
(global-set-key [lwindow] 'ignore)
(global-set-key [rwindow] 'ignore)

;; Map certain keypad keys into ASCII characters
;; that people usually expect.
(define-key function-key-map [tab] [?\t])
(define-key function-key-map [linefeed] [?\n])
(define-key function-key-map [clear] [11])
(define-key function-key-map [return] [13])
(define-key function-key-map [escape] [?\e])
(define-key function-key-map [M-tab] [?\M-\t])
(define-key function-key-map [M-linefeed] [?\M-\n])
(define-key function-key-map [M-clear] [?\M-\013])
(define-key function-key-map [M-return] [?\M-\015])
(define-key function-key-map [M-escape] [?\M-\e])

;; These don't do the right thing (voelker)
;(define-key function-key-map [backspace] [127])
;(define-key function-key-map [delete] [127])
;(define-key function-key-map [M-backspace] [?\M-\d])
;(define-key function-key-map [M-delete] [?\M-\d])

;; These tell read-char how to convert
;; these special chars to ASCII.
(put 'tab 'ascii-character ?\t)
(put 'linefeed 'ascii-character ?\n)
(put 'clear 'ascii-character 12)
(put 'return 'ascii-character 13)
(put 'escape 'ascii-character ?\e)
(put 'backspace 'ascii-character 127)
(put 'delete 'ascii-character 127)

;;; winnt.el ends here

