#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2009 Free Software Foundation
#
# FILE:
# GRLayout.py
#
# DESCRIPTION:
# Classes related to the Layout section
#
# NOTES:
#


from gnue.common.apps import GDebug
from gnue.reports.base import GRExceptions
import string
from xml.sax import saxutils
from ContainerElement import ContainerElement
from GRDefault import GRDefault


############################################################
#
# <section> tag
#
class GRSection (ContainerElement):

  def __init__(self, parent):
    ContainerElement.__init__(self, parent, 'GRSection')

    self._childSections = []

    self._validTriggers = {'PRE-SECTION':'Pre-Section',
                           'POST-SECTION':'Post-Section',
                           'ON-PROCESS':'On-Process'}

    self._triggerGlobal = True

  def _buildObject(self):
    if not hasattr(self,'name') or self.name == None:
      self._name = id(self)
    else:
      self._name = string.lower(self.name)

    # Keep track of "child" section names (for the firstRow stuff)
    psec = self.getParent ().findParentOfType('GRSection')
    if psec is not None:
      psec._childSections = self._childSections[:]
      psec._childSections.insert(0, self._name)

    return ContainerElement._buildObject(self)


  #
  # Retrieve the first row of data, if available.
  # Handle default data if no rows retrieved.
  #
  def processAsController(self, dest, mapper):

    assert gDebug(10,
       "Controlling section %s's processAsController() called" % self._name)

    if self._source == None:
      for s in mapper.sourceMap[None]:
        #   and run formula triggers
        for formula in s.formulas.keys():
          s.formulas[formula] = s.formulaObjects[formula].processTrigger('On-Process')

      # This is a dataless section...
      # simply process it's children

      self._process(dest, mapper, 1, 1, None, None)

    else:
      firstSection, nextSection = self.getGotos(mapper, 1)

      if firstSection:
        # This is a data-bound section and there
        # was data returned, so process section
        self._process(dest, mapper, 1, nextSection == None, self, nextSection)

      else:
        # This is a data-bound section and there
        # was no data returned, so generate default
        # text (if any)

        for c2 in self._children:
          if isinstance (c2, GRDefault):
            c2.processDefault(dest, mapper)



  #
  # Used internally to determine next section to move
  #
  def getGotos(self, mapper, init=0):
    # Determine the next section to process our data.
    # (i.e., handle section grouping)
    if init:
      goto, gotoNext = mapper.getFirstRecord(self._source)
    else:
      goto, gotoNext = mapper.getNextRecord(self._source)

    # Get the actual GRSection objects
    if not goto:
      return (None, None)

    firstSection = mapper.sectionMap[goto]._object
    if gotoNext:
      nextSection = mapper.sectionMap[gotoNext]._object
    else:
      nextSection = None

    # And return them
    return (firstSection, nextSection)


  #
  #  getAncestorWithSource()
  #
  def getAncestorWithSource(self, source):
    if self._source == source:
      return self
    elif isinstance(self.getParent (), GRSection):
      return self.getParent ().getAncestorWithSource(source)
    else:
      return None


  # Generic process() method.  Process the
  # current record and handle any children.
  def process(self, dest, mapper, isfirst, islast, firstSection, nextSection):
    self.processTrigger('Pre-Section')
    
    assert gDebug(10,"Repeating Section %s" % self._name)
    ## structuralComment(dest,"<!-- [section:%s] -->" % self._name)
    nextSection = self.processChildren(dest, mapper, isfirst, islast, firstSection, nextSection)

    try:
      ## structuralComment(dest,"<!-- [Setting %s to clear on next set] -->" % self._name)
      mapper.sectionMap[string.lower(self.name)]._clearOnNextSet = 1
    except AttributeError:
      pass
    except KeyError:
      pass

    ## structuralComment(dest,"<!-- [/section:%s] -->" % self._name)
    assert gDebug(10,"Leaving section %s" % self._name)
    self.processTrigger('Post-Section')
    return nextSection
