// Include file for the Fast Fourier Transform.  Supports FFT's of both
// real and complex  vectors and Matrices.  

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This file is part of the GNUSSL software package.  This package is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this distribution; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _fft_h_
#define _fft_h_

#include<utilities.h>

template<class Real,class Complex>
class LUT 
{
// This provides a look up table mechanism for calculating the array of 
// complex numbers used in the 1-dim fft routine.  The advantage of a look-up 
// table formulation is speed at the expense of using more memory.  However, 
// this routine only generates a new look-up table when a new fft size is 
// requested.  The array sizes are 1/2 of the (1-d) fft sizes and fft sizes 
// up to 2**max_power are allowed.  Requesting a larger size or a size which 
// is not a power of 2 causes the program to exit with an error message.
private:
  const int max_power=16;      // the maximum size fft allowed = 2**max_power
public:
  Complex* t[max_power+1];
  LUT() { for(int j=0;j<=max_power;j++) t[j]=0; }
  ~LUT() { for(int j=0;j<=max_power;j++) if(t[j]!=0) delete [] t[j]; }
  Complex* init(int size);
};


template<class Real,class Complex>
class FFT 
{
// Class for performing 1- and 2-dimensional radix-2 Fast Fourier Transforms.  
public:

  LUT<Real,Complex> lut;       // stores the look-up-table between transforms
  mVector<Complex,allocator<Complex > > vc;     // coefficient vector
  mVector<Real,allocator<Real> > vd;                        // real coefficient vector

#ifdef _MEMBER_TEMPLATES_
  template<class S>   			//  1-dimensional FFT for complex
    void fft(mVector<Complex,S>& f,int direction=1);
  template<class S>   			//  1-dimensional FFT for real
    void fft(mVector<Real,S>& f,int direction=1);
  template<class S>   			// real-valued fast sine transform.
    void sin(mVector<Real,S >& f);
  template<class S>   			// real-valued fast cosine transform.
    void cos(mVector<Real,S >& f);
  template<class S>			// 2-dimensional FFT
    void fft(mMatrix<Complex,S>& a,int direction=1);
  template<class S>			// 2-dimensional FST
    void sin(mMatrix<Real,S>& a);
  template<class S>			// 2-dimensional FCT
    void cos(mMatrix<Real,S>& a);
#else
      //  1-dimensional FFT for complex
  void fft(mVector<Complex,allocator<Complex > >& f,int direction=1);
      //  1-dimensional FFT for real
  void fft(mVector<Real,allocator<Real> >& f,int direction=1);
      // real-valued fast sine transform.
  void sin(mVector<Real,allocator<Real> >& f);
      // real-valued fast cosine transform.
  void cos(mVector<Real,allocator<Real> >& f);
#endif

private:
#ifdef _MEMBER_TEMPLATES_
  template<class S>   			// internal function
    void for_fft(mVector<Real,S >& f);
  template<class S>   			// internal function
    void back_fft(mVector<Real,S >& f);
#else
      // internal function
  void for_fft(mVector<Real,allocator<Real> >& f);  
      // internal function
  void back_fft(mVector<Real,allocator<Real> >& f);
#endif
};


template<class Real,class Complex>
Complex* 
LUT<Real,Complex>::init(int size) 
{
  // returns a pointer to the array of complex values

    if(size==1) {              // pointless, but makes the routine bullet-proof
      if(t[0]==0) t[0]=new Complex (1,0);     // exp(0)
      return t[0];
    }
    for(int j=0;j<=max_power;j++)    // loop over all possible sizes
      if(size==(1<<j)) {             // calculate the size by bit shifting
        if(t[j]!=0) return t[j];     // if it has been alloc., then return it
        int arrsz=1<<(j-1);
        t[j]=new Complex [arrsz]; // alloc an array with size/2 elements
        Real p=2.0*M_PI/size;           // min wavenumber
        t[j][0]=Complex (1,0);
        for(int k=1;k<arrsz;k++) t[j][k]=Complex (cos(k*p),sin(k*p));
        return t[j];
      }
#ifdef EX_HANDLE
    throw gError("lut(): supports only radix-2 FFT's.");
#else
    cerr << "lut(): supports only radix-2 FFT's\n";
    exit(1);
#endif
    return t[0];
}


#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void 
FFT<Real,Complex>::fft(mVector<Complex,S>& f,int direction=1) 
#else
template<class Real,class Complex>
void 
FFT<Real,Complex>::fft(mVector<Complex,allocator<Complex > >& f,int direction=1) 
#endif
{
  //  1-dimensional FFT for complex (cf., Num. Rec. in C, Ch. 12). The 
  // default direction=1 is the forward FFT, direction=-1 is the backward FFT.
  // The forward transformation is defined by:
  //   F(j) = \sqrt(1/n) \sum_{k=1}^{n} (f(k) \exp(2 i \pi (k-1) (j-1)/n) )
  // The inverse transformation is defined by:
  //   f(j) = \sqrt(1/n) \sum_{k=1}^{n} (F(k) \exp(-2 i \pi (k-1) (j-1)/n) )
    int m,n;
    Complex  temp;
    int size=f.size();
    long int brj=0;              // bit reversed j may need twice as many bits
    for(int j=1;j<=size-2;j++) { // begin bit reversal loop, skip first and last
      m=(size*size)>>2;          // puts a 1 at bit position (2*log(2,rsize)-1)
      brj=j;
      while(m>2)  {              // brj&1  gives rightmost bit, m*  shifts it
        brj=brj | (m*(brj&1));   // to the proper position, brj| turns it on
        brj>>=1;                 // move to the next bit
        m>>=2;
      }
      if(brj>j) {                // swap elements
        temp=f(j+1);
        f(j+1)=f(brj+1);
        f(brj+1)=temp;
      }
    }                            // end bit reversal loop
    m=2;
    while(m<=size)  {            // begin transform loop, m=order
      Complex  *w;               // get the appropriate table for this step
      w=lut.init(m);             // w[k]=exp(2*M_PI*I*k/m)
      n=size/m;           //  # of transforms of order m in array of length size
      for(int k=0;k<m/2;k++)  {     // loop over elements of a m-point transform
        for(int j=0;j<=n-1;j++)  {  // loop over all transforms of order m
          int tpos=k+m*j+1;         // the +1 offset is added here
          temp=(direction==1)?w[k]*f(tpos+m/2):conj(w[k])*f(tpos+m/2);
          f(tpos+m/2)=f(tpos)-temp;
          f(tpos)=f(tpos)+temp;
        }                        // end order m loop
      }                          // end element loop
      m*=2;                      // proceed to next order transform
    }                            // end transform loop
    Real sqt=sqrt(size);
    for(int j=1;j<=size;j++)  f(j)/=sqt;     // normalization
}            //  END VECTOR FFT

#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void
FFT<Real,Complex>::fft(mVector<Real,S>& f,int direction=1)
#else
template<class Real,class Complex>
void 
FFT<Real,Complex>::fft(mVector<Real,allocator<Real> >& f,int direction=1) 
#endif
{
    // Sets up and performs the real-valued FFT.  Based on the complex 
    // FFT, simply copying the real vector into a complex vector and
    // shuffling the coefficients afterward.  The argument f is unchanged,
    // the coefficients of the positive frequencies are left in vc, except
    // vc(1), which is left as vc(1)=Complex (F(1),F(n/2+1)), because 
    // F(1) and F(n/2+1) are inherently real.  Otherwise, symmetry gives the 
    // relation that conj(F(n+1-j)))=F(j+1):=vc(j+1), for j=1 to n/2. 
    // Note that f.size() must be n=(2**j) for some integer j, and vc.size() 
    // is n/2. (cf. Num. Rec. in C, 2nd ed., p 512)
    // The forward transformation is then defined by:
    //   F(j) = \sqrt(1/n) \sum_{k=1}^{n} (f(k) \exp(2 i \pi (k-1) (j-1)/n) )
    if(direction==1) for_fft(f);
    else back_fft(f);
}
    
#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void
FFT<Real,Complex>::for_fft(mVector<Real,S>& f)
#else
template<class Real,class Complex>
void 
FFT<Real,Complex>::for_fft(mVector<Real,allocator<Real> >& f) 
#endif
{
    int n=f.size();                     // effective real dimension
    if(vc.size()!=n/2) vc.reserve(n/2); // complex size() is 1/2 of real size()
       // set up the equivalent complex vector
    for(int j=1;j<=n/2;j++) vc(j) = Complex (f(2*j-1),f(2*j));
    fft(vc,1);                          // take the complex FFT
    Complex  *w=lut.init(n);         // w[k]=exp(2*M_PI*I*k/n)
    vc(1)=Complex (real(vc(1))+imag(vc(1)),real(vc(1))-imag(vc(1)))/sqrt(2);
    for(int j=1;j<=n/4;j++) {           // transform coefficients for a real FFT
      Complex  t1,t2;
      t1= 0.5*(vc(j+1) + conj(vc(n/2-j+1)));
      t1-= Complex (0,0.5)*(vc(j+1) - conj(vc(n/2-j+1)))*w[j];
      t2= 0.5*(vc(n/2-j+1) + conj(vc(j+1)));
      t2-= Complex (0,0.5)*(vc(n/2-j+1) - conj(vc(j+1)))*w[n/2-j];
      vc(j+1)=t1/sqrt(2);
      vc(n/2-j+1)=t2/sqrt(2);
    }
  }


#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void
FFT<Real,Complex>::back_fft(mVector<Real,S>& f)
#else
template<class Real,class Complex>
void 
FFT<Real,Complex>::back_fft(mVector<Real,allocator<Real> >& f) 
#endif
{
    // inverts for_fft(f)
    int n=f.size();                     // effective real dimension
    Complex  *w=lut.init(n);         // w[k]=exp(2*M_PI*I*k/n)
    vc(1)=0.5*Complex (real(vc(1)) + imag(vc(1)),real(vc(1)) - imag(vc(1)));
    for(int j=1;j<=n/4;j++) {           // transform coefficients for a real FFT
      Complex  t1,t2;
      t1= 0.5*(vc(j+1) + conj(vc(n/2-j+1)));
      t1+= Complex (0,0.5)*(vc(j+1) - conj(vc(n/2-j+1)))*conj(w[j]);
      t2= 0.5*(vc(n/2-j+1) + conj(vc(j+1)));
      t2+= Complex (0,0.5)*(vc(n/2-j+1) - conj(vc(j+1)))*conj(w[n/2-j]);
      vc(j+1)=t1;
      vc(n/2-j+1)=t2;
    }
    fft(vc,-1);                         // take the complex FFT
       // set up the equivalent complex vector
    for(int j=1;j<=n/2;j++) {
      f(2*j-1)=real(vc(j));
      f(2*j)=imag(vc(j));
    }
    f*=sqrt(2);
}

#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void 
FFT<Real,Complex>::fft(mMatrix<Complex,S>& a,int direction=1) 
{
    // 2-dimensional FFT for complex, comprised of 1-D FFT's.  Note that 
    // the design of the mMatrix class removes the overhead associated with
    // performing the FFT's over the columns.
    for(int j=1;j<=a.rsize();j++) fft(a.row(j),direction);      // FFT rows
    for(int j=1;j<=a.csize();j++) fft(a.col(j),direction);      // FFT cols
}
#endif

#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void
FFT<Real,Complex>::sin(mVector<Real,S>& f)
#else
template<class Real,class Complex>
void 
FFT<Real,Complex>::sin(mVector<Real,allocator<Real> >& f) 
#endif
{
    // Sets up and performs the real-valued fast sine transform.
    // Note that f.size() must be (2**j - 1) for some integer j.
    // (cf. Num. Rec. in C, 2nd ed., p 514)
    // The forward transformation is defined by:
    //   F(j) = \sqrt(2/n) \sum_{k=1}^{n-1} (f(k) \sin(\pi k j/n) )
    // The inverse transformation is identical.
    int n=f.size()+1;                        // effective real dimension
    if(vd.size()!=n) vd.reserve(n);          // alloc the coeff vector
    Complex  *w=lut.init(2*n);            // w[k]=exp(M_PI*I*k/n)
    vd(1)=0;
    for(int j=1;j<n;j++)                     // compute the auxilliary vector
      vd(j+1)=imag(w[j])*(f(j)+f(n-j)) + 0.5*(f(j)-f(n-j));
    fft(vd,1);                               // take the real transform
    f(1)=0.5*real(vc(1));
    for(int j=1;j<n/2;j++)  {                // compute the real coefficients
      f(2*j)=imag(vc(j+1));
      f(2*j+1)=f(2*j-1) + real(vc(j+1));
    }
    f*=sqrt(2.0);
  }

#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void 
FFT<Real,Complex>::sin(mMatrix<Real,S>& a) 
{                                                     // 2-dimensional FST
    for(int j=1;j<=a.rsize();j++) sin(a.row(j));      // FFT rows
    for(int j=1;j<=a.csize();j++) sin(a.col(j));      // FFT cols
}
#endif

#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void
FFT<Real,Complex>::cos(mVector<Real,S>& f)
#else
template<class Real,class Complex>
void 
FFT<Real,Complex>::cos(mVector<Real,allocator<Real> >& f) 
#endif
{
    // Sets up and performs the real-valued fast cosine transform.
    // Here f.size() must be (n+1)=(2**j+1) and j an integer.  Note that this 
    // transformation does not satisfy a discrete Parsival's relation.
    // The forward transformation is defined by:
    //   F(j) = \sqrt(2/n)[ 0.5 (f(1) + (-1)^(j-1) f(n+1) ) + 
    //          \sum_{k=2}^{n} (f(k) \sin(\pi (k-1) (j-1)/n) )]
    // The inverse transformation is identical.
    int n=f.size()-1;                        // effective real dimension
    if(vd.size()!=n) vd.reserve(n);          // alloc the coeff vector
    Complex  *w=lut.init(2*n);            // w[k]=exp(M_PI*I*k/n)
    for(int j=0;j<n;j++)                     // compute the auxilliary vector
      vd(j+1)=0.5*(f(j+1)+f(n-j+1)) - imag(w[j])*(f(j+1)-f(n-j+1));
    fft(vd,1);                               // take the real transform
    Real t1=0.5*(f(1)-f(n+1));
    for(int j=1;j<n;j++) t1+=f(j+1)*real(w[j]);
    Real t2=0.5*(f(1)+f(n+1));
    int s=-1;
    for(int j=1;j<n;j++,s*=-1) t2+=s*f(j+1);
    f(1)=real(vc(1));
    f(2)=t1/sqrt(n);
    f(n+1)=t2/sqrt(n);
    for(int j=1;j<n/2;j++)  {                // compute the real coefficients
      f(2*j+1)=real(vc(j+1));
      f(2*j+2)=f(2*j) + imag(vc(j+1));
    }
    f*=sqrt(2.0);
}

#ifdef _MEMBER_TEMPLATES_
template<class Real,class S>
void 
FFT<Real,Complex>::cos(mMatrix<Real,S>& a) 
{                                                     // 2-dimensional FCT
    for(int j=1;j<=a.rsize();j++) cos(a.row(j));      // FFT rows
    for(int j=1;j<=a.csize();j++) cos(a.col(j));      // FFT cols
  }
#endif

#endif

