// This file implements a class for discrete transforms and least-squares
// fitting, Transform.

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This file is part of the GNUSSL software package.  This package is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this distribution; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _transform_h_
#define _transform_h_

#include<mMatrix.h>
#include<lin_alg.h>
#include<expansion.h>

template<class T,class S,class F> 
class Transform 
{
// Defines a function expansion for a functor of data.  Allows
// both interpolation and extrapolation of the fitted function.  
// Fitting is least squares via the SVD.  T is the type 
// of the paramatrizing variable and S is the type of 
// the fitted data.  F is a functor class, taking an integer
// index in [1,order()] and a mesh point of type T and returning a value
// of type T.  
public:
  F fnc;                             // transformation functions
  size_t order() const { return order_; }    // # of approximating functions
  size_t length() const { return length_; }  // # of pixels of data to fit
  mMatrix<T,allocator<T> > opMat,invOpMat;   // expansion matrix, pseudo-inverse
  mVector<T,allocator<T> > mesh;     // mesh on which functions are evaluated
  mVector<S,allocator<S> > coef;     // coefficients fitted to data
  mVector<S,allocator<S> > data;     // data for fitting coefficients

  Transform(F f) : fnc(f) {}
  Transform(F f,size_t l,size_t o) : fnc(f) { resize(l,o); }

  void resize(size_t l,size_t o);       // reserve the storage arrays
#ifdef _MEMBER_TEMPLATES_
  template<class Real> 
    void initOp(Real tol);              // initialize the opMat's from mesh
#else         NOT _MEMBER_TEMPLATES_
  void initOp(long double tol);         // initialize the opMat's from mesh
#endif         NOT _MEMBER_TEMPLATES_
  void initSelfInvOp();                 // if opMat is its own inverse
  void initMesh();                      // set mesh using the F::mesh() function
  void initMesh(T xmin,T xmax);         // set regular mesh in [xmin,xmax]

  void forward() { dot(invOpMat,data,coef); } // computes coef from data 
  void backward() { dot(opMat,coef,data); }   // computes data from coef 

  T operator()(T x);                    // interpolation and extrapolation
private:
  size_t length_,order_;
}; 

// CLASS DEFINITIONS

template<class T,class S,class F> 
T 
Transform<T,S,F>::operator()(T x) 
{    
// interpolation and extrapolation
  mVector<T,allocator<T> > p(order());
  for(size_t j=1;j<=order();j++) p(j)=fnc(j,x);
  return sum(coef*p);
}

template<class T,class S,class F> 
void 
Transform<T,S,F>::resize(size_t l,size_t o)
{
// resize storage matrices on a supplied mesh
  if(o!=order()) {
    coef.reserve(o);
  }
  if(l!=length()) {
    mesh.reserve(l);
    data.reserve(l);
  }
  if((l!=length())||(o!=order())) {
    opMat.reserve(l,o);
    invOpMat.reserve(o,l);
  }
  order_=o;
  length_=l;
}

#ifdef _MEMBER_TEMPLATES_
template<class T,class S,class F,class Real> 
void 
Transform<T,S,F>::initOp(Real tol)
#else         NOT _MEMBER_TEMPLATES_
template<class T,class S,class F> 
void 
Transform<T,S,F>::initOp(long double tol)
#endif         NOT _MEMBER_TEMPLATES_
{
// initialize the operator matrices from the stored mesh.
// tol specifies the tolerance and real type for the pseudo-inverse
  initSelfInvOp();
  mMatrix<T,allocator<T> > uf(length(),order());
  mMatrix<T,allocator<T> > vf(order(),order());
  for(size_t j=1;j<=order();j++) uf(j,j)=vf(j,j)+=(T)1.0;
#ifdef _MEMBER_TEMPLATES_
  mVector<Real,allocator<Real> > svf(order());
  pseudo_inverse(invOpMat,uf,svf,vf,(Real)0.0);  // invert
#else         NOT _MEMBER_TEMPLATES_
  mVector<long double,allocator<long double> > svf(order());
  pseudo_inverse(invOpMat,uf,svf,vf,(long double)0.0);  // invert
#endif         NOT _MEMBER_TEMPLATES_
}

template<class T,class S,class F> 
void 
Transform<T,S,F>::initSelfInvOp()
{
  for(size_t j=1;j<=order();j++)    // loop over functions
    for(size_t k=1;k<=length();k++) // loop over the mesh
      opMat(k,j)=fnc(j,mesh(k));  // evaluate j'th function on k'th mesh point
  invOpMat.resize(length(),order());
  invOpMat=opMat;
}

template<class T,class S,class F> 
void 
Transform<T,S,F>::initMesh(T xmin,T xmax) 
{
// set the transformation mesh to regular intervals
  for(size_t j=1;j<=length();j++) mesh(j)=xmin + (j-1)*(xmax-xmin)/(length()-1.0);
}

template<class T,class S,class F> 
void 
Transform<T,S,F>::initMesh()
{
// set the transformation mesh using the mesh() member of class F
  for(size_t j=1;j<=length();j++) mesh(j)=fnc.mesh(j);
}

template<class T,class S,class F> 
ostream& 
operator<<(ostream &os,Transform<T,S,F> &s) 
{
  os << s.order() << "\t";
  os << s.length() << "\n";
  os << s.mesh;
  os << s.coef;
  os << s.data;
  return os;
}

template<class T,class S,class F> 
istream& operator>>(istream &is,Transform<T,S,F> &sp) 
{
  size_t o,l;
  is >> o;
  is >> l;
  is >> sp.mesh;
  is >> sp.coef;
  is >> sp.data;
  sp.resize(o,l);
  sp.initOp((long double)0.0);
  return is;
}

#endif

